unit TrivWinU;  // D3DX-Minimaldemo, 01-AUG-00 as
{ Ziemlich direkte Umsetzung des SDK-Beispiels TRIVWIN.
  Voraussetzungen: DirectX-Header von Erik Unger
  D3DX-Header und D3DXAS.DLL, DirectX7 }

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls,
  Forms, Dialogs, DirectDraw, Direct3D, D3DX, DXTimer, Menus;

type
  TTWForm = class(TForm)
    MainMenu1: TMainMenu;
    mFile: TMenuItem;
    mFileQuit: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure mFileQuitClick(Sender: TObject);
  protected
    DXTimer: TDXTimer;
    m_fAngle: FLOAT;  // rotation angle
    procedure OnDXTimer(Sender: TObject);
    procedure InterpretError(Res: HResult);
    // Nur zur Demonstration, im Fensterbetrieb unntig
    procedure WMEnterMenuLoop(var Msg: TWMEnterMenuLoop);
       message WM_ENTERMENULOOP;
    procedure WMExitMenuLoop(var Msg: TWMExitMenuLoop);
      message WM_EXITMENULOOP;
  public
    function ReleaseD3DX: HResult;
    function InitD3DX: HResult;
    function Draw: HResult;
    function HandleWindowedModeChanges: HResult;
  public
    m_bD3DXReady, m_bActive: Boolean;
    m_pD3DX: ID3DXContext;
    m_pD3DDev: IDirect3DDevice7;
    m_pDD: IDirectDraw7;
    m_pD3D: IDirect3D7;
  end;

var TWForm: TTWForm;

implementation
{$R *.DFM}
procedure TTWForm.InterpretError(Res: HResult);
var Msg: String;
begin
  SetLength(Msg,100);
  D3DXGetErrorString(Res, 100, PChar(Msg));
  ShowMessage('D3DX Error: '+PChar(Msg));
end;

procedure TTWForm.FormCreate(Sender: TObject);
var Res: HResult;
begin
  mFileQuit.ShortCut := ShortCut(VK_ESCAPE,[]);
  m_bActive := True; m_bD3DXReady := False;
  Res := InitD3DX;
  if FAILED(Res) then
  begin
    InterpretError(Res); PostQuitMessage(0);
  end else
  begin
    DXTimer := TDXTimer.Create(Self);
    DXTimer.OnTimer := OnDXTimer;
    DXTimer.Interval := 20;
    DXTimer.Enabled := True;
  end;
end;

procedure TTWForm.FormDestroy(Sender: TObject);
begin
  ReleaseD3DX;
end;

function TTWForm.InitD3DX: HResult;
begin  // XRef: FormCreate, HandleWindowedModeChanges
  Result := D3DXInitialize;
  if Succeeded(Result) then
    Result := D3DXCreateContext(
      // D3DX handle, flags, Window handle, colorbits,
      D3DX_DEFAULT, 0, Handle,  D3DX_DEFAULT, // colorbits
      D3DX_DEFAULT, m_pD3DX); // numdepthbits, D3DX interface
  if Failed(Result) then
  begin
    ShowMessage('D3DXCreateContext failed, trying HWLEVEL_2D "Acceleration"');
    Result := D3DXCreateContext(
      // D3DX handle, flags, Window handle, colorbits,
      D3DX_HWLEVEL_2D, 0, Handle,  D3DX_DEFAULT, // colorbits
      D3DX_DEFAULT, m_pD3DX); // numdepthbits, D3DX interface
  end;
  m_bD3DXReady := Succeeded(Result);
  if not m_bD3DXReady then Exit;
  if not (D3DXD3DDeviceFromContext(m_pD3DX, m_pD3DDev)
   and D3DXDDFromContext(m_pD3DX, m_pDD)) then
  begin
    Result := E_FAIL; Exit;
  end;
  Result := m_pD3DDev.SetRenderState(
    D3DRENDERSTATE_DITHERENABLE, Ord(True));
  if FAILED(Result) then Exit;
  Result := m_pD3DDev.SetRenderState(
    D3DRENDERSTATE_CULLMODE, Ord(D3DCULL_NONE));
  if FAILED(Result) then Exit;
  // Hintergrundfarbe
  Result := m_pD3DX.SetClearColor(D3DRGBA(0.8,0.3,0.3,0));
  if FAILED(Result) then Exit;
  Result := m_pD3DX.Clear(D3DCLEAR_TARGET or D3DCLEAR_ZBUFFER);
end;

function TTWForm.ReleaseD3DX: HResult;
begin
  // gibt sonst einen GPF beim nchsten Start?
  m_pD3DDev := nil; m_pDD := nil; m_pD3D := nil;
  m_pD3DX := nil;
  m_bD3DXReady := False;
  D3DXUninitialize;
  Result := S_OK;
end;

procedure TTWForm.FormResize(Sender: TObject);
var Res: HResult;
begin
  if m_bD3DXReady then
  begin
    Res := m_pD3DX.Resize(ClientWidth,ClientHeight);
    if Failed(Res) then
    begin
      m_bD3DXReady := False;
      InterpretError(Res);
      Close;
    end;
  end;
end;

procedure TTWForm.FormPaint(Sender: TObject);
begin
  Draw;
end;

procedure TTWForm.mFileQuitClick(Sender: TObject);
begin
  Close;
end;

procedure TTWForm.OnDXTimer(Sender: TObject);
begin
  if not (m_bD3DXReady and m_bActive) then Exit;
  m_fAngle := m_FAngle + 2.0*3.141592654 / 90.0;
  Draw;
end;

function TTWForm.HandleWindowedModeChanges: HResult;
begin  // Kommentare aus dem SDK-Original
  Result := m_pDD.TestCooperativeLevel;
  if SUCCEEDED(Result) then
  begin
    // This means that mode changes had taken place, surfaces
    // were lost but still we are in the original mode, so we
    // simply restore all surfaces and keep going.
    Result := m_pDD.RestoreAllSurfaces;
    if FAILED(Result) then Exit;
  end else if Result = DDERR_WRONGMODE then
  begin
    // This means that the desktop mode has changed
    // we can destroy and recreate everything back again.
    Result := ReleaseD3DX; if FAILED(Result) then Exit;
    Result := InitD3DX; if FAILED(Result) then Exit;
  end else if Result = DDERR_EXCLUSIVEMODEALREADYSET then
  begin
    // This means that some app took exclusive mode access we
    // need to sit in a loop till we get back to the right mode.
    repeat
       Sleep(500);
       Result := m_pDD.TestCooperativeLevel;
    until Result <> DDERR_EXCLUSIVEMODEALREADYSET;
    if SUCCEEDED(Result) then
    begin
    // This means that the exclusive mode app relinquished its
    // control and we are back to the safe mode, so simply restore
      Result := m_pDD.RestoreAllSurfaces;
      if FAILED(Result) then Exit;
    end else if Result = DDERR_WRONGMODE then
    begin
      // This means that the exclusive mode app relinquished its
      // control BUT we are back to some strange mode, so destroy
      // and recreate.
      Result := ReleaseD3DX; if FAILED(Result) then Exit;
      Result := InitD3DX; if FAILED(Result) then Exit;
    end;
  end;
end;

var vTriangle: Array[0..2] of TD3DLVERTEX;  // completely zeroed
// = D3DVECTOR(0,1,0), D3DVECTOR(1,0,0), D3DVECTOR(-1,0,0)
var v: Array[0..2] of TD3DVECTOR;

function TTWForm.Draw: HResult;
var fSin, fCos: FLOAT; i: Integer;

// position data in vTriangle is filled in by rotating the above points
begin
  if not m_bD3DXReady then begin Result := E_FAIL; Exit; end;
  if not m_bActive then begin Result := S_OK; Exit; end;
  // pos. data in vTriangle filled in by rotating the above points
  v[0].y := 1.0; v[1].x := 1.0; v[2].z := -1.0;
  Result := m_pD3DDev.BeginScene;
  if SUCCEEDED(Result) then
  begin
    m_pD3DX.Clear(D3DCLEAR_TARGET or D3DCLEAR_ZBUFFER);

    // Rotate triangle about Y axis before drawing
    fSin := sin(m_fAngle);
    fCos := cos(m_fAngle);
    for i := 0 to 2 do
    begin
      vTriangle[i].x := v[i].x * fCos - v[i].z * fSin;
      vTriangle[i].y := v[i].y;
      vTriangle[i].z := v[i].x * fSin + v[i].z * fCos + 2.5;
    end;
    m_pD3DDev.DrawPrimitive(D3DPT_TRIANGLELIST, D3DFVF_LVERTEX,
      vTriangle, 3, D3DDP_WAIT );

    m_pD3DDev.EndScene;
  end;

  Result := m_pD3DX.UpdateFrame(0);
  if (Result = DDERR_SURFACELOST) or (Result = DDERR_SURFACEBUSY) then
        Result := HandleWindowedModeChanges;
end;

procedure TTWForm.WMEnterMenuLoop(var Msg: TWMEnterMenuLoop);
begin
  if Assigned(m_pD3DX) then
      m_pD3DX.SetClearColor(D3DRGBA(0.1,0.3,0.3,0));
  Draw;  // damit die neue Hintergrundfarbe auch erscheint
  m_bActive := False; // Vollbildmodus: PauseDrawing
  inherited;
end;

procedure TTWForm.WMExitMenuLoop(var Msg: TWMExitMenuLoop);
begin
  if Assigned(m_pD3DX) then
      m_pD3DX.SetClearColor(D3DRGBA(0.8,0.3,0.3,0));
  m_bActive := True; // Vollbildmodus: RestartDrawing
  inherited;
end;

end.
