/**
 * ShortcutPopup: zeigt eine Liste aller definierten Shortcuts an
 * und fgt ausgewhlten Shortcut in das aktive Programm ein
 * Verwendet die Nachricht sysNotifyProcessPenStrokeEvent
 * luft ab Palm OS 5 
 */
#include <PalmOS.h>
#include "ShortcutPopup_res.h"

/**
 * Gib Anzahl der gespeicherten Shortcuts zurck
 * (ohne die mit "." beginnenden System-Shortcuts)
 */
static UInt16 getNumShortcuts()
{
UInt16 i;
UInt16 count=0;
Err err=errNone;
// Endlosschleife bis ein Fehler auftritt
for(i=0;;i++)
	{
	Char shortcutname[grfNameLength+1];
	err=GrfGetMacroName(i,shortcutname);
	if(err!=errNone)
		break;
	if(shortcutname[0]=='.') // ignoriere "."-Shortcuts
		continue;
	count++;
	} 
return count;
}

/**
 * Bestimme den Namen des Shortcuts aus dem Index
 * Ignoriere System-Shortcuts in der Zhlung
 */
static void getShortcutName(UInt16 index,Char *name)
{
UInt16 i;
UInt16 count=0;
Err err=errNone;
// Endlosschleife bis ein Fehler auftritt
for(i=0;;i++)
	{
	Char shortcutname[grfNameLength+1];
	err=GrfGetMacroName(i,shortcutname);
	if(err!=errNone)
		break;
	if(shortcutname[0]=='.') // ignoriere "."-Shortcuts
		continue;
	if(count==index)
		{
		// Shortcut gefunden, kopiere Namen
		StrCopy(name,shortcutname);
		break;
		}
	count++;
	} 
}

/**
 * Callback-Funktion zum Zeichnen eines Listeintrags,
 * wird vom System beim Zeichnen der Liste aufgerufen. 
 */
static void ListDrawFunc(UInt16 index, RectanglePtr bounds,Char **data)
{
Char shortcutname[grfNameLength+1];
getShortcutName(index,shortcutname);
WinDrawChars(shortcutname, StrLen(shortcutname),
            bounds->topLeft.x, bounds->topLeft.y);
}

/**
 * Fge Shortcut-Text in das angegebene Feld ein
 */
static void insertText(UInt16 index,FieldPtr fldP)
{
Err err;
Char shortcutname[grfNameLength+1];
Char text[100];
UInt16 sizetext=sizeof(text);

MemSet(text,sizeof(text),0); // setze Text auf 0

getShortcutName(index,shortcutname);
// expandiere Shortcut-Text (z.B. Zeitstempel)
err=GrfGetAndExpandMacro(shortcutname,text,&sizetext);
if(err==errNone)
	{
	// insert macro text
	FldInsert(fldP,text,StrLen(text));
	}
}
/**
 * erzeugt ein neues unsichtbares Form 
 * und darin eine Popup-Liste
 * fge ausgewhlten Eintrag ein
 */
static void popupShortcutList()
{
Int16 selectedind;
FormPtr frmList;
ListPtr list;
UInt16 numShortcuts=getNumShortcuts(); // Gesamtanzahl der Eintrge
UInt16 numVisibleItems=numShortcuts; // Anzahl sichtbare Eintrge
FormPtr frm= FrmGetActiveForm(); // speichere aktuell aktives Form

if(numVisibleItems==0)
	return; // nichts zu tun, leere Liste

if(numVisibleItems>10)
	numVisibleItems=10; // begrenze sichtbare Eintrge
// erzeuge unsichtbares Form
frmList = FrmNewForm(1000, "", 160, 0, 0, 0, false, 0, 0, 0);
FrmSetActiveForm(frmList);
FrmDrawForm(frmList);
// erzeuge Liste an Position rechts unten (relativ zu Form-Position)
LstNewList((void **)&frmList, 1000, -50,
              160 - (numVisibleItems * FntCharHeight()), 50,
              numVisibleItems * FntCharHeight(), stdFont, numVisibleItems, 0);
list=FrmGetObjectPtr(frmList,FrmGetObjectIndex(frmList,1000));			 
LstSetDrawFunction(list,(ListDrawDataFuncType*)ListDrawFunc);
LstSetListChoices(list, NULL, numShortcuts);
LstSetSelection(list,noListSelection);

// zeige Liste und warte auf Auswahl
selectedind=LstPopupList(list);

// lsche Form und aktiviere das vorhergehende
FrmEraseForm(frmList);
FrmDeleteForm(frmList);
FrmSetActiveForm(frm);

// falls ein Eintrag ausgewhlt wurde, fge den zugehrigen Text ein
if(selectedind!=noListSelection && frm!=NULL)
	{
	FieldPtr focusfld;
	// bestimme Feld, das den Fokus hat
    UInt16 focusobjind	=FrmGetFocus(frm);
    if(focusobjind==noFocus)
    	return; // kein Feld mit Fokus
	// Tabellenfeld oder normales Feld? 
	if(FrmGetObjectType(frm,focusobjind)==frmFieldObj)
		focusfld=(FieldPtr)FrmGetObjectPtr(frm,focusobjind);
	else
		focusfld=TblGetCurrentField((TablePtr)FrmGetObjectPtr(frm,focusobjind));
	// fge Text ein
	insertText(selectedind,focusfld);
	}
}

/**
 * Gibt den virtuellen Button-Character-Code fr die 
 * gegebene Position zurck oder 0 falls der Punkt 
 * zu keinem Button gehrt
 */
static UInt16 getButtonForPosition(Coord x,Coord y)
{
UInt16 numbuttons;
UInt16 i;
const PenBtnInfoType *buttons=EvtGetPenBtnList(&numbuttons);
// Schleife ber alle virtuellen Buttons
for(i=0;i<numbuttons;i++)
	{
	if(RctPtInRectangle(x,y,&buttons[i].boundsR))
		return buttons[i].asciiCode; 
	}
return 0;
}

/**
 * Prfe ob der aktuelle Strich der Aktivierungsstrich ist, d.h.
 * vom Suchen-Button zum Rechner-Button fhrt.
 */
static Boolean isActivationStroke(PointType startPt,PointType endPt)
{
UInt16 startBtn=getButtonForPosition(startPt.x,startPt.y);
UInt16 endBtn=getButtonForPosition(endPt.x,endPt.y);
return (startBtn==vchrFind && endBtn==vchrCalc);
}

/**
 * Registriere Programm fr die Systemnachricht
 * sysNotifyProcessPenStrokeEvent zum Erkennen von Strichen mit dem Stift 
 */
static void registerNotifications()
{
UInt16 cardNo;
LocalID dbID;
SysCurAppDatabase(&cardNo,&dbID);
SysNotifyRegister (cardNo,dbID,sysNotifyProcessPenStrokeEvent,
	NULL,sysNotifyNormalPriority,NULL);
}

/**
 * Hilfsfunktion zum Prfen der richtigen OS-Version.
 */
Boolean isOS5OrLater()
{
    UInt32 romVersion;
    FtrGet(sysFtrCreator,sysFtrNumROMVersion,&romVersion);
    return (romVersion>=sysMakeROMVersion(5,0,0,0,0));
}

/**
 * Ereignisbehandlung fr das einzige Form
 */
Boolean frmMain_HandleEvent(EventPtr event)
{
	FormPtr form;
	Boolean handled = false;

	switch (event->eType)
	{
		case frmOpenEvent:
			// Repaint form on open
			form = FrmGetActiveForm();
			FrmDrawForm(form);
			// setze Fokus auf Texteingabefeld
			FrmSetFocus(form,FrmGetObjectIndex(form,fldText));
			handled = true;
			break;
		default:
			break;
	}

	return handled;
}

/**
 * Anwendungsereignisschleife
 */
static Boolean ApplicationHandleEvent(EventPtr event)
{
	UInt16 formID;
	FormPtr form;
	Boolean handled = false;
	
	switch (event->eType)
	{
		case frmLoadEvent:
			formID = event->data.frmLoad.formID;
			form = FrmInitForm(formID);
			FrmSetActiveForm(form);		
			switch (formID)
				{
				case frmMain:
					// setze Event-Handler fr Form main
					FrmSetEventHandler(form, 
						(FormEventHandlerPtr) frmMain_HandleEvent);
					break;
				default:
					break;
				}
			handled = true;
			break;
		default:
			break;
		}

	return handled;
}

/**
 * Allgemeine Ereignisbehandlung.
 */
static void EventLoop(void)
{
	Err error;
	EventType event;

	// Main event loop
	do
		{
		// Get next event
		EvtGetEvent(&event, evtWaitForever);
	
		// Handle event
		if (!SysHandleEvent(&event))
			if (!MenuHandleEvent(0, &event, &error))
				if (!ApplicationHandleEvent(&event))
					FrmDispatchEvent(&event);	
		}
	while (event.eType != appStopEvent);
}


/**
 * main()-Routine jedes Palm-OS-Programms.
 * Diese Routine wird beim Programmstart und 
 * auch fr alle Nachrichten und Launch Codes aufgerufen.
 */
UInt32 PilotMain(UInt16 cmd, void *cmdPBP, UInt16 launchFlags)
{
	switch (cmd) 
	 	{
		case sysAppLaunchCmdNormalLaunch:
			if(!isOS5OrLater())
				{
				FrmAlert(alertOS5);
				return 0;
				}
			// registriere Anwendung nach Programmstart
			registerNotifications();
			FrmGotoForm(frmMain);
			EventLoop();
			FrmCloseAllForms();
			break;
		case sysAppLaunchCmdNotify:
    		{
    		//  notification
    		SysNotifyParamType *param=(SysNotifyParamType *)cmdPBP;  
        	switch(param->notifyType)
        		{
    			case sysNotifyProcessPenStrokeEvent:
    				{
    				SysNotifyPenStrokeType *penstroke=
						(SysNotifyPenStrokeType*)param->notifyDetailsP;
    	    		// pen stroke
					if(isActivationStroke(penstroke->startPt,penstroke->endPt))
						{
						popupShortcutList();
						}
    				}
    				break;
				}
			}
  		case sysAppLaunchCmdSyncNotify:
			// registriere Anwendung direkt nach dem ersten HotSync
			if(isOS5OrLater())
				registerNotifications();
			break;			
		default:
			break;
		}
	return 0;
}
