/***
 * c't Motorsteuerung
 * motor.c
 *
 * eigentliche Motorsteuerung
 *
 * (c) 2002 Redaktion c't, Hannover
 *          Herbert Schmid
 ***/

#include <windows.h>

#include "usb.h"

#include "motor.h"

#include "resource.h"



/* 
 * alle Funktionen liefern bei Erfolg 0, im Fehlerfalle einen Wert >= 1
 *
 * Rckgabewerte:
 *   0 .. alles ok
 *   1 .. USB-Baustein nicht erreichbar
 *  11 .. USB-Baustein nicht erreichbar, Motor lief
 *   2 .. Motor lief nicht an oder blieb hngen
 *   3 .. Ziel trotz Richtungskorrektur nicht erreicht
 *   9 .. sonstiger Fehler bei der Initialisierung
 */


/* zum Debuggen ohne USB-Baustein */
/* Motor simulieren? */
#undef DEBUG
/* Motorfehler simulieren? */
/* Fehler bei Init und MotorTimer simulieren */
#undef DEBUGERR
/* Fehler im Richtungswechsel */
#undef DEBUGERR1
/* Fehler in MotorStart und MotorStop */
#undef DEBUGERR2

/* Fehler nur in der Debug-Version simulieren */
#ifdef DEBUGERR
#define DEBUG
#endif
#ifdef DEBUGERR1
#define DEBUG
#endif
#ifdef DEBUGERR2
#define DEBUG
#endif
/* Ende Debug */


/* Abstand zwischen den TimerTicks der Motorsteuerung */
#define TIMERELAPSE 50 /* ms */


/* Definition der Variablen in motor.h*/
int MotorPoti=0; /* enthlt die aktuelle Position */

int MotorGoal=0; /* Zielvorgabe fr den Motor, 0 .. 255 */
BOOL MotorInvertPoti=FALSE; /* Potiwert negieren */
BOOL MotorExchangeLR=FALSE; /* Drehrichtung vertauschen */


/* nur in Motor.c verwendete Variablen */

/* ID des verwendeten Timers */
int	TimerID=0;
/* Motor darf laufen */
BOOL MotorEnabled=FALSE;
/* Motor luft */
BOOL MotorRuns=FALSE;

/* Motorberwachung in MotorTimer (& MotorStop) */
int CorrCounter=0; /* wie oft wurde automatisch die Richtung gewechselt */
int NotMovedCounter=0; /* Anzahl der Ticks, die sich der Motor nicht bewegt hat */

/***
 * USB-Treiber ffnen und Timer starten
 *
 *  hWnd: Handle des Dialog-Fensters
 *			(wird zum setzen des Timers bentigt)
 ***/
int MotorInit(HWND hWnd)
{
  int ret;
#ifdef DEBUGERR
  /* Fehler simulieren */
  static int Errored = 0;
  if(Errored<1)
  {   Errored++;
	  return 1;
  }
#endif
#ifdef DEBUG
  MessageBox(hWnd, "Debug-Modus mit Simulation:\n\tder USB-Baustein wird nicht angesprochen!", 
	         "c't Motorsteuerung", MB_ICONINFORMATION | MB_OK | MB_SETFOREGROUND);
#else

  /* Gertetreiber ffnen */
  if (USBOpenDriver())
	return 1;

#endif
  
  if( ret=MotorStop(hWnd) )
	  return ret;
  
  if(TimerID == 0)
    TimerID=SetTimer(hWnd, 3005, TIMERELAPSE, NULL);
  
  if ( TimerID == 0 )
  {
#ifndef DEBUG
	  /* Timer konnte nicht gestartet werden, USB-Treiber wieder schlieen */
	  if (USBCloseDriver()) 
		  return 1;
#endif
	  return 9;
  }
  
  return ret;
}



/***
 * Timer und Motor stoppen, USB-Treiber schlieen
 *
 *  hWnd: Handle des Dialog-Fensters
 ***/
int MotorClose(HWND hWnd)
{
	
	if(TimerID)
	  KillTimer(hWnd, TimerID);
	
	MotorStop(hWnd);

#ifdef DEBUG
	return 0;
#else
	return USBCloseDriver() ? 1 : 0;
#endif
}


/***
 * Motor ausschalten
 *
 *  hWnd: Handle des Dialog-Fensters
 ***/ 
int MotorStop(HWND hWnd)
{
    int ret=0;
	
#ifdef DEBUGERR2
  /* Fehler simulieren */
  static int Errored = -1;
  if(Errored<1)
  {   Errored++;
      if(Errored>0)
	    return 1;
  }
#endif

  MotorEnabled=FALSE;

#ifndef DEBUG
	/* Motor ber USB abschalten */
	if (USBWritePort(1, 0x0a)) 
		ret=1;
#endif
	
	/* Variablen der Motorberwachung zurcksetzen */
	CorrCounter=0; 
    NotMovedCounter=0; 
	
	MotorRuns=FALSE;
	MotorSetGui(hWnd, MotorRuns);
	
	return ret;
}
	  

/***
 * Motor starten
 *
 *  hWnd: Handle des Dialog-Fensters
 ***/
int MotorStart(HWND hWnd)
{
  unsigned char control=0;

#ifdef DEBUGERR2
  /* Fehler simulieren */
  static int Errored = -5;
  if(Errored<1)
  {   Errored++;
	  if(Errored>0)
	    return 1;
  }
#endif
  
  MotorEnabled=TRUE;
  MotorRuns=TRUE;
  MotorSetGui(hWnd, MotorRuns);

  /* Poti meldet nur 256 Positionen */
  if (MotorGoal < 0)   MotorGoal = 0;
  if (MotorGoal > 255) MotorGoal = 255;
  
  /* Richtung bestimmen */
  
  /* Werte werden mit XOR gesetzt, 
	  0x0a bewirkt eine Negation */
  if (MotorExchangeLR)
	  control = 0x0a;
  
  if (MotorGoal < MotorPoti-1)
	  control ^= 0x02;
  if (MotorGoal > MotorPoti+1)
	  control ^= 0x08;

#ifdef DEBUG
  return 0;
#else  

  return USBWritePort(1, control) ? 1 : 0;

#endif
}

/***
 * MotorTimer
 *
 *   berwacht den Motor und liest regelmig den Wert des Poti aus
 *	 ruft gegebenenfalls die Callback-Funktion MotorSetGui auf 
 *
 *  hWnd: Handle des Dialog-Fensters
 ***/
BOOL MotorTimer(HWND hWnd)
{
	unsigned char val;

	/* Variablen zur Motorberwachung */
	static int OldPoti=-1; /* PotiWert beim letzten Timer-Tick */
	
	int ret=0;

#ifdef DEBUGERR
	static Errored=0;
	if(Errored>40) /* 2s */
	{  Errored=0;
	   return MotorRuns ? 11 : 1 ; /* USB-Fehler simulieren */
	}
	Errored++;
#endif
	
#ifdef DEBUG
	if (MotorRuns)
	{
	  MotorPoti += (MotorExchangeLR ? -2 : 2);

	  if (MotorPoti  < 0)
		MotorPoti = 0;
	  if (MotorPoti > 255)
		MotorPoti = 255;

	  MotorPoti = MotorInvertPoti ? 255  - MotorPoti : MotorPoti;
	}
#else

	/* Poti einlesen */			
	if (USBReadPort(0, &val))
		return MotorRuns ? 11 : 1 ; /* USB-Fehler */
	
	MotorPoti = MotorInvertPoti ? 255 - val : val;

#endif
     
	/* darf der Motor berhaupt laufen? */
	if(MotorRuns && !MotorEnabled)
	   return MotorStop(hWnd);
	
	if(!MotorEnabled)
		return 0; /* nichts weiter zu tun */

	/* Ziel erreicht? */
	if (   MotorPoti >= (MotorGoal-1) && MotorPoti <= (MotorGoal+1) ) 
    {  /* Motor stoppen */
		return MotorStop(hWnd);
	}


	/* Motorberwachung */
    ret = 0;

	if(OldPoti<0) 
		OldPoti = MotorPoti; /* erste Runde */


    /* Stillstand entdecken */
	if(OldPoti <= MotorPoti+1 && OldPoti >= MotorPoti-1 )
		NotMovedCounter++;
	else
		NotMovedCounter=0;

	if(NotMovedCounter > 40) /* 20 [* 50ms] = 1s */
    { 
		NotMovedCounter=0;
		ret=MotorStop(hWnd);
		/* USB-Fehler wiegt schwerer als stillstehender Motor */
		if(!ret) ret=2;
	}
    
#if 1
	/* falsche Richtung entdecken */
	else if(CorrCounter > 2)
	{   /* zu oft korrigiert */
		CorrCounter = 0;
		ret = MotorStop(hWnd);
		if(!ret) ret=3;
	}
    else if( (MotorGoal > MotorPoti) ?   /* je nach Richtung: */
		     ( OldPoti > MotorPoti )  :   /* liegt die alte Position nher am Ziel? */
             ( OldPoti < MotorPoti ) )
	{
		/* mit Toleranz, da manche Grillmotoren stark holpern und der A/D-Wandler ungenau arbeitet */
#pragma message("motor.c: MotorTimer: Toleranz fr Richtungswechsel an die Motorgeschwindigkeit anpassen!")
		if(OldPoti-MotorPoti > 1 || OldPoti-MotorPoti < -1)
		{ 
			/* Richtung wechseln */
			CorrCounter++;
			#ifndef DEBUGERR1
			MotorExchangeLR = !MotorExchangeLR;
			#endif
			ret = MotorStart(hWnd); 
		}
	}
#endif

	OldPoti=MotorPoti;
	
	/* Anzeige aktualisieren */
	MotorSetGui(hWnd, MotorRuns);
	 
	return ret;

}


