using System;
using System.Runtime.InteropServices;

namespace System.Sound
{
  /// <summary>
  /// Zusammenfassung fr WaveHeader.
  /// </summary>
  [StructLayout(LayoutKind.Sequential)]
  public class WaveHeader: IDisposable
  {
    [DllImport("kernel32.dll")]
    private static extern IntPtr LocalAlloc(int flags, int sz); 
    [DllImport("kernel32.dll")]
    private static extern IntPtr LocalFree(IntPtr buf); 
    const int LPTR             = 0x0040; 

    public IntPtr Handle; 
    public IntPtr Buffer; 
    public int BufSize; 
    private bool ownbuffers; 
    private bool disposed = false; 
    public const int size = 32;  
    private byte[] barr = null; 
    private short[] sarr = null; 


//    0: LPSTR       lpData;                 /* pointer to locked data buffer */
//    4: DWORD       dwBufferLength;         /* length of data buffer */
//    8: DWORD       dwBytesRecorded;        /* used for input only */
//   12: DWORD_PTR   dwUser;                 /* for client's use */
//   16: DWORD       dwFlags;                /* assorted flags (see defines) */
//   20: DWORD       dwLoops;                /* loop control counter */
//   24: struct wavehdr_tag FAR *lpNext;     /* reserved for driver */
//   28: DWORD_PTR   reserved;               /* reserved for driver */


    /// <summary>
    /// Legt neue WAVEHDR-Struktur im unverwalteten Speicher an. 
    /// Fordert dafr Puffer via LocalAlloc() an. 
    /// </summary>
    /// <param name="bufSize">Puffergre in Bytes</param>
    public WaveHeader(int bufSize)
    {
      BufSize = bufSize; 
      Handle = LocalAlloc(LPTR, size); 
      Buffer = LocalAlloc(0, bufSize); 
      ownbuffers = true; 
      Marshal.WriteIntPtr(Handle, Buffer);                          // lpData
      Marshal.WriteInt32(new IntPtr(Handle.ToInt32()+4), BufSize);  // dwBufferLength
    }

    //properties
    public IntPtr lpData
    {
      get
      {
        return Marshal.ReadIntPtr(Handle); 
      }
    }
    public int BufferLength
    {
      get
      {
        return Marshal.ReadInt32(new IntPtr(Handle.ToInt32()+4)); 
      }
    }
    public int BytesRecorded
    {
      get
      {
        return Marshal.ReadInt32(new IntPtr(Handle.ToInt32()+8)); 
      }
      set
      {
        Marshal.WriteInt32(new IntPtr(Handle.ToInt32()+8), value);  
      }
    }
    public int Flags
    {
      get
      {
        return Marshal.ReadInt32(new IntPtr(Handle.ToInt32()+16)); 
      }
      set
      {
        Marshal.WriteInt32(new IntPtr(Handle.ToInt32()+16), value);  
      }
    }

    /// <summary>
    /// liefert Pufferinhalt als byte-Array
    /// </summary>
    /// <returns></returns>
    public byte[] ToByteArray()
    {
      int len = BytesRecorded; 
      if (barr == null || barr.Length != len) // beim ersten Mal und bei Reset
        barr = new byte[len]; 
      Marshal.Copy(lpData, barr, 0, len); 
      return barr; 
    }

    /// <summary>
    /// liefert Pufferinhalt als short-Array
    /// </summary>
    /// <returns></returns>
    public short[] ToShortArray()
    {
      int len = BytesRecorded/2; 
      if (sarr == null || sarr.Length != len) // beim ersten Mal und bei Reset
        sarr = new short[len]; 
      Marshal.Copy(lpData, sarr, 0, len); 
      return sarr; 
    }

    public void Dispose()
    {
      disposed = true; 
      if(ownbuffers)
      {
        LocalFree(Buffer); 
        LocalFree(Handle); 
      }
    }
    ~WaveHeader()
    {
      if (!disposed) 
        Dispose(); 
    }
  }

  /// <summary>
  /// Umsetzung der Struktur WAVEHDR als C#-Klasse mit unverwaltetem Speicherblock
  /// </summary>
  /// #define WAVE_FORMAT_PCM     1 
  /// typedef struct waveformat_tag {
  ///   WORD        wFormatTag;         /*  0: format type */
  ///   WORD        nChannels;          /*  2: number of channels (i.e. mono, stereo...) */
  ///   DWORD       nSamplesPerSec;     /*  4: sample rate */
  ///   DWORD       nAvgBytesPerSec;    /*  8: for buffer estimation */
  ///   WORD        nBlockAlign;        /* 12: block size of data */
  ///   WORD        wBitsPerSample;     /* 14: number of bits per sample of mono data */
  ///   WORD        cbSize;             /* 16: the count in bytes of the size of */
  /// }</code>
  /// <summary>
  /// headerblock for WaveIn operations (defines WAVEFORMATEX from mmsystem.h)
  /// </summary>
  [StructLayout(LayoutKind.Sequential)]
  public class WaveFormatEx
  {
    private enum FormatTags : ushort
    {
      /// <summary> uncompressed PCM-Format</summary>
      _PCM    = 1       
    }

    /// <summary>
    /// Initializes structure with wave format
    /// </summary>
    /// <param name="FormatTag">Format description: mono/stereo, kHz, 8/16 bit</param>
    public WaveFormatEx(WAVEFORMAT FormatTag)
    {
      wFormatTag = FormatTags._PCM;   // uncompressed
      int ft = (short)FormatTag; 
      bool done = false;
      for(nSamplesPerSec = 11025; !done && nSamplesPerSec < 50000; nSamplesPerSec*=2)
      {
        for (short i = 0; i < 4 && !done; i++, ft >>= 1)
        {
          nChannels = (short) (i % 2 + 1); 
          wBitsPerSample = (short) ((i / 2+1) * 8); 
          if((ft & 1) == 1)
          {
            done = true; 
          }
        }
      }
      nSamplesPerSec/=2; 
      nBlockAlign = (short) (nChannels * wBitsPerSample / 8); 
      nAvgBytesPerSec = nSamplesPerSec * nBlockAlign; 
    } 
    private FormatTags   wFormatTag = FormatTags._PCM;         
    public short        nChannels = 0;          
    public int          nSamplesPerSec = 0;     
    public int          nAvgBytesPerSec = 0;    
    public short        nBlockAlign = 0;        
    public short        wBitsPerSample = 0;     
    public short        cbSize = 0;             
  }

}
