using System;
using System.Runtime.InteropServices;

namespace System.Sound
{

	/// <summary>
	/// .NET implementation of WAVEHDR and WAVEBUFFER structures as defined in mmsystem.h
	/// </summary>
	public class WaveHeader: IDisposable
	{
		// Imports
		[DllImport("coredll.dll")]
		private static extern IntPtr LocalAlloc(int flags, int sz); 
		[DllImport("coredll.dll")]
		private static extern IntPtr LocalFree(IntPtr buf); 
		// Fields
		const int LPTR = 0x0040; // for LocalAlloc: FixedMem + ZeroInit
		public IntPtr Handle; 
		public IntPtr Buffer; 
		public int BufSize; 
		private bool disposed = false; 
		public const int size = 32;  
		private byte[] barr = null; 
		private short[] sarr = null; 
		// SDK: C-Definitions
		//    0: LPSTR       lpData;                 /* pointer to locked data buffer */
		//    4: DWORD       dwBufferLength;         /* length of data buffer */
		//    8: DWORD       dwBytesRecorded;        /* used for input only */
		//   12: DWORD_PTR   dwUser;                 /* for client's use */
		//   16: DWORD       dwFlags;                /* assorted flags (see defines) */
		//   20: DWORD       dwLoops;                /* loop control counter */
		//   24: struct wavehdr_tag FAR *lpNext;     /* reserved for driver */
		//   28: DWORD_PTR   reserved;               /* reserved for driver */

		/// <summary>
		/// Constructor, allocates new WAVEHDR structure including buffer in unmanaged memory 
		/// </summary>
		/// <param name="bufSize">buffer size in bytes</param>
		public WaveHeader(int bufSize)
		{
			BufSize = bufSize; 
			Handle = LocalAlloc(LPTR, size); 
			Buffer = LocalAlloc(LPTR, bufSize); 
			Marshal.WriteInt32(Handle, Buffer.ToInt32());                          // lpData
//			Marshal.WriteIntPtr(Handle, Buffer);                          // lpData
//			Marshal.WriteInt32(new IntPtr(Handle.ToInt32()+4), BufSize);  // dwBufferLength
			Marshal.WriteInt32(Handle, 4, BufSize);  // dwBufferLength
		}

		//properties
		public IntPtr lpData
		{
			  get{return new IntPtr(Marshal.ReadInt32(Handle)); }
//			get{return Marshal.ReadIntPtr(Handle); }
		}
		public int BufferLength
		{
			get{ return Marshal.ReadInt32(new IntPtr(Handle.ToInt32()+4)); }
		}
		public int BytesRecorded
		{
			get{ return Marshal.ReadInt32(new IntPtr(Handle.ToInt32()+8)); }
			set{ Marshal.WriteInt32(new IntPtr(Handle.ToInt32()+8), value); }
		}
		public int Flags
		{
			get{ return Marshal.ReadInt32(new IntPtr(Handle.ToInt32()+16)); }
			set{ Marshal.WriteInt32(new IntPtr(Handle.ToInt32()+16), value); }
		}

		/// <summary>
		/// returns unmanaged buffer as managed byte array (8 Bit)
		/// </summary>
		/// <returns>byte array</returns>
		public byte[] ToByteArray()
		{
			int len = BytesRecorded; 
			if (barr == null || barr.Length != len) // at first call and reset
				barr = new byte[len]; 
			Marshal.Copy(lpData, barr, 0, len); 
			return barr; 
		}

		/// <summary>
		/// returns unmanaged buffer as managed short array (16 bit)
		/// </summary>
		/// <returns>short array</returns>
		public short[] ToShortArray()
		{
			int len = BytesRecorded/2; 
			if (sarr == null || sarr.Length != len) // at first call and reset
				sarr = new short[len]; 
			Marshal.Copy(lpData, sarr, 0, len); 
			return sarr; 
		}

		public void Dispose()
		{
			disposed = true; 
			LocalFree(Buffer); 
			LocalFree(Handle); 
		}
		~WaveHeader()
		{
			if (!disposed) 
				Dispose(); 
		}
	}
}

