using System;
using System.Text; 
using System.Runtime.InteropServices;

namespace System.Sound
{
  #region enums
  /// <summary>
  /// Message-IDs for waveform input callbacks
  /// </summary>
  public enum MM_MIM : int
  {
    OPEN         = 0x3BE,           
    CLOSE        = 0x3BF,
    DATA         = 0x3C0,
  }
  /// <summary>
  /// Error codes for wavein and waveout operations
  /// </summary>
  public enum MMRESULT
  { 
    MMSYSERR_BASE         = 0,
    MMSYSERR_NOERROR      = 0,                    /* no error */
    MMSYSERR_ERROR        = (MMSYSERR_BASE + 1),  /* unspecified error */
    MMSYSERR_BADDEVICEID  = (MMSYSERR_BASE + 2),  /* device ID out of range */
    MMSYSERR_NOTENABLED   = (MMSYSERR_BASE + 3),  /* driver failed enable */
    MMSYSERR_ALLOCATED    = (MMSYSERR_BASE + 4),  /* device already allocated */
    MMSYSERR_INVALHANDLE  = (MMSYSERR_BASE + 5),  /* device handle is invalid */
    MMSYSERR_NODRIVER     = (MMSYSERR_BASE + 6),  /* no device driver present */
    MMSYSERR_NOMEM        = (MMSYSERR_BASE + 7),  /* memory allocation error */
    MMSYSERR_NOTSUPPORTED = (MMSYSERR_BASE + 8),  /* function isn't supported */
    MMSYSERR_BADERRNUM    = (MMSYSERR_BASE + 9),  /* error value out of range */
    MMSYSERR_INVALFLAG    = (MMSYSERR_BASE + 10), /* invalid flag passed */
    MMSYSERR_INVALPARAM   = (MMSYSERR_BASE + 11), /* invalid parameter passed */
    MMSYSERR_HANDLEBUSY   = (MMSYSERR_BASE + 12), /* handle being used */
    /* simultaneously on another */
    /* thread (eg callback) */
    MMSYSERR_INVALIDALIAS = (MMSYSERR_BASE + 13), /* specified alias not found */
    MMSYSERR_BADDB        = (MMSYSERR_BASE + 14), /* bad registry database */
    MMSYSERR_KEYNOTFOUND  = (MMSYSERR_BASE + 15), /* registry key not found */
    MMSYSERR_READERROR    = (MMSYSERR_BASE + 16), /* registry read error */
    MMSYSERR_WRITEERROR   = (MMSYSERR_BASE + 17), /* registry write error */
    MMSYSERR_DELETEERROR  = (MMSYSERR_BASE + 18), /* registry delete error */
    MMSYSERR_VALNOTFOUND  = (MMSYSERR_BASE + 19), /* registry value not found */
    MMSYSERR_NODRIVERCB   = (MMSYSERR_BASE + 20), /* driver does not call DriverCallback */
    MMSYSERR_MOREDATA     = (MMSYSERR_BASE + 21), /* more data to be returned */
    MMSYSERR_LASTERROR    = (MMSYSERR_BASE + 21), /* last error in range */
    WAVERR_BADFORMAT      = (WAVERR_BASE + 0),    /* unsupported wave format */
    WAVERR_STILLPLAYING   = (WAVERR_BASE + 1),    /* still something playing */
    WAVERR_UNPREPARED     = (WAVERR_BASE + 2),    /* header not prepared */
    WAVERR_SYNC           = (WAVERR_BASE + 3),    /* device is synchronous */
    WAVERR_LASTERROR      = (WAVERR_BASE + 3),    /* last error in range */
    WAVERR_BASE           = 32
  }

  public enum FormatFlags : uint
  {
    INVALIDFORMAT      = 0,       /* invalid format */
    WAVE_FORMAT_QUERY  = 1,
    WAVE_ALLOWSYNC     = 2,
    WAVE_MAPPED        = 4,
    WAVE_FORMAT_DIRECT = 8,

  }

  [Flags] public enum FormatTags : ushort
  {
    /// <summary> uncompressed PCM-Format</summary>
    _PCM    = 1       
  }
  public enum WAVEFORMAT : uint
  {
    /// <summary> 11.025 kHz, Mono,   8-bit  </summary>
    _1M08       = 0x00000001,
    /// <summary> 11.025 kHz, Stereo,   8-bit  </summary>
    _1S08       = 0x00000002,
    /// <summary> 11.025 kHz, Mono,   16-bit </summary>
    _1M16       = 0x00000004,       
    /// <summary> 11.025 kHz, Stereo,   16-bit </summary>
    _1S16       = 0x00000008,
    /// <summary> 22.05  kHz, Stereo, 8-bit </summary>
    _2M08       = 0x00000010,
    /// <summary> 22.05  kHz, Stereo, 8-bit </summary>
    _2S08       = 0x00000020,       
    /// <summary> 22.05  kHz, Mono,   16-bit </summary>
    _2M16       = 0x00000040,       
    /// <summary> 22,05   kHz, Stereo, 16-bit </summary>
    _2S16       = 0x00000080,       
    /// <summary> 44.1   kHz, Mono, 8-bit </summary>
    _4M08       = 0x00000100,       
    /// <summary> 44.1   kHz, Stereo, 8-bit </summary>
    _4S08       = 0x00000200,       
    /// <summary> 44.1   kHz, Mono,16-bit </summary>
    _4M16       = 0x00000400,       
    /// <summary> 44.1   kHz, Stereo,16-bit </summary>
    _4S16       = 0x00000800 
  }
  #endregion

  /// <summary>
	/// Encapsulates WaveIn-Device as defined in MMSystem.h for use with coredll.dll
	/// </summary>
  public class WaveIn
  {
    #region types
    /// <summary>
    /// delegate type for Wavein_Open callback
    ///  void CALLBACK waveInProc1(HWAVEIN hwi, UINT uMsg, DWORD dwInstance, DWORD dwParam1, DWORD dwParam2)
    /// </summary>
    private delegate void WaveInCb(uint msg, uint data, uint param1, uint param2);
    public delegate void WaveInCallback(uint msg, uint data);
    public delegate void WaveInCbSmart(int msg, int data, int param1, int param2);
    /// <summary>
    /// describes delegate for Open
    /// </summary>
    private enum CALLBACK : long
    {
      TYPEMASK   = 0x00070000L,    /* callback type mask */
      NULL       = 0x00000000L,    /* no callback */
      WINDOW     = 0x00010000L,    /* dwCallback is a HWND */
      //    TASK       = 0x00020000L,    /* dwCallback is a HTASK */
      FUNCTION   = 0x00030000L,    /* dwCallback is a FARPROC */
      //    THREAD     (CALLBACK_TASK)/* thread ID replaces 16 bit task */
      //    EVENT      = 0x00050000L,    /* dwCallback is an EVENT Handle */
    }

    /// <summary>
    /// Exception class combining error number and error text for MMResult value
    /// </summary>
    public class WaveInException : System.Exception
    {
      WaveInException(MMRESULT mmRes): base ("WaveInError " + mmRes.ToString() + "\n" + GetErrorText(mmRes))
      {}
    }
    #endregion
    #region Imports
    // Importe
    [DllImport("coredll.dll", SetLastError = true)]
    public static extern int PlaySound(string szSound, ref int hMod,  int
      playSoundFlags);
    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInGetErrorText(
      MMRESULT mmrError, 
      StringBuilder pszText, 
      uint cchText 
      ); 
    [DllImport("coredll.dll", SetLastError = true)]
    static extern int waveInGetNumDevs();
    /// <summary>
    /// Closes the specified waveform-audio input device. 
    /// Return Errors:
    /// MMSYSERR_NOERROR Success; 
    /// MMSYSERR_INVALHANDLE Specified device handle is invalid; 
    /// MMSYSERR_NODRIVER No device driver is present; 
    /// MMSYSERR_NOMEM Unable to allocate or lock memory; 
    /// WAVERR_STILLPLAYING There are still buffers in the queue; 
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device. If the function succeeds, 
    /// the handle is no longer valid after this call</param>
    /// <returns>Error code</returns>
    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInClose(System.IntPtr hwi);

    /// <summary>
    /// Opens a specified waveform input device for recording
    /// </summary>
    /// <param name="phwi">Address filled with a handle identifying 
    /// the open waveform-audio input device. Use this handle to identify 
    /// the device when calling other waveform-audio input functions. 
    /// This parameter can be NULL if WAVE_FORMAT_QUERY is specified for fdwOpen. </param>
    /// <param name="uDeviceID">Identifier of the waveform-audio input device to open. 
    /// It can be either a device identifier or a Handle to an open waveform-audio input device. 
    /// You can use the flag WAVE_MAPPER (0xffffffff) for automatic selection
    /// instead of a device identifier. </param>
    /// <param name="pwfx">Pointer to a WAVEFORMATEX structure that identifies the 
    /// desired format for recording waveform-audio data. You can free this structure 
    /// immediately after waveInOpen returns. </param>
    /// <param name="dwCallback">Specifies the address of a fixed callback function, 
    /// an event handle, a handle to a window, or the identifier of a thread to be called during 
    /// waveform-audio recording to process messages related to the progress of recording. If no 
    /// callback function is required, this value can be zero. </param>
    /// <param name="dwInstance">Specifies user-instance data passed to the callback mechanism. 
    /// This parameter is not used with the window callback mechanism. </param>
    /// <param name="fdwOpen">Flags for opening the device as defined in FormatFlags enumeration</param>
    /// <returns>Errorcode</returns>
    [DllImport("coredll.dll", EntryPoint = "waveInOpen", SetLastError = true)]
    static extern MMRESULT waveInOpenHWin(out System.IntPtr phwi, int uDeviceID, WaveFormatEx pwfx, 
      System.IntPtr hWin, uint dwInstance, FormatFlags fdwOpen);

    /// <summary>
    /// overload for querying device
    /// </summary>
    [DllImport("coredll.dll", EntryPoint = "waveInOpen", SetLastError = true)]
    static extern MMRESULT waveInOpenQuery(uint handle, int uDeviceID, WaveFormatEx pwfx, 
      uint dwCallback, uint dwInstance, FormatFlags fdwOpen);
    
    /// <summary>
    /// Retrieves the capabilities of a specified waveform-audio input device
    /// </summary>
    /// <param name="uDeviceID">Identifier of the waveform-audio output device. It can be either 
    /// a device identifier or a Handle to an open waveform-audio input device. </param>
    /// <param name="pwic">Pointer to a WAVEINCAPS structure to be filled with information about 
    /// the capabilities of the device. </param>
    /// <param name="cbwic">Size, in bytes, of the WAVEINCAPS structure</param>
    /// <returns>Error code</returns>

    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInGetDevCaps(uint uDeviceID, WaveInCaps pwic, int cbwic ); 
    /// <summary>
    /// sends an input buffer to the specified waveform-audio input device. When the buffer is filled, 
    /// the application is notified. Buffer must be prepared with the waveInPrepareHeader(). 
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    /// <param name="pwh">Pointer to a WAVEHDR structure that identifies the buffer</param>
    /// <param name="cbwh">Size, in bytes, of the WAVEHDR structure</param>
    /// <returns>MMSYSERR_NOERROR=Success; 
    /// MMSYSERR_INVALHANDLE=Specified device handle is invalid;
    /// MMSYSERR_INVALPARAM=The buffer's base address is not aligned with the sample size;
    /// MMSYSERR_NODRIVER No device driver is present. 
    /// MMSYSERR_NOMEM Unable to allocate or lock memory. 
    /// </returns>
    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInAddBuffer(IntPtr hwi, IntPtr pwh, int cbwh); 
    /// <summary>
    /// prepares a buffer for waveform input.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    /// <param name="pwh">Pointer to a WAVEHDR structure that identifies the buffer</param>
    /// <param name="cbwh">Size, in bytes, of the WAVEHDR structure</param>
    /// <returns>MMSYSERR_NOERROR=Success; 
    /// MMSYSERR_INVALHANDLE=Specified device handle is invalid;
    /// MMSYSERR_INVALPARAM=The buffer's base address is not aligned with the sample size;
    /// MMSYSERR_NODRIVER=No device driver is present. 
    /// MMSYSERR_NOMEM=Unable to allocate or lock memory. 
    /// </returns>
    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInPrepareHeader(IntPtr hwi, IntPtr pwh, int cbwh ); 

    /// <summary>
    /// cleans up the preparation performed by waveInPrepareHeader. The function must 
    /// be called after the device driver fills a data buffer and returns it to the application. 
    /// You must call this function before freeing the data buffer.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    /// <param name="pwh">Pointer to a WAVEHDR structure that identifies the buffer</param>
    /// <param name="cbwh">Size, in bytes, of the WAVEHDR structure</param>
    /// <returns>MMSYSERR_NOERROR=Success; 
    /// MMSYSERR_INVALHANDLE=Specified device handle is invalid;
    /// MMSYSERR_INVALPARAM=The buffer's base address is not aligned with the sample size;
    /// MMSYSERR_NODRIVER=No device driver is present. 
    /// MMSYSERR_NOMEM=Unable to allocate or lock memory. 
    /// WAVERR_STILLPLAYING=Buffer pointed to by the pwh parameter is still in the queue.
    /// </returns>
    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInUnprepareHeader(IntPtr hwi, IntPtr pwh, int cbwh);

    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInStart(IntPtr hwi);

    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInStop(IntPtr hwi);

    /// <summary>
    /// stops input on a specified waveform input device and resets the current position to 0. All 
    /// pending buffers are marked as done and returned to the application.
    /// </summary>
    /// <param name="hwi">Handle to the waveform-audio input device.</param>
    /// <returns>MMSYSERR_NOERROR=Success; 
    /// MMSYSERR_INVALHANDLE=Specified device handle is invalid
    /// MMSYSERR_NODRIVER No device driver is present. 
    /// MMSYSERR_NOMEM Unable to allocate or lock memory. </returns>
    [DllImport("coredll.dll", SetLastError = true)]
    static extern MMRESULT waveInReset(IntPtr hwi);

    #endregion 
    #region static
    /// <summary>
    /// Retrieves error text for error code
    /// </summary>
    /// <param name="mmrError">error number</param>
    /// <returns>Error text</returns>
    public static string GetErrorText(MMRESULT mmrError)
    {
      const int MAXERRORLENGTH  = 256; 
      StringBuilder pszText = new StringBuilder(MAXERRORLENGTH);
      waveInGetErrorText(mmrError, pszText, MAXERRORLENGTH); 
      return pszText.ToString();
    }
    /// <summary>
    /// Retrieves the capabilities of a specified waveform-audio input device.
    /// </summary>
    /// <param name="devID">Identifier of the waveform-audio output device. 
    /// It can be either a device identifier or a Handle to an open waveform-audio input device</param>
    /// <param name="wic">Pointer to a WAVEINCAPS structure to be filled with information about 
    /// the capabilities of the device.</param>
    public static void GetDevCaps(uint devID, WaveInCaps wic)
    {
      HandleErr(waveInGetDevCaps(devID, wic, Marshal.SizeOf(wic)));
    }

    /// <summary>
    /// Retrieves number of installed wavein devices
    /// </summary>
    /// <returns>Number of wavein devices</returns>
    public static int GetNumDevs()
    { 
      return waveInGetNumDevs(); 
    }

    /// <summary>
    /// Tests if device supports requested wave format 
    /// </summary>
    /// <param name="wfe">Address of Structure describing desired wave format</param>
    /// <param name="devId">Id of desired wavein device. WAVEMAPPER (= -1) selects 
    /// next available device</param>
    /// <returns>true, if format is supported</returns>
    public static bool Query(WAVEFORMAT format, int devId)
    {
      WaveFormatEx wfe = new WaveFormatEx(format); 
      FormatFlags flags = FormatFlags.WAVE_FORMAT_QUERY; 
      MMRESULT mres = waveInOpenQuery(0, devId, wfe, 0, 0, flags); 
      return mres == MMRESULT.MMSYSERR_BASE;
    }

    private static void HandleErr(MMRESULT res)
    {
      if (res != MMRESULT.MMSYSERR_NOERROR)
        throw new Exception(GetErrorText(res)); 
    }
    #endregion
    #region dynamic
    private WaveIn.WaveInCallback cbWin = null; 
    private System.IntPtr hWave = System.IntPtr.Zero;
    private bool _isOpen = true; 
    public bool IsOpen
    {
      get
      {
        return _isOpen; 
      }
    }

    /// <summary>
    /// Opens wavein device. Throws WaveInException if devId is not available or format not 
    /// supported by device
    /// </summary>
    /// <param name="wfe">Address of Structure describing desired wave format</param>
    /// <param name="devId">Id of desired wavein device. WAVEMAPPER (= -1) selects 
    /// next available device</param>
    /// <param name="hWin">Window handle for message window receiving callback messages </param>
    public WaveIn(WaveFormatEx wfe, int devId, System.IntPtr hWin)
    {
      HandleErr(waveInOpenHWin(out hWave, devId, wfe, hWin, 11, (FormatFlags)CALLBACK.WINDOW)); 
			Reset();
    }



    /// <summary>
    /// Constructor: Opens wavein device. Throws WaveInException if devId is not available or format not 
    /// supported by device.
    /// </summary>
    /// <param name="format">Description of desired wave format</param>
    /// <param name="devId">Id of desired wavein device. WAVEMAPPER (0xffffffff) selects 
    /// next available device</param>
    /// <param name="hWin">Window handle for message window receiving callback messages </param>
    public WaveIn(WAVEFORMAT format, int devId, System.IntPtr hWin) : 
      this(new WaveFormatEx(format), devId, hWin)
    {
    }
    /// <summary>
    /// Constructor: Opens wavein device using WAVEMAPPER. Throws WaveInException if format is not 
    /// supported by device
    /// </summary>
    /// <param name="format">Description of desired wave format</param>
    /// <param name="hWin">Window handle for message window receiving callback messages </param>
    public WaveIn(WAVEFORMAT format, System.IntPtr hWin):
      this(new WaveFormatEx(format), -1, hWin)
    {
    }
    /// <summary>
    /// prepares Wave Header for waveform input.
    /// </summary>
    /// <param name="wHdr"></param>
    public void PrepareHeader(WaveHeader wHdr)
    {
      HandleErr(waveInPrepareHeader(hWave, wHdr.Handle, WaveHeader.size)); 
    }

    /// <summary>
    /// cleans up the preparation performed by PrepareHeader(). The function must be called after 
    /// the device driver fills a data buffer and returns it to the application. 
    /// You must call this function before freeing the data buffer.
    /// </summary>
    /// <param name="wHdr">Wave Header with buffer</param>
    public void UnPrepareHeader(WaveHeader wHdr)
    {
      HandleErr(waveInUnprepareHeader(hWave, wHdr.Handle, WaveHeader.size)); 
    }

    public void Start()
    {
      HandleErr(waveInStart(hWave)); 
    }
    public void Stop()
    {
      HandleErr(waveInStop(hWave)); 
    }
    public void Reset()
    {
      HandleErr(waveInReset(hWave)); 
    }

    /// <summary>
    /// Adds a wave buffer to the wavein device. 
    /// </summary>
    /// <param name="wHdr">Wave header with unmanaged buffer</param>
    public void AddBuffer(WaveHeader wHdr)
    {
      HandleErr(waveInAddBuffer(hWave, wHdr.Handle, WaveHeader.size)); 
    }
    /// <summary>
    /// Closes wavein device. Throws WaveInException if internal handle is not valid.
    /// </summary>
    /// <param name="hin">handle for wavein device</param>
    public void Close()
    {
      if (_isOpen)
      {
        _isOpen = false; 
        HandleErr(waveInClose(hWave));
      }
      hWave = IntPtr.Zero; 
      GC.KeepAlive(cbWin); 
      cbWin = null; 
    }

    ~WaveIn()
    {
      Close(); 
    }
    #endregion
  }

	#region WaveFormatStrukturen
  /// <summary>
  /// headerblock for WaveIn operations (defines WAVEFORMATEX from mmsystem.h)
  /// </summary>
  [StructLayout(LayoutKind.Sequential)]
  public class WaveFormatEx
  {
    /// <summary>
    /// Constructor, initializes structure with given wave format
    /// </summary>
    /// <param name="FormatTag">Format description: mono/stereo, kHz, 8/16 bit</param>
    public WaveFormatEx(WAVEFORMAT FormatTag)
    {
      wFormatTag = FormatTags._PCM;   // uncompressed
      uint ft = (ushort)FormatTag; 
      bool done = false;
      nSamplesPerSec = 0;
      nChannels = 0; 
      wBitsPerSample = 0; 
      cbSize =0; 
      for(nSamplesPerSec = 11025; !done && nSamplesPerSec < 50000; nSamplesPerSec*=2)
      {
        for (ushort i = 0; i < 4 && !done; i++, ft >>= 1)
        {
          nChannels = (ushort) (i % 2 + 1); 
          wBitsPerSample = (ushort) ((i / 2+1) * 8); 
          if((ft & 1) == 1)
            done = true; 
        }
      }
      nSamplesPerSec/=2; 
      nBlockAlign = (ushort) (nChannels * wBitsPerSample / 8); 
      nAvgBytesPerSec = nSamplesPerSec * nBlockAlign; 
			cbSize = 0; 
    } 
    /// <summary> 
    /// Format type described by WaveFormats enumeration 
    /// </summary>
    public FormatTags    wFormatTag;         
    /// <summary>
    /// Number of channels (i.e. mono, stereo...) as defined by <see cref="wFormatTag"/>
    /// </summary>
    public ushort        nChannels;          
    /// <summary>
    /// Sample rate as defined by <see cref="wFormatTag"/>
    /// </summary>
    public uint          nSamplesPerSec;     
    /// <summary>
    /// For buffer estimation 
    /// </summary>
    public uint          nAvgBytesPerSec;    
    /// <summary>
    /// Block size of data
    /// </summary>
    public ushort        nBlockAlign;        
    /// <summary>
    /// number of bits per sample of mono data as defined by <see cref="wFormatTag"/>
    /// </summary>
    public ushort        wBitsPerSample;     
    /// <summary>
    /// Count in bytes of the size of extra information (after cbSize) 
    /// </summary>
    private ushort       cbSize;             
  }

  [StructLayout(LayoutKind.Sequential, Size = 80)]
  public class WaveInCaps
  {
    /// <summary>
    /// Helper class to reserve stack space for fixed size C-style char[]-Arrays
    /// </summary>
    private struct Eightchars
    {
      char c0, c1, c2, c3, c4, c5, c6, c7; 
      public new string ToString()
      {
        return "" + c0 + c1 + c2 + c3 + c4 + c5 + c6 + c7; 
      }
      public static string operator + (Eightchars a, Eightchars b)
      {
        return a.ToString() + b.ToString(); 
      }
      public static string operator + (string a, Eightchars b)
      {
        return a + b.ToString(); 
      }
    }
    /// <summary>
    /// driver name
    /// </summary>
    public string szName
    {
      get 
      { 
        return a0 + a1 + a2 + a3; 
      }
    }
    /// <summary>
    /// Manufacturer ID
    /// </summary>
    public ushort   wMid;                    
    /// <summary>
    ///  Product ID 
    /// </summary>
    public ushort   wPid;                  
    /// <summary>
    /// Version of the driver
    /// </summary>
    public byte ver0, ver1, ver2, ver3;   // vDriverVersion;  
    /// <summary>
    /// Product name (Marshalled from NULL terminated string)
    /// </summary>
    private Eightchars a0, a1, a2, a3; 
    /// <summary>
    /// Formats supported
    /// </summary>
    public uint     dwFormats;               
    /// <summary>
    /// Number of channels supported
    /// </summary>
    public short    wChannels;               
    /// <summary>
    /// structure packing
    /// </summary>
    public short    wReserved1;              
  }
#endregion
}

