/* $Id: padlock.c 42 2005-05-03 05:40:16Z olau $ */

#include "padlock.h"

#ifdef DEBUG
#include <stdio.h>
#endif /* DEBUG */

#ifdef WIN32

/************************/
/*                      */
/* Windows stuff        */
/*                      */
/************************/


/* Abfrage, ob der Prozessor den CPUID-Befehl versteht.
0 (FALSE), wenn nicht, 1 (TRUE), wenn ja. */
bool
has_cpuid() {
  __asm {
    pushfd               ; Save EFLAGS to stack
      pop eax              ; Store EFLAGS in EAX
      mov ebx, eax         ; Save in EBX for testing later
      xor eax, 00200000h   ; Switch bit 21
      push eax             ; Copy changed value to stack
      popfd                ; Save changed EAX to EFLAGS
      pushfd               ; Push EFLAGS to top of stack
      pop eax              ; Store EFLAGS in EAX
      cmp eax, ebx         ; See if bit 21 has changed
      jz NO_CPUID          ; If no change, no CPUID
  }
  return TRUE;

  __asm {
NO_CPUID:
  }
  return FALSE;
}


#define PUSHREG() __asm { \
   __asm push   ebx \
   __asm push   ecx \
   __asm push   edx \
   __asm push   esi \
   __asm push   edi \
}
#define POPREG()  __asm { \
   __asm pop    edi \
   __asm pop    esi \
   __asm pop    edx \
   __asm pop    ecx \
   __asm pop    ebx \
}

/* PadLock-Einheit instruieren, den Schlssel frisch aus dem Speicher zu laden */
__forceinline void
padlock_reload_key() {
  __asm
  {
    pushf
    popf
  };
}


/* If a CPUID with EAX=0xC0000000 returns a value in EAX >= 0xC0000001
   then Centaur Extended Feature Flags are supported. A CPUID with
   EAX=0xC0000001 then returns the Centaur Extended Feature Flags in
   EDX.  There are two bits in EDX that describe the ACE:

   EDX[6] == 0 ACE does not exist on this chip. FCR[28] cannot be set
   to enable the ACE and the new xcrypt instructions cause an Invalid
   Opcode Fault.

   EDX[6] == 1 ACE exists. The behavior of xcrypt instructions is
   dependent upon whether the ACE is enabled or not.

   EDX[7] == 0 ACE is disabled. The new xcrypt instructions cause an
   Invalid Opcode Fault. If the ACE is present, FCR[28] can be set to
   enable the ACE

   EDX[7] == 1 ACE is enabled (usually the RESET default. The xcrypt
   instructions behave as defined in this application note (provided
   CR4[9]=1)
*/
unsigned int
padlock_status() {
  uint32_t result;
  /* does CPU support the Centaur Extended Feature Flags? */
  PUSHREG();
  __asm 
  {
    mov    eax, 0c0000000h
    cpuid
    mov    result, eax
  };
  if (result < 0xc0000001U) {
    POPREG();
    return 0;
  }
  /* calling CPUID with EAX=c0000001h returns extended feature 
    flags in EDX. If EDX[6] is set, the PadLock(tm) engine is
    available. If EDX[7] is set, the PadLock(tm) engine is
    enabled. 
  */
  __asm
  {
    mov    eax, 0c0000001h
    cpuid
    mov    result, edx
  };
  POPREG();
  return (unsigned int) result & (PADLOCK_EXISTS | PADLOCK_ENABLED);
}


/* Gre der Cache Line ermitteln (bei C3 mormalerweise 32 Byte) */
__forceinline unsigned int 
get_cache_line_size() {
  uint32_t cacheline;
  __asm 
  {
    mov   eax, 80000006h
	  cpuid
    mov   cacheline, eax
  };
  return (unsigned int) cacheline & 0x000000ffU;
}


/* Vorlage fr xcrypt-Funktionsgenerator (s.u.) */
#define PADLOCK_XCRYPT_ASM(name, xcrypt) \
__forceinline size_t \
name(const void *ibuf, \
     const void *obuf, \
     size_t cnt, \
     const AES_KEY *key, \
     controlword_t *ctl, \
     const void *iv) { \
  size_t bytesstored; \
  __asm { \
    __asm mov eax, iv \
    __asm mov ebx, key \
    __asm mov ecx, cnt \
    __asm mov edx, ctl \
    __asm mov esi, ibuf \
    __asm mov edi, obuf \
    __asm _emit 0xf3 \
    __asm _emit 0x0f \
    __asm _emit 0xa7 \
    __asm _emit xcrypt \
  }; \
  return bytesstored; \
}

/* xcrypt-Funktionen erzeugen */
PADLOCK_XCRYPT_ASM(padlock_xcrypt_ecb, 0xc8);
PADLOCK_XCRYPT_ASM(padlock_xcrypt_cbc, 0xd0);
PADLOCK_XCRYPT_ASM(padlock_xcrypt_cfb, 0xe0);
PADLOCK_XCRYPT_ASM(padlock_xcrypt_ofb, 0xe8);


#else

/************************/
/*                      */
/* Unix stuff           */
/*                      */
/************************/

/* Abfrage, ob der Prozessor den CPUID-Befehl versteht.
   0 (FALSE), wenn nicht, 1 (TRUE), wenn ja. */
bool has_cpuid() {
  unsigned int idflag;
  asm volatile ("pushf\n\t"
                "pop    %%eax\n\t"
                "mov    %%eax, %%ecx\n\t"
                "xor    $0x200000, %%eax\n\t"
                "push   %%eax\n\t"
                "popf\n\t"
                "pushf\n\t"
                "pop    %%eax\n\t"
                "xor    %%ecx, %%eax\n\t"
                "mov    %%eax, %0\n\t"
                "push   %%ecx\n\t"
                "popf\n"
                : "=r" (idflag)
                : :"%eax", "%ecx");
  return idflag;
}

#define PUSHREG() asm volatile ("pushl   %ebx\n\t" \
                                "pushl   %ecx\n\t" \
                                "pushl   %edx\n\t" \
                                "pushl   %esi\n\t" \
                                "pushl   %edi\n\t" )
#define POPREG()  asm volatile ("popl    %edi\n\t" \
                                "popl    %esi\n\t" \
                                "popl    %edx\n\t" \
                                "popl    %ecx\n\t" \
                                "popl    %ebx\n" )


/* PadLock-Einheit instruieren, den Schlssel frisch aus dem Speicher zu laden */
inline void padlock_reload_key() {
  asm volatile ("pushfl\n\t"
                "popfl\n");
}


/* If a CPUID with EAX=0xC0000000 returns a value in EAX >= 0xC0000001
   then Centaur Extended Feature Flags are supported. A CPUID with
   EAX=0xC0000001 then returns the Centaur Extended Feature Flags in
   EDX.  There are two bits in EDX that describe the ACE:

   EDX[6] == 0 ACE does not exist on this chip. FCR[28] cannot be set
   to enable the ACE and the new xcrypt instructions cause an Invalid
   Opcode Fault.

   EDX[6] == 1 ACE exists. The behavior of xcrypt instructions is
   dependent upon whether the ACE is enabled or not.

   EDX[7] == 0 ACE is disabled. The new xcrypt instructions cause an
   Invalid Opcode Fault. If the ACE is present, FCR[28] can be set to
   enable the ACE

   EDX[7] == 1 ACE is enabled (usually the RESET default. The xcrypt
   instructions behave as defined in this application note (provided
   CR4[9]=1)
*/
unsigned int padlock_status() {
  uint32_t register result;
  /* does CPU support the Centaur Extended Feature Flags? */
  PUSHREG();
  asm volatile ("movl    $0xc0000000, %%eax\n\t"
                "cpuid\n"
                : "=a" (result) );
  if (result < 0xc0000001U) {
    POPREG();
    return 0;
  }
  /* calling CPUID with EAX=c0000001h returns extended feature 
    flags in EDX. If EDX[6] is set, the PadLock(tm) engine is
    available. If EDX[7] is set, the PadLock(tm) engine is
    enabled. 
  */
  asm volatile ("movl    $0xc0000001, %%eax\n\t"
                "cpuid\n"
                : "=d" (result) );
  POPREG();
  return (unsigned int) result & (PADLOCK_EXISTS | PADLOCK_ENABLED);
}


/* Gre der Cache Line ermitteln (bei C3 mormalerweise 32 Byte) */
inline
unsigned int get_cache_line_size() {
  uint32_t cl;
  asm volatile ("movl   $0x80000006, %%eax\n\t"
				"cpuid\n\t"
				: "=c"(cl)
                );
  return cl & 0xff;
}


/* htonl()/ntohl()
   taken from http://www.logix.cz/michal/devel/padlock/openssl-0.9.7d-padlock-engine.diff
   and slightly modified
*/
inline
void padlock_bswapl(AES_KEY *ks) {
  size_t i = sizeof(ks->rd_key) / sizeof(ks->rd_key[0]);
  uint32_t *key = (uint32_t *) ks->rd_key;
  while (i--) {
	asm volatile ("bswapl  %[key]" : [key] "+r"(*key));
	key++;
  }
}


/* Vorlage fr xcrypt-Funktionsgenerator (s.u.) */
#define PADLOCK_XCRYPT_ASM(name, xcrypt) \
inline \
size_t name(const void *ibuf, \
                   const void *obuf, \
                   size_t cnt, \
                   const AES_KEY *key, \
                   controlword_t *ctl, \
                   const void *iv) { \
  size_t bytesstored; \
  char *nextblock = (char *)(ibuf + cnt); \
  asm volatile (".byte 0xf3,0x0f,0xa7," xcrypt "\n\t" \
                : "=S"(bytesstored) \
                : [nextblock] "m"(nextblock), \
                  "a"(iv), "b"(key), "c"(cnt), "d"(ctl), \
                  "D"(obuf), "S"(ibuf) \
                ); \
  return bytesstored; \
}

/* xcrypt-Funktionen erzeugen */
PADLOCK_XCRYPT_ASM(padlock_xcrypt_ecb, "0xc8");
PADLOCK_XCRYPT_ASM(padlock_xcrypt_cbc, "0xd0");
PADLOCK_XCRYPT_ASM(padlock_xcrypt_cfb, "0xe0");
PADLOCK_XCRYPT_ASM(padlock_xcrypt_ofb, "0xe8");

#endif /* WIN32 */

