/* $Id: sdktest.c 60 2005-05-06 11:36:47Z olau $ */

#ifdef WIN32

#include <windows.h>
#include "../padlock-sdk/windows/padlock.h"

#else

#include "../padlock-sdk/linux/include/padlock.h"

#endif

#ifdef __GNUC__
#include <unistd.h>
#endif

#include <ctype.h>
#include <getopt.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <malloc.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <assert.h>
#include <math.h>

#include "../types.h"
#include "../core/padlock.h"


static const char *appname = "sdktest";
static const char *appdescription = "c't PadLock SDK tests";
static const char *appversion = "$LastChangedRevision: 60 $";
static const char *appauthor = "Oliver Lau <ola@ct.heise.de>";


/* somewhat random default key */
static byte default_key[32] = {
  0x71, 0x63, 0xa1, 0x5b, 
  0xbf, 0x08, 0x3b, 0x2b,
  0xb4, 0x38, 0xc1, 0x9b,
  0x9a, 0x21, 0x72, 0x80,
  0x45, 0x59, 0x60, 0x33,
  0xf6, 0x2d, 0x74, 0x88,
  0x7f, 0x36, 0xb3, 0x04,
  0xa7, 0x2d, 0xbf, 0x4a };

/* somewhat random default initialization vector */
static byte default_iv[16] = {
  0x99, 0xcc, 0xdf, 0x02,
  0x66, 0xe9, 0xb9, 0x44,
  0x68, 0xe7, 0xbf, 0x25,
  0x15, 0xfd, 0xef, 0xaf };


struct algo_t {
  const char *name;
  const int mode;
};

static int keysizes[] = { KEY_128BITS, KEY_192BITS, KEY_256BITS, -1 };
static struct algo_t algorithms[] = {
  { "ECB", ACE_AES_ECB    },
  { "CBC", ACE_AES_CBC    },
  { "CFB", ACE_AES_CFB128 },
  { "OFB", ACE_AES_OFB128 },
  {     0, -1             },
};

struct keysize_t {
  const int hint;
  const int bytes;
};

static struct keysize_t keybytes[] = {
  { KEY_128BITS, 16 },
  { KEY_192BITS, 24 },
  { KEY_256BITS, 32 },
  {          -1, -1 },
};

void print_block(byte *block, int width, int height) {
  int i, j;
  for (i = 0; i < height; i++) {
    printf("  ");
    for (j = 0; j < width; j++) {
      printf(" %02x", block[i * width + j]);
      if (((j + 1) % 4) == 0)
        printf(" ");
    }
    printf("\n");
  }
}


/* usage() */
void usage(void) {
  printf("\nAufruf: %s <Datei_mit_Testdaten>\n", appname);
}


/* here we go ... */
int main(int argc, char *argv[]) {
  struct stat fattr;
  FILE *fp;
  char *inputfilename;
  char outputfilename[100];
  size_t bufsize;
  int mode;
  char *modename;
  int keylen;
  int i, j, k;
  byte *inbuf, *outbuf, *chkbuf;
  byte *key, *iv;
  AES_RESULT res;
  unsigned int clsize;
  unsigned int alignment;
  size_t filesize;
  size_t byteswritten;

  printf("%s (%s)\n%s\nCopyright (c) 2005 %s\nAlle Rechte vorbehalten.\n",
    appname, appversion, appdescription, appauthor);

  if (!has_cpuid()) {
    fprintf(stderr, "Fehler: CPUID-Instruktion nicht verfgbar\n");
    return 1;
  }

  clsize = get_cache_line_size();
  alignment = ((clsize & 0x0f) != 0)? PADLOCK_BLOCK_BOUNDARY : clsize;
#ifdef DEBUG
  printf("cache line size = %u; alignment = %u\n", clsize, alignment);
#endif

  if ((padlock_status() & PADLOCK_EXISTS) == 0) {
    fprintf(stderr, "Fehler: PadLock(tm) nicht verfgbar\n");
    return 1;
  }
  if ((padlock_status() & PADLOCK_ENABLED) == 0) {
    fprintf(stderr, "Fehler: PadLock(tm) verfgbar, aber nicht eingeschaltet\n");
    return 1;
  }
  if (argc != 2) {
    usage();
    return 1;
  }
  inputfilename = argv[1];
  if (stat(inputfilename, &fattr) != 0) {
    perror("Dateizugriff fehlgeschlagen");
    return 1;
  }

  /* append padding if necessary */
  bufsize = ((fattr.st_size & 0xf) != 0)
    ? (fattr.st_size + PADLOCK_BLOCK_BOUNDARY) & 0xfffffff0UL 
    : fattr.st_size;

  /* reserve memory for input and output buffers */
#ifdef WIN32
  inbuf  = (byte *) VirtualAlloc(NULL, bufsize, MEM_COMMIT, 0);
  outbuf = (byte *) VirtualAlloc(NULL, bufsize, MEM_COMMIT, 0);
  chkbuf = (byte *) VirtualAlloc(NULL, bufsize, MEM_COMMIT, 0);
  key    = (byte *) VirtualAlloc(NULL, MAX_KEY_LENGTH, MEM_COMMIT, 0);
  iv     = (byte *) VirtualAlloc(NULL, IV_LENGTH, MEM_COMMIT, 0);
#else
  inbuf   = (byte *) memalign(alignment, bufsize);
  outbuf  = (byte *) memalign(alignment, bufsize);
  chkbuf  = (byte *) memalign(alignment, bufsize);
  memset(inbuf, 0, bufsize);
  key = (byte *) memalign(alignment, MAX_KEY_LENGTH);
  iv  = (byte *) memalign(alignment, IV_LENGTH);
#endif /* WIN32 */

  /* read input file */
  printf("Lesen der Datei '%s' .. ", inputfilename);
  fflush(stdout);
  fp = fopen(inputfilename, "rb");
  filesize = fread(inbuf, 1, fattr.st_size, fp);
  fclose(fp);
  if (filesize != bufsize)
    printf("\nACHTUNG! %lu Bytes Padding angehngt.\n",
           (unsigned long) (bufsize - filesize));

#ifdef DEBUG
  printf("\ninbuf[%u]:\n", (unsigned int) bufsize);
  print_block(inbuf, 16, 1);
#endif

  for (i = 0; keysizes[i] != -1; i++) {
    int keysize = keysizes[i];
    for (j = 0; algorithms[j].name != 0; j++) {
      mode = algorithms[j].mode;
      modename = (char *) algorithms[j].name;
      for (k = 0; keybytes[k].hint != -1; k++) {
        if (keybytes[k].hint == keysize)
          break;
      }
      keylen = keybytes[k].bytes;
      memcpy(key, default_key, keylen);
      printf("\n%3s %3u\n",
        algorithms[j].name,
        keylen * 8);

      memset(outbuf, 0, bufsize);
      memset(chkbuf, 0, bufsize);

      printf("  Verschlsselung .. ");
      fflush(stdout);

      memcpy(iv, default_iv, 16);
      res = padlock_aes_fast_encrypt(key, keysize, mode, inbuf, outbuf, bufsize, iv);
      if (res != AES_SUCCEEDED) {
        fprintf(stderr, "padlock_aes_fast_encrypt() fehlgeschlagen\n");
        return 1;
      }
#ifdef DEBUG
      printf("\n  outbuf[%u]:\n", bufsize);
      print_block(outbuf, 16, 1);
#endif

      sprintf(outputfilename, "%s-%d-%s", inputfilename, keylen*8, modename);
      printf("  Daten schreiben in Datei '%s' .. ", outputfilename);
      fflush(stdout);
      fp = fopen(outputfilename, "wb+");
      byteswritten = fwrite(outbuf, 1, bufsize, fp);
      fclose(fp);
      printf("%lu Bytes geschrieben.\n", (unsigned long) byteswritten);

      printf("  Entschlsselung .. ");
      fflush(stdout);

      memcpy(iv, default_iv, 16);
      res = padlock_aes_fast_decrypt(key, keysize, mode, outbuf, chkbuf, bufsize, iv);
      if (res != AES_SUCCEEDED) {
        fprintf(stderr, "padlock_aes_fast_decrypt() fehlgeschlagen\n");
        return 1;
      }

      if (memcmp(inbuf, chkbuf, bufsize) == 0) {
        printf("\n\n !!!!!!!!!!!!!  OK  !!!!!!!!!!!!!!!!!\n\n");
        print_block(chkbuf, 16, 1);
        fflush(stdout);
      }
      else {
        printf("\n\n>>>>>>>>>>>>>>FEHLER<<<<<<<<<<<<<<<<<\n\n");
        printf("chkbuf[%u]:\n", bufsize);
        print_block(chkbuf, 16, 1);
      }

    }
  }
  printf("\nFertig.\n");
  return 0;
}
