<?php

/*
Klasse zum Bearbeiten von Dokumentvorlagen im ODF- und OpenXML-Format
(c) Herbert Braun, c't, August 2006 | heb at ct Punkt heise Punkt de
zu: Herbert Braun, Bürodynamik, ODF- und OpenXML-Dokumente mit PHP bearbeiten,
c't 18/06, S. 206
Bearbeiten und Weitergeben des Quelltexts gestattet, wenn dieser Header
erhalten bleibt.
Dieses Skript hat eher Demo-Charakter, als dass es sich für Produktionseinsatz
eignet. Es soll die Möglichkeiten veranschaulichen, ODF- und OpenXML-Dateien
mit PHP und anderen Skriptsprachen zu bearbeiten.
Nicht auf öffentlich zugänglichen Webseiten einsetzen!
*/


class Odf {
 protected $dateiname;	// das Office-Dokument (ohne Dateiendung)
 protected $dateiendung;	// Dateiendung des Office-Dokuments
 protected $format;	// "ODF" oder "OpenXML"
 protected $xmldatei;	// gepackte XML-Datei mit Inhalten im Office-Dokument
 protected $xmlverzeichnis;	// Pfad der XML-Datei im Office-Dokument
 protected $xmlinhalt;	// Inhalt der XML-Datei
 protected $tmpverzeichnis;	// Pfad für Vorschau und XML-Inhalt
 protected $ausgabeverzeichnis;	// Schreibpfad für Dokumente
 protected $vorschau;	// Vorschaubild (nur ODF)
 protected $platzhalter_anfang;	// kennzeichnet Platzhalter im Dokument
 protected $platzhalter_ende;
 protected $serienbrief = false;	// Vorlage ist Serienbrief (nur ODF)
 protected $dom;	// Dokument-Objekt, für Serienbrief
 protected $ser_feld;	// Seriendruckfelder im Dokument-Objekt


 // Ermittelt die Inhaltsdateien für das jeweilige Office-Dokument
 // 1. Parameter: Dokumentname (inkl. Pfad)
 // 2. Parameter: Arbeitsblatt (Excel) oder Folie (PowerPoint)
 // (nur mit OpenOffice Writer und Microsoft Word 2007 getestet!)
 // nutzt: $this->checkPath()
 // setzt: $this->dateiname, $this->dateiendung, $this->format,
 // $this->xmldatei, $this->xmlverzeichnis, $this->vorschau
 public function setNames($_name, $_arbeitsblatt) {
  foreach (array($_name, $_arbeitsblatt) as $_para) {
   if ($_para) $_para = $this->checkPath($_para);
  }
  if (!$_name) throw new Exception("Kein Dateiname!", 1);
  // trennt Dateiendung vom Dateinamen ab
  preg_match('/(.+)\.(\w{3,4})$/', $_name, $_tmp);
  $this->dateiname = $_tmp[1];
  $this->dateiendung = $_tmp[2];
  // ODF-Dokumente (.odw etc.)
  if (preg_match('/o[td][a-z]/', $this->dateiendung)) {
   $this->format = 'ODF';
   $this->xmldatei = 'content.xml';
   $this->xmlverzeichnis = '';
   $this->vorschau = 'Thumbnails/thumbnail.png';
  } else if (preg_match('/(do|xl|pp|po)[ctsa][xm]/', $this->dateiendung, $_tmp)) {
   // OpenXML-Dokumente (.docx etc)
   $this->format = 'OpenXML';
   // unterschiedliche Archivdateinamen bei Word, Excel, PowerPoint
   switch($_tmp[1]) {
    case 'do':
     $this->xmlverzeichnis = 'word/';
     $this->xmldatei = 'document.xml';
     break;
    case 'xl':
     $this->xmlverzeichnis = 'xl/worksheets/';
     $this->xmldatei = $_arbeitsblatt? $_arbeitsblatt : 'sheet1.xml';
     break;
    default:
     $this->xmlverzeichnis = 'ppt/slides/';
     $this->xmldatei = $_arbeitsblatt? $_arbeitsblatt : 'slide1.xml';
   }
   $this->vorschau = '';
  } else {
   throw new Exception("Kein zulässiges Dateiformat: $_name", 2);
  }
 }


 // Definiert das Verzeichnis, in das die Vorschau und die
 // temporären XML-Dateien geschrieben werden, und legt es ggf. an.
 // Parameter: Verzeichnisname
 // setzt: $this->tmpverzeichnis
 public function setTmpDir($_verzeichnis) {
  if (!($_verzeichnis = $this->checkPath($_verzeichnis))) throw new Exception("Kein temporäres Verzeichnis angegeben!", 10);
  // hängt einen Slash an den Verzeichnisnamen
  if (!preg_match('/\/$/', $_verzeichnis)) $_verzeichnis .= '/';
  $this->tmpverzeichnis = $_verzeichnis;
  // legt Verzeichnis im Dateisystem an, falls nicht vorhanden
  if (!is_dir($this->tmpverzeichnis)) mkdir($this->tmpverzeichnis);
 }


 // Definiert das Verzeichnis, in das die geänderten
 // Dokumente geschrieben werden, und legt es ggf. an.
 // Parameter: Verzeichnisname
 // setzt: $this->ausgabeverzeichnis
 public function setOutputDir($_verzeichnis) {
  if (!($_verzeichnis = $this->checkPath($_verzeichnis))) throw new Exception("Kein Ausgabeverzeichnis angegeben!", 11);
  // hängt einen Slash an den Verzeichnisnamen
  if (!preg_match('/\/$/', $_verzeichnis)) $_verzeichnis .= '/';
  $this->ausgabeverzeichnis = $_verzeichnis;
  // legt Verzeichnis im Dateisystem an, falls nicht vorhanden
  if (!is_dir($this->ausgabeverzeichnis)) mkdir($this->ausgabeverzeichnis);
 }


 // Definiert die Platzhalterzeichen im Dokument
 // 1. Parameter: vor dem Platzhalter
 // 2. Parameter: nach dem Platzhalter
 // setzt: $this->platzhalter_anfang, $this->platzhalter_ende
 public function setSeparators($_anfang, $_ende) {
  $this->platzhalter_anfang = isset($_anfang)? $_anfang : '';
  $this->platzhalter_ende = isset($_ende)? $_ende : '';
 }


 // Legt fest, ob eine Dokumentenvorlage ein Serienbrief ist
 // Parameter: true = ja, false (Default) = nein
 // benötigt: $this->format mit dem Wert 'ODF'
 // setzt: $this->serienbrief
 public function isSerial($_true) {
  if ($this->format != 'ODF') throw new Exception('Serienbriefvorlagen sind bisher nur mit dem OpenDocument-Format möglich', 45);
  $this->serienbrief = isset($_true)? true : false;
 }


 // Liest den Inhalt der zu ändernden XML-Datei ein, schreibt sie in
 // $this->xmlinhalt und gibt sie zurück
 // benötigt: $this->dateiname, $this->dateiendung, $this->xmldatei
 // nutzt: $this->openArchive()
 // setzt: $this->xmlinhalt
 public function getXmlContent() {
  if (!($this->dateiname && $this->dateiendung && $this->xmldatei)) throw new Exception("Variablen nicht gesetzt", 50);
  return $this->xmlinhalt = $this->openArchive($this->dateiname . '.' . $this->dateiendung, $this->xmlverzeichnis . $this->xmldatei, false);
 }


 // Schreibt Vorschaubild im ODF-Dokument ins temporäre Verzeichnis
 // und gibt Bildname (inkl. Pfad) zurück
 // benötigt: $this->vorschau, $this->dateiname, $this->dateiendung,
 // $this->tmpverzeichnis
 // nutzt: $this->openArchive()
 public function writeThumbnail() {
  if (!$this->vorschau) return false;
  if (!($this->dateiname && $this->dateiendung && $this->tmpverzeichnis)) throw new Exception("Variablen für Vorschau nicht gesetzt", 51);
  $_bild = $this->openArchive($this->dateiname . '.' . $this->dateiendung, $this->vorschau, $this->tmpverzeichnis);
  $_h = getDate();
  return sprintf($_bild . "?_%4d-%02d-%02d_%02d-%02d-%02d", $_h['year'], $_h['mday'], $_h['mon'], $_h['hours'], $_h['minutes'], $_h['seconds']);
 }


 // Ersetzt im Dokument die Schlüssel des übergebenen Arrays durch die Werte
 // ersetzt Platzhalter, die durch $this->platzhalter_anfang und
 // $this->platzhalter_ende gekennzeichnet sind, oder Serienbrieffelder
 // Parameter: benannter Array
 // benötigt: $this->xmlinhalt, $this->serienbrief, $this->format
 // nutzt: $this->cleanXml()
 // ändert: $this->xmlinhalt
 // zu tun: Serienbriefersetzung auch für OpenXML;
 // bedingte Felder in Serienbriefen löschen oder nutzen
 public function replaceFields($_felder_werte) {
  if (!$this->xmlinhalt) throw new Exception("Kein XML-Inhalt ausgelesen", 60);
  // Vorlage ist Serienbrief:
  // such Seriendruckfelder und vergleich sie mit den Array-Schlüsseln
  if ($this->serienbrief) {
   // Serienbriefersetzung ist nur für ODF umgesetzt
   if ($this->format != 'ODF') throw new Exception('Serienbriefvorlagen sind bisher nur mit dem OpenDocument-Format möglich', 45);
   // erzeugt ein DOM-Objekt aus dem String $this->xmlinhalt
   $this->dom = new DOMDocument;
   $this->dom->loadXML($this->xmlinhalt);
   // durchläuft alle Elemente <database-display>
   $this->ser_feld = $this->dom->getElementsByTagName('database-display');
   $_feld_nr = 0;
   while ($_feld = $this->ser_feld->item($_feld_nr)) {
    // der Feldname steht im Attribut column-name
    $_attr = $_feld->getAttribute('column-name');
    if (isset($_felder_werte[$_attr])) {
     $_wert = $_felder_werte[$_attr]? stripslashes($_felder_werte[$_attr]) : '';
     // entfernt Wagenrücklaufzeichen
     $_wert = str_replace("\r", '', $_wert);
     // trennt Textabsätze in Array auf
     $_absaetze = explode("\n\n", $_wert);
     // ermittelt den Absatz (Element p), in dem das Feld liegt
     $_paragraph = $_feld;
     while ($_paragraph) {
      $_paragraph = $_paragraph->parentNode;
      if ($_paragraph->nodeName == 'text:p') break;
     }
     if ($_paragraph->nodeName != 'text:p' && count($_absaetze) > 1) throw new Exception('Kein Absatz zum Duplizieren gefunden!', 46);

     // durchläuft alle Textabsätze (auch wenn es nur einer ist)
     $_abs_nr = 0;
     while (isset($_absaetze[$_abs_nr])) {
      // trennt an einfachen Umbrüchen in Array auf
      $_zeilen = explode("\n", $_absaetze[$_abs_nr]);
      // 2. oder weiterer Absatz: Absatzelement nach dem aktuellen einfügen
      if ($_abs_nr > 0) {
       $_paragraph = $_paragraph->parentNode->insertBefore($_paragraph->cloneNode(), $_paragraph->nextSibling);
      }
      // durchläuft alle Textzeilen (auch wenn es nur eine ist)
      $_zeile_nr = 0;
      while (isset($_zeilen[$_zeile_nr])) {
       // erzeugt Textknoten
       $_textknoten = $this->dom->createTextNode($_zeilen[$_zeile_nr]);
       // 1. Absatz, 1. Zeile: anstelle des Seriendruckfelds einhängen
       if ($_abs_nr == 0 && $_zeile_nr == 0) {
        $_feld->parentNode->replaceChild($_textknoten, $_feld);
       } else {
        // 2. oder weitere Zeile: Zeilenumbruch-Element und Textknoten einfügen
        if ($_zeile_nr > 0) {
         $_einhaengepunkt = $_paragraph->insertBefore($this->dom->createElement('text:line-break'), $_einhaengepunkt->nextSibling);
         $_paragraph->insertBefore($_textknoten, $_einhaengepunkt->nextSibling);
         // 2. oder weiterer Absatz, neu angelegt: Textknoten anhängen
        } else {
         $_paragraph->appendChild($_textknoten);
        }
       }
       $_einhaengepunkt = $_textknoten;
       $_zeile_nr++;
      }
      $_abs_nr++;
     }
    } else {
     // setzt den Zähler für nicht ersetzte Platzhalter hoch
     $_feld_nr++;
    }
   }
   // konvertiert XML in String zurück
   $this->xmlinhalt = $this->dom->saveXML();

   // Vorlage arbeitet mit Klartext-Platzhaltern:
   // such die übergebenen Array-Schlüssel im Dokument
  } else {
   // definiert reguläre Ausdrücke für Textabsätze
   $_abs_anfang = ($this->format == 'ODF')? '<text:p.[^>]*>' : '<w:p[^>]*> [^<]*? <w:r[^>]*> [^<]*? <w:t[^>]*>';
   $_abs_ende = ($this->format == 'ODF')? '<\/text:p>' : '<\/w:t> .*? <\/w:r> .*? <\/w:p>';
   // definiert das Zeichen für Zeilenumbrüche
   $_umbruch = ($this->format == 'ODF')? '<text:line-break/>' : '<w:br/>';
   foreach ($_felder_werte as $_feldname => $_wert) {
    if ($_feldname == 'dokument') continue;
    // beseitige XML-Müll, den Word 2007 ins Dokument schreibt
    if ($this->format == "OpenXML") $this->cleanXml($_feldname);
    $_platzhalter = $this->platzhalter_anfang . $_feldname . $this->platzhalter_ende;
    $_wert = stripslashes($_wert);
    // entfernt Wagenrücklaufzeichen
    $_wert = str_replace("\r", '', $_wert);
    // trennt Textabsätze in Array auf
    $_absaetze = explode("\n\n", $_wert);
    // wenn es mehrere Absätze gibt ...
    if (count($_absaetze) > 1) {
     // ... definere reguläre Ausdrücke für die Absatz-Tags,
     // um den kompletten Textabsatz einzufangen
     preg_match("/ ($_abs_anfang) [^<]* $_platzhalter [^<]* ($_abs_ende) /x", $this->xmlinhalt, $_catch);
     // $1 enthält den ganzen Suchtreffer, $2 und $3 die
     // Anfangs- und Endtags des Absatzes
     if (count($_catch) != 3) throw new Exception("Probleme mit dem Einfügen von Absätzen: $_feldname $_wert", 61);
     // entfernt evtl. Text zwischen den schließenden Tags (bei OpenXML)
     $_catch[2] = preg_replace('/>.*?</', '><', $_catch[2]);
     // setzt die Textabsätze als String zusammen
     $_wert = implode($_catch[2] . $_catch[1], $_absaetze);
    }
    // ersetzt Zeilenumbrüche
    $_wert = str_replace("\n", $_umbruch, $_wert);
    // ersetzt den Platzhalter
    $this->xmlinhalt = str_replace($_platzhalter, $_wert, $this->xmlinhalt);
   }
  }
 }


 // Schreibt die Office-Datei ins Ausgabeverzeichnis
 // kopiert XML-Datei in temporäres Verzeichnis
 // gibt den neuen Dateinamen inkl. Pfad zurück
 // benötigt: $this->tmpverzeichnis, $this->ausgabeverzeichnis,
 // $this->dateiname, $this->dateiendung, $this->xmldatei, $this->xmlinhalt
 // nutzt: $this->copyFile() und PclZip
 public function saveFile() {
  if (!($this->tmpverzeichnis && $this->ausgabeverzeichnis && $this->dateiname && $this->dateiendung && $this->xmldatei && $this->xmlinhalt)) throw new Exception("Fehlendes Ausgabeverzeichnis, Dateiname oder Dateiinhalt", 65);
  if (!file_put_contents($this->tmpverzeichnis . $this->xmldatei, $this->xmlinhalt)) throw new Exception("Konnte temporäre XML-Datei nicht schreiben", 66);
  // versieht Office-Datei mit Datumssuffix, um Überschreiben zu verhindern
  $_h = getDate();
  $_erweiterung = sprintf("_%4d-%02d-%02d_%02d-%02d-%02d_", $_h['year'], $_h['mday'], $_h['mon'], $_h['hours'], $_h['minutes'], $_h['seconds']);
  $_name_neu = $this->ausgabeverzeichnis . $this->dateiname . $_erweiterung;
  // zählt durch, falls in 1 Sekunde zwei Dokumente geschrieben werden
  $_index = 1;
  while (is_file($_name_neu . $_index . '.' . $this->dateiendung)) ++$_index;
  $_name_neu .= $_index . '.' . $this->dateiendung;
  // kopiert die Office-Datei
  $this->copyFile($this->dateiname . '.' . $this->dateiendung, $_name_neu);
  // bindet PclZip ein
  require_once('pclzip.lib.php');
  $archiv = new PclZip($_name_neu);
  // löscht XML-Datei mit Inhalten aus Office-Archiv
  if ($archiv->delete(PCLZIP_OPT_BY_NAME, $this->xmlverzeichnis . $this->xmldatei) == 0) throw new Exception("Fehler beim Aktualisieren des Zip-Archivs: " . $archiv->errorInfo(true));
  // kopiert die geänderte Datei aus dem temporären Verzeichnis hinein
  if ($archiv->add($this->tmpverzeichnis . $this->xmldatei, PCLZIP_OPT_REMOVE_PATH, $this->tmpverzeichnis, PCLZIP_OPT_ADD_PATH, $this->xmlverzeichnis) == 0) throw new Exception("Fehler beim Schreiben des Zip-Archivs: " . $archiv->errorInfo(true));
  return($_name_neu);
 }


 // Hilfsfunktion: Prüft, dass nicht auf Systemdateien zugegriffen wird
 // Parameter: Dateiname inkl. Pfad
 // gibt Dateinamen oder false zurück
 protected function checkPath($_pfad) {
  if (!$_pfad) return false;
  // ändert Backslashes in Slashes
  $_pfad = str_replace(array('\\','/'), '/', $_pfad);
  if (preg_match('/\.\.\//', $_pfad) || preg_match('/^\//', $_pfad)) throw new Exception("Argument $_pfad enthält unerlaubten Pfad", 30);
  if (!preg_match('/\w/', $_pfad)) return false;
  return $_pfad;
 }


 // Hilfsfunktion: Liest Dokument aus Zip-Archiv
 // 1. Parameter: Archivname
 // 2. Parameter: auszulesendes Dokument
 // 3. Parameter: Verzeichnis, in welches das Dokument geschrieben wird;
 // ist der 3. Parameter leer, gibt die Funktion nur den Dateiinhalt zurück
 // nutzt: $this->checkPath() und PclZip
 protected function openArchive($_archivname, $_dateiname, $_schreibpfad) {
  foreach (array($_archivname, $_dateiname) as $_para) {
   $_pfad = $this->checkPath($_para);
   if (!$_para) throw new Exception("Kein Archiv angegeben", 20);
  }
  $_schreibpfad = $this->checkPath($_schreibpfad);
  if (!file_exists($_archivname)) throw new Exception("Konnte $_archivname nicht finden", 21);
  // bindet PclZip ein
  require_once('pclzip.lib.php');
  $archiv = new PclZip($_archivname);
  // kein Schreibpfad -> Dokument in String entpacken
  if (!$_schreibpfad) {
   $_ret = $archiv->extract(PCLZIP_OPT_BY_NAME, $_dateiname, PCLZIP_OPT_EXTRACT_AS_STRING);
  // ansonsten Pfad der zu extrahierenden Datei vom Dateinamen abtrennen
  } else {
   preg_match('/^(.+)\/(.+)$/', $_dateiname, $_archivpfad);
   // Doubletten löschen - PclZip überschreibt Dateien nicht zuverlässig
   if (file_exists($_schreibpfad . $_archivpfad[2])) unlink($_schreibpfad . $_archivpfad[2]);
   // in Datei entpacken
   $_ret = $archiv->extract(PCLZIP_OPT_BY_NAME, $_dateiname, PCLZIP_OPT_PATH, $_schreibpfad, PCLZIP_OPT_REMOVE_PATH, $_archivpfad[1]);
  }
  // Rückgabewert 0 = Fehler
  if ($_ret == 0) throw new Exception("Probleme beim Entpacken: " . $archiv->errorInfo(true), 22);
  // gibt den Dateinamen zurück, falls eine Datei geschrieben wurde,
  // ansonsten den Inhalt
  // $_ret[0] ist ein benannter Array, der das gezippte Dokument enthält
  return $_schreibpfad? $_ret[0]['filename'] : $_ret[0]['content'];
 }


 // Hilfsfunktion: Beseitigt überflüssige XML-Tags aus Word-XML,
 // die sich zwischen die Platzhalter schieben
 // Parameter: Feldname
 // benötigt: $this->xmlinhalt, $this->platzhalter_anfang,
 // $this->platzhalter_ende
 // ändert: $this->xmlinhalt
 // zu tun: ähnliche Funktionen für Excel und PowerPoint
 protected function cleanXml($_feldname) {
  if (!($this->xmlinhalt && $this->platzhalter_anfang && $this->platzhalter_ende)) throw new Exception("Fehlende Parameter beim Säubern der OpenXML-Datei", 87);
  // beseitigt proofErr-Tags
  $this->xmlinhalt = preg_replace('/<w:proofErr[^\/]*\/>/', '', $this->xmlinhalt);
  $this->xmlinhalt = preg_replace('/<w:bookmark[^\/]*\/>/', '', $this->xmlinhalt);
  // beseitigt Tags, die sich zwischen den Platzhalter geschoben haben
  $this->xmlinhalt = preg_replace("/($this->platzhalter_anfang)<\/w:t>\s*<\/w:r>\s*<w:r[^>]*>\s*<w:t>($_feldname)/", "$1$2", $this->xmlinhalt);
  $this->xmlinhalt = preg_replace("/($_feldname)<\/w:t>\s*<\/w:r>\s*<w:r[^>]*>\s*<w:t>($this->platzhalter_ende)/", "$1$2", $this->xmlinhalt);
 }


 // Hilfsfunktion: Kopiert Office-Datei
 // 1. Parameter: Dateipfad der Quelle
 // 2. Parameter: Dateipfad des Ziels
 // nutzt $this->checkPath()
 protected function copyFile($_dateiname, $_dateiname_neu) {
  foreach (array($_dateiname, $_dateiname_neu) as $_pfad) $this->checkPath($_pfad);
  if (!copy($_dateiname, $_dateiname_neu)) throw new Exception("Kopieren der Datei $_dateiname nach $_dateiname_neu gescheitert", 70);
 }

}

?>
