namespace Sudoku
{
    public enum SudokuStatus
    {
        NoSolution, OneSolution, MoreSolutions, Timeout
    }

    public partial class SudokuField
    {
        public int Iterations;
        private int MaxIterations;
        private SudokuField Solution, SecondSolution;

        /// <summary>
        /// Solves actual SudokuField using smarter backtracking.
        /// </summary>
        /// <returns>NoSolution, OneSolution, MoreSolutions, Timeout</returns>
        public SudokuStatus Solve()
        {
            SudokuStatus Status = GetStatus(int.MaxValue);
            if (Status == SudokuStatus.OneSolution || Status == SudokuStatus.MoreSolutions)
                Copy(Solution);
            return Status;
        }

        /// <summary>
        /// Calculates the status of the actual SudokuField without changing its Digits and Lists.
        /// </summary>
        /// <param name="MaxNumberOfIterations">Max. number of Iterations</param>
        /// <returns>NoSolution, OneSolution, MoreSolutions, Timeout (result is in Solution)</returns>
        private SudokuStatus GetStatus(int MaxNumberOfIterations)
        {
            Iterations = 0;
            MaxIterations = MaxNumberOfIterations;
            return new SudokuField(this).InitLists().Search(this, SudokuStatus.NoSolution);
        }

        private SudokuStatus Search(SudokuField Parent, SudokuStatus Status)
        {
            // Try some logic first to reduce backtracking.
            SolveLogic();
            // Beginning with the empty cell with minimum possibility list saves calculation time.
            int Mini = -1, Minj = -1;
            SudokuList MinList = GetMinList(ref Mini, ref Minj);
            if (MinList == null)                      // solved, if no empty cell found
            {
                Parent.SecondSolution = Parent.Solution;
                Parent.Solution = this;
                return Status + 1;
            }
            if (++Parent.Iterations >= Parent.MaxIterations)
                return SudokuStatus.Timeout;
            foreach (int Digit in MinList)
            {
                Status = new SudokuField(this).       // copy this for backtracking,
                         Lock(Mini, Minj, Digit).     // set and lock new Digit,
                         Search(Parent, Status);      // and try to solve again
                if (Status >= SudokuStatus.MoreSolutions)
                    return Status;                    // don't search further
            }
            return Status;
        }

        private SudokuList GetMinList(ref int Mini, ref int Minj)
        {
            int MinCount = nn + 1;
            Mini = Minj = -1;
            for (int i = 0; i < nn; i++)
                for (int j = 0; j < nn; j++)
                    if (Digits[i, j] == 0)         // cell needs to be solved
                    {
                        int Count = 0, LastDigit = 0;
                        foreach (int Digit in Lists[i, j])
                        {
                            Count++;               // count possibilities
                            LastDigit = Digit;
                        }
                        if (Count == 0)            // cell has no possibilities
                            return Lists[i, j];
                        if (Count == 1)            // solve unique digit immediately
                            Lock(i, j, LastDigit);
                        if (Count < MinCount)      // minimum possibilites so far
                        {
                            MinCount = Count;
                            Mini = i;
                            Minj = j;
                        }
                    }
            return Mini >= 0 ? Lists[Mini, Minj] : null;
        }

        /// <summary>
        /// Implements logic solve methods A and B of http://www.sudokusolver.co.uk/solvemethods.html 
        /// to reduce the remaining possibilities (reduce backtracking).
        /// </summary>
        private void SolveLogic()
        {
            SolveLogicA();
            SolveLogicB();
        }

        private void SolveLogicA()
        {
            for (int Digit = 1; Digit <= nn; Digit++)
            {
                for (int i = 0; i < nn; i++)
                {
                    int Count = 0, ii = -1, jj = -1;
                    for (int j = 0; j < nn; j++)
                        if (TestLists(i, j, Digit))
                        {
                            if (++Count > 1)
                                break;
                            ii = i;
                            jj = j;
                        }
                    if (Count == 1)            // solve uniques immediately
                        Lock(ii, jj, Digit);
                }
                for (int j = 0; j < nn; j++)
                {
                    int Count = 0, ii = -1, jj = -1;
                    for (int i = 0; i < nn; i++)
                        if (TestLists(i, j, Digit))
                        {
                            if (++Count > 1)
                                break;
                            ii = i;
                            jj = j;
                        }
                    if (Count == 1)            // solve uniques immediately
                        Lock(ii, jj, Digit);
                }
                for (int b = 0; b < nn; b++)
                {
                    int Count = 0, ii = -1, jj = -1;
                    for (int i = n * (b % n); i < n * (b % n) + n; i++)
                        for (int j = n * (b / n); j < n * (b / n) + n; j++)
                            if (TestLists(i, j, Digit))
                            {
                                if (++Count > 1)
                                    break;
                                ii = i;
                                jj = j;
                            }
                    if (Count == 1)            // solve uniques immediately
                        Lock(ii, jj, Digit);
                }
            }
        }

        private void SolveLogicB()
        {
            for (int Digit = 1; Digit <= nn; Digit++)
            {
                for (int i = 0; i < nn; i++)
                {
                    int b1 = n;
                    for (int b = 0; b < n; b++)
                    {
                        for (int j = 0; j < n; j++)
                            if (TestLists(i, b * n + j, Digit))   // search for Digit in row i, block b
                            {
                                b1 = b1 == n || b1 == b ? b : -1; // set b1 to block b or -1, if more
                                break;                            // blocks are involved
                            }
                        if (b1 == -1)
                            break;
                    }                        // If Digit is only in row i at block b1 (no other blocks), 
                    if (b1 >= 0 && b1 < n)   // then remove it from the other rows of block b1.
                        for (int ii = i / n * n; ii < (i + n) / n * n; ii++)
                            if (ii != i)
                                for (int jj = b1 * n; jj < (b1 + 1) * n; jj++)
                                    Lists[ii, jj] -= Digit;
                }
                for (int j = 0; j < nn; j++)
                {
                    int b1 = n;
                    for (int b = 0; b < n; b++)
                    {
                        for (int i = 0; i < n; i++)
                            if (TestLists(b * n + i, j, Digit))   // search for Digit in column j, block b
                            {
                                b1 = b1 == n || b1 == b ? b : -1; // set b1 to block b or -1, if more
                                break;                            // blocks are involved
                            }
                        if (b1 == -1)
                            break;
                    }                        // If Digit is only in column j at block b1 (no other blocks), 
                    if (b1 >= 0 && b1 < n)   // then remove it from the other columns of block b1.
                        for (int jj = j / n * n; jj < (j + n) / n * n; jj++)
                            if (jj != j)
                                for (int ii = b1 * n; ii < (b1 + 1) * n; ii++)
                                    Lists[ii, jj] -= Digit;
                }
                for (int b = 0; b < nn; b++)
                {
                    int i0 = n * (b % n), j0 = n * (b / n);
                    int i1 = nn, j1 = nn;
                    for (int i = i0; i < i0 + n; i++)
                    {
                        for (int j = j0; j < j0 + n; j++)
                            if (TestLists(i, j, Digit))            // search for Digit in block b, row i
                            {
                                i1 = i1 == nn || i1 == i ? i : -1; // set i1 to row i or -1, if more
                                break;                             // rows are involved
                            }
                        if (i1 == -1)
                            break;
                    }                        // If Digit is only in block b at row i1 (no other rows), 
                    if (i1 >= 0 && i1 < nn)  // then remove it from the other blocks of row i1.
                        for (int jj = 0; jj < nn; jj++)
                            if (jj < j0 || jj >= j0 + n)
                                Lists[i1, jj] -= Digit;
                    for (int j = j0; j < j0 + n; j++)
                    {
                        for (int i = i0; i < i0 + n; i++)
                            if (TestLists(i, j, Digit))            // search for Digit in block b, column j
                            {
                                j1 = j1 == nn || j1 == j ? j : -1; // set j1 to column j or -1, if more
                                break;                             // columns are involved
                            }
                        if (j1 == -1)
                            break;
                    }                        // If Digit is only in block b at column j1 (no other columns), 
                    if (j1 >= 0 && j1 < nn)  // then remove it from the other blocks of column j1.
                        for (int ii = 0; ii < nn; ii++)
                            if (ii < i0 || ii >= i0 + n)
                                Lists[ii, j1] -= Digit;
                }
            }
        }

        /// <summary>
        /// Tests if Digit fits into the given Sudoku cell (using SudokuForm.Lists).
        /// </summary>
        /// <param name="i">Column of cell</param>
        /// <param name="j">Row of cell</param>
        /// <param name="Digit">Digit of cell</param>
        /// <returns>true, if Digit fits into Sudoku cell</returns>
        private bool TestLists(int i, int j, int Digit)
        {
            return Digits[i, j] == 0 && Lists[i, j] * Digit;
        }
    }
}
