// $Id: ApfelDemo.cpp 277 2007-04-30 10:31:18Z olau $

#include "stdafx.h"

#define MAX_LOADSTRING 100

// global variables
HINSTANCE hInst;
TCHAR szTitle[MAX_LOADSTRING];
TCHAR szWindowClass[MAX_LOADSTRING];

// Forward declarations of functions included in this code module:
ATOM				MyRegisterClass(HINSTANCE hInstance);
BOOL				InitInstance(HINSTANCE, int);
LRESULT CALLBACK	WndProc(HWND, UINT, WPARAM, LPARAM);
INT_PTR CALLBACK	About(HWND, UINT, WPARAM, LPARAM);
BOOL                DeleteInstance();
BOOL                StartAutomaticZoomThread();
BOOL                StartTriggerThread();
BOOL                StartWorkerThreads();
void                TriggerAction(int);
void                KillThreads();
void                AbortAll();
void                PrintSystemInfo(LPSYSTEM_INFO);

// static (private) variables
static HANDLE hZoomerThread;
static THREADID zoomerThreadId;
static HANDLE hTriggerThread;
static THREADID triggerThreadId;
static TriggerParam triggerParam;
static HANDLE hWorker[MaxNumThreads];
static THREADID workerId[MaxNumThreads];
static SYSTEM_INFO systemInfo;
static int oldXPos = 0;
static int oldYPos = 0;
static int xPos = 0;
static int yPos = 0;
static bool holdingButton = false;

// global (shared) variables
int debugLevel = DEFAULTDEBUGLEVEL;
HWND hMainWindow = NULL;
HANDLE hReadyEvent[MaxNumThreads];
MandelbrotSetParam param[MaxNumThreads];
__declspec(align(16)) double threshold = DefaultThreshold;
__declspec(align(16)) double scaleFactor = DefaultScaleFactor;
__declspec(align(16)) double centerR = AutomaticCenterR;
__declspec(align(16)) double centerI = AutomaticCenterI;
int maxIterations = DefaultMaxIterations;
int numThreads = DefaultNumThreads;
bool doAbort = false;
bool doPause = false;
ParallelLog logBuf;
int zooming = NotZooming;
CRITICAL_SECTION critSecAbort;
CRITICAL_SECTION critSecZooming;
HANDLE hTriggerEvent;
double zoomDelta = 1.0;


// here we go ...
int APIENTRY _tWinMain(HINSTANCE hInstance,
                       HINSTANCE hPrevInstance,
                       LPTSTR    lpCmdLine,
                       int       nCmdShow)
{
    UNREFERENCED_PARAMETER(hPrevInstance);
    UNREFERENCED_PARAMETER(lpCmdLine);


    MSG msg;
    HACCEL hAccelTable;

    RedirectIOToConsole();

    ZeroMemory(param, MaxNumThreads * sizeof(MandelbrotSetParam));

    LoadString(hInstance, IDS_APP_TITLE, szTitle, MAX_LOADSTRING);
    LoadString(hInstance, IDC_APFELDEMO, szWindowClass, MAX_LOADSTRING);
    MyRegisterClass(hInstance);

    GetSystemInfo(&systemInfo); 
    if (debugLevel > 0)
        PrintSystemInfo(&systemInfo);

    if (!InitInstance(hInstance, nCmdShow))
    {
        return FALSE;
    }

    hAccelTable = LoadAccelerators(hInstance, MAKEINTRESOURCE(IDC_APFELDEMO));

    WaitForWorkerThreads();

    while (GetMessage(&msg, NULL, 0, 0))
    {
        if (!TranslateAccelerator(msg.hwnd, hAccelTable, &msg))
        {
            TranslateMessage(&msg);
            DispatchMessage(&msg);
        }
    }

    if (!DeleteInstance())
    {
        return FALSE;
    }

    return (int) msg.wParam;
}


ATOM MyRegisterClass(HINSTANCE hInstance)
{
    WNDCLASSEX wcex;
    wcex.cbSize         = sizeof(WNDCLASSEX);
    wcex.style			= CS_HREDRAW | CS_VREDRAW/* | CS_DBLCLKS*/;
    wcex.lpfnWndProc	= WndProc;
    wcex.cbClsExtra		= 0;
    wcex.cbWndExtra		= 0;
    wcex.hInstance		= hInstance;
    wcex.hIcon			= LoadIcon(hInstance, MAKEINTRESOURCE(IDI_APFELDEMO));
    wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
    wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW+1);
    wcex.lpszMenuName	= MAKEINTRESOURCE(IDC_APFELDEMO);
    wcex.lpszClassName	= szWindowClass;
    wcex.hIconSm		= LoadIcon(wcex.hInstance, MAKEINTRESOURCE(IDI_SMALL));
    return RegisterClassEx(&wcex);
}


BOOL InitInstance(HINSTANCE hInstance, int nCmdShow)
{
    hInst = hInstance;
    HWND hWnd = CreateWindow(szWindowClass, szTitle, WS_OVERLAPPEDWINDOW,
        CW_USEDEFAULT, 0, 600, 400, NULL, NULL, hInstance, NULL);
    if (!hWnd)
        return FALSE;

    hMainWindow = hWnd;

    InitializeCriticalSection(&critSecAbort);
    InitializeCriticalSection(&critSecZooming);

    if (!StartTriggerThread())
        return FALSE;
    
    if (!StartWorkerThreads())
        return FALSE;
    
    if (!StartAutomaticZoomThread())
        return FALSE;

    ShowWindow(hWnd, nCmdShow);
    UpdateWindow(hWnd);

    return TRUE;
}


BOOL DeleteInstance(void)
{
    WaitForMultipleObjects(numThreads, hWorker, TRUE, INFINITE);
    for (int i = 0; i < numThreads; ++i)
    {
        CloseHandle(hWorker[i]);
        CloseHandle(hReadyEvent[i]);
        CloseHandle(param[i].hTriggerEvent);
        VirtualFree(param[i].bitmap.bmBits, 0, MEM_RELEASE);
        // _aligned_free(param[i].bitmap.bmBits);
    }

    if (!doPause)
    {
        WaitForSingleObject(hZoomerThread, 0);
        CloseHandle(hZoomerThread);
    }

    WaitForSingleObject(hTriggerThread, 0);
    CloseHandle(hTriggerThread);

    return TRUE;
}


LRESULT CALLBACK WndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
    int wmId, wmEvent;
    PAINTSTRUCT ps;
    HDC hdc;

    switch (message)
    {
    case WM_COMMAND:
        wmId    = LOWORD(wParam);
        wmEvent = HIWORD(wParam);
        switch (wmId)
        {
        case IDM_ABOUT:
            DialogBox(hInst, MAKEINTRESOURCE(IDD_ABOUTBOX), hWnd, About);
            break;
        case IDM_EXIT:
            KillThreads();
            DestroyWindow(hWnd);
            break;
        case IDM_SAVEAS:
            // TODO: save Mandelbrot graphics as BMP/PNG/TIFF/...
            break;
        case ID_UP:
            if (doPause)
            {
                centerI += scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_DOWN:
            if (doPause)
            {
                centerI -= scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_LEFT:
            if (doPause)
            {
                centerR -= scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_RIGHT:
            if (doPause)
            {
                centerR += scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_UPFAST:
            if (doPause)
            {
                centerI += 10 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_DOWNFAST:
            if (doPause)
            {
                centerI -= 10 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_LEFTFAST:
            if (doPause)
            {
                centerR -= 10 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_RIGHTFAST:
            if (doPause)
            {
                centerR += 10 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_UPREALLYFAST:
            if (doPause)
            {
                centerI += 100 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_DOWNREALLYFAST:
            if (doPause)
            {
                centerI -= 100 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_LEFTREALLYFAST:
            if (doPause)
            {
                centerR -= 100 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_RIGHTREALLYFAST:
            if (doPause)
            {
                centerR += 100 * scaleFactor;
                TriggerAction(TriggerPan);
            }
            break;
        case ID_DECREASEDEBUGLEVEL:
            if (debugLevel > 0)
                --debugLevel;
            RedrawWindow(hMainWindow, NULL, NULL, RDW_INVALIDATE);
            break;
        case ID_INCREASEDEBUGLEVEL:
            if (debugLevel < MaxDebugLevel)
                ++debugLevel;
            RedrawWindow(hMainWindow, NULL, NULL, RDW_INVALIDATE);
            break;
        case ID_ZOOMIN:
            if (doPause)
            {
                scaleFactor *= AutomaticZoomInFactor;
                TriggerAction(TriggerZoom);
            }
            break;
        case ID_ZOOMOUT:
            if (doPause)
            {
                scaleFactor *= AutomaticZoomOutFactor;
                TriggerAction(TriggerZoom);
            }
            break;
        case ID_ZOOMINFAST:
            if (doPause)
            {
                scaleFactor *= pow(AutomaticZoomInFactor, 10);
                TriggerAction(TriggerZoom);
            }
            break;
        case ID_ZOOMOUTFAST:
            if (doPause)
            {
                scaleFactor *= pow(AutomaticZoomOutFactor, 10);
                TriggerAction(TriggerZoom);
            }
            break;
        case ID_ZOOMINREALLYFAST:
            if (doPause)
            {
                scaleFactor *= pow(AutomaticZoomInFactor, 20);
                TriggerAction(TriggerZoom);
            }
            break;
        case ID_ZOOMOUTREALLYFAST:
            if (doPause)
            {
                scaleFactor *= pow(AutomaticZoomOutFactor, 10);
                TriggerAction(TriggerZoom);
            }
            break;
        case ID_STARTSTOP:
            doPause = !doPause;
            if (doPause)
            {
                SuspendThread(hZoomerThread);
            }
            else
            {
                ResumeThread(hZoomerThread);
            }
            break;
        default:
            return DefWindowProc(hWnd, message, wParam, lParam);
        }
        break;
    case WM_PAINT:
        hdc = BeginPaint(hWnd, &ps);
        for (int i = 0; i < numThreads; ++i)
        {
            if (param[i].ready)
            {
                Stopwatch timer;
                int rc = SetDIBits(
                    param[i].hdc,           // handle to DC
                    param[i].hBitmap,       // handle to bitmap
                    0,                      // first scan line to set
                    param[i].height,        // number of scan lines to copy
                    param[i].bitmap.bmBits, // array for bitmap bits
                    &param[i].bi,           // bitmap data buffer
                    DIB_RGB_COLORS          // RGB or palette index
                    );
                EnterCriticalSection(&critSecZooming);
                int currentlyZooming = zooming;
                LeaveCriticalSection(&critSecZooming);
                switch (currentlyZooming)
                {
                case NotZooming:
                    BitBlt(
                        hdc,                  // handle to destination DC
                        0,                    // x-coord of destination upper-left corner
                        i * param[i].height,  // y-coord of destination upper-left corner
                        param[i].width,       // width of destination rectangle
                        param[i].height,      // height of destination rectangle
                        param[i].hdc,         // handle to source DC
                        oldXPos-xPos,         // x-coordinate of source upper-left corner
                        oldYPos-yPos,         // y-coordinate of source upper-left corner
                        SRCCOPY               // raster operation code
                        );
                    break;
                case ZoomingIn:
                case ZoomingOut: {
                    int wDest = (int) (zoomDelta * param[i].width);
                    int hDest = (int) (zoomDelta * param[i].height);
                    int xDest = (int) ((1 - zoomDelta) * param[i].width / 2);
                    int yDest = (int) ((1 - zoomDelta) * param[i].height / 2) + i * param[i].height;
                    StretchBlt(
                        hdc,             // handle to destination DC
                        xDest,           // x-coord of destination upper-left corner
                        yDest,           // y-coord of destination upper-left corner
                        wDest,           // width of destination rectangle
                        hDest,           // height of destination rectangle
                        param[i].hdc,    // handle to source DC
                        0,               // x-coord of source upper-left corner
                        0,               // y-coord of source upper-left corner
                        param[i].width,  // width of source rectangle
                        param[i].height, // height of source rectangle
                        SRCCOPY          // raster operation code
                        ); 
                    }
                    break;
                default:
                    fprintf(stderr, "Oops! Should've never got to here (%s at line %d)\n", __FILE__, __LINE__);
                    break;
                }
                timer.stop();
                if (debugLevel > 0) 
                {
                    const int BUF_SIZE = 128;
                    TCHAR msgBuf[BUF_SIZE];
                    size_t cchStringSize;
                    StringCchPrintf(msgBuf, BUF_SIZE, TEXT("%I64d iterations in %g ms = %.1lfM iters/sec\n"), param[i].iterations, 1e3 * param[i].deltaT, 1e-6 * (double) param[i].iterations / param[i].deltaT); 
                    StringCchLength(msgBuf, BUF_SIZE, &cchStringSize);
                    TextOut(hdc, 0, i * param[i].height + 0, msgBuf, (int)cchStringSize);
                    StringCchPrintf(msgBuf, BUF_SIZE, TEXT("drawn in %g ms\n"), 1e3 * timer.elapsed()); 
                    StringCchLength(msgBuf, BUF_SIZE, &cchStringSize);
                    TextOut(hdc, 0, i * param[i].height + 16, msgBuf, (int)cchStringSize);
                }
            }
        }
        EndPaint(hWnd, &ps);
        break;
    case WM_DESTROY:
        KillThreads();
        PostQuitMessage(0);
        break;
    case WM_LBUTTONDOWN:
        if (doPause)
        {
            holdingButton = true;
            oldXPos = GET_X_LPARAM(lParam); 
            oldYPos = GET_Y_LPARAM(lParam);
        }
        break;
    case WM_LBUTTONUP:
        if (doPause)
        {
            holdingButton = false;
            xPos = GET_X_LPARAM(lParam); 
            yPos = GET_Y_LPARAM(lParam);
            centerR -= scaleFactor * (xPos - oldXPos);
            centerI += scaleFactor * (yPos - oldYPos);
            if (oldXPos != xPos || oldYPos != yPos)
            {
                oldXPos = xPos;
                oldYPos = yPos;
                TriggerAction(TriggerPan);
            }
        }
        break;
    case WM_MOUSEMOVE:
        if (holdingButton)
        {
            xPos = GET_X_LPARAM(lParam);
            yPos = GET_Y_LPARAM(lParam);
            RedrawWindow(hMainWindow, NULL, NULL, RDW_INVALIDATE);
        }
        break;
    case WM_MOUSEWHEEL:
        if (doPause)
        {
            int fwKeys = GET_KEYSTATE_WPARAM(wParam);
            int zoomMultiple = ((fwKeys & MK_CONTROL) == MK_CONTROL)? 10 : 1;
            int zDelta = GET_WHEEL_DELTA_WPARAM(wParam);
            EnterCriticalSection(&critSecZooming);
            zooming = (zDelta > 0)? ZoomingIn : ZoomingOut;
            LeaveCriticalSection(&critSecZooming);
            double tempScale;
            switch (zooming)
            {
            case ZoomingIn:
                tempScale = pow(AutomaticZoomInFactor, zoomMultiple);
                scaleFactor *= tempScale;
                zoomDelta *= 1/tempScale;
                RedrawWindow(hMainWindow, NULL, NULL, RDW_INVALIDATE);
                TriggerAction(TriggerZoom);
                break;
            case ZoomingOut:
                tempScale = pow(AutomaticZoomOutFactor, zoomMultiple);
                scaleFactor *= tempScale;
                zoomDelta *= 1/tempScale;
                RedrawWindow(hMainWindow, NULL, NULL, RDW_INVALIDATE);
                TriggerAction(TriggerZoom);
                break;
            default:
                break;
            }
            
        }
        break;
    default:
        return DefWindowProc(hWnd, message, wParam, lParam);
    }
    return 0;
}


// Message handler for about box.
INT_PTR CALLBACK About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
    UNREFERENCED_PARAMETER(lParam);
    switch (message)
    {
    case WM_INITDIALOG:
        return (INT_PTR)TRUE;

    case WM_COMMAND:
        if (LOWORD(wParam) == IDOK || LOWORD(wParam) == IDCANCEL)
        {
            EndDialog(hDlg, LOWORD(wParam));
            return (INT_PTR)TRUE;
        }
        break;
    }
    return (INT_PTR)FALSE;
}


// initialize and start worker threads
BOOL StartWorkerThreads()
{
    DWORD threadAffinitymask = 0x00000001u;
    HDC hdc = GetDC(hMainWindow);
    RECT rect;
    GetClientRect(hMainWindow, &rect);
    int width = rect.right - rect.left;
    int height = rect.bottom - rect.top;

    for (int i = 0; i < numThreads; i++)
    {
        int tileHeight = height / numThreads;

        param[i].ready = false;
        param[i].restart = false;
        param[i].id = i;
        param[i].width = width;
        param[i].height = tileHeight;
        param[i].totalHeight = height;
        param[i].hdc = CreateCompatibleDC(hdc);
        param[i].hBitmap = CreateCompatibleBitmap(hdc, width, tileHeight);

        SelectObject(param[i].hdc, param[i].hBitmap);
        GetObject(param[i].hBitmap, sizeof(BITMAP), (LPVOID) &param[i].bitmap);
        DWORD bpp = (int) (param[i].bitmap.bmPlanes * param[i].bitmap.bmBitsPixel) / 8;
        // param[i].bitmap.bmBits = _aligned_malloc(bpp * width * tileHeight, CACHELINE_SIZE);
        param[i].bitmap.bmBits = VirtualAlloc(NULL, bpp * width * tileHeight, MEM_COMMIT, PAGE_READWRITE);
        param[i].bi.bmiHeader.biSize = sizeof(BITMAPINFOHEADER);
        param[i].bi.bmiHeader.biWidth = width;
        param[i].bi.bmiHeader.biHeight = tileHeight;
        param[i].bi.bmiHeader.biPlanes = param[i].bitmap.bmPlanes;
        param[i].bi.bmiHeader.biBitCount = param[i].bitmap.bmBitsPixel;
        param[i].bi.bmiHeader.biCompression = BI_RGB;

        param[i].hTriggerEvent = CreateEvent(NULL, FALSE, FALSE, NULL);
        if (param[i].hTriggerEvent == NULL) 
        {
            fprintf(stderr, "CreateEvent() error (%s in line %d): %d\n", __FILE__, __LINE__, GetLastError());
            return FALSE;
        }

        hReadyEvent[i] = CreateEvent(NULL, FALSE, FALSE, NULL);
        if (hReadyEvent[i] == NULL) 
        {
            fprintf(stderr, "CreateEvent() error (%s in line %d): %d\n", __FILE__, __LINE__, GetLastError());
            return FALSE;
        }

        hWorker[i] = (HANDLE)_beginthreadex(NULL, 0, &RenderThread, &param[i], CREATE_SUSPENDED, &workerId[i]);
        if (hWorker[i] == NULL)
        {
            fprintf(stderr, "_beginthreadex() error (%s in line: %d): %d\n", __FILE__, __LINE__, GetLastError());
            return FALSE;
        }

        if (!SetThreadPriority(hWorker[i], THREAD_PRIORITY_BELOW_NORMAL))
        {
            fprintf(stderr, "SetThreadPriority() error (%s in line: %d): 0x%x (%d)\n", __FILE__, __LINE__, GetLastError(), GetLastError());
            return FALSE;
        }

        if (debugLevel > 0)
            printf("SetThreadAffinityMask(%d, 0x%04x)\n", i, threadAffinitymask);
        if (SetThreadAffinityMask(hWorker[i], threadAffinitymask) == 0)
        {
            fprintf(stderr, "SetThreadAffinityMask() error (%s in line: %d): 0x%x (%d)\n", __FILE__, __LINE__, GetLastError(), GetLastError());
            return FALSE;
        }

        // calculate new thread affinity mask in a round-robin fashion
        threadAffinitymask <<= 1;
        if ((threadAffinitymask & (DWORD) systemInfo.dwActiveProcessorMask) == 0)
            threadAffinitymask = 0x00000001u;
        // wake up thread
        ResumeThread(hWorker[i]);
    }
    return TRUE;
}


BOOL StartAutomaticZoomThread()
{

    hZoomerThread = (HANDLE)_beginthreadex(NULL, 0, &AutomaticZoomThread, NULL, CREATE_SUSPENDED, &zoomerThreadId);
    if (hZoomerThread == NULL)
    {
        fprintf(stderr, "_beginthreadex() error (%s in line: %d): %d\n", __FILE__, __LINE__, GetLastError());
        return FALSE;
    }

    if (!SetThreadPriority(hZoomerThread, THREAD_PRIORITY_NORMAL))
    {
        fprintf(stderr, "SetThreadPriority() error (%s in line: %d): 0x%x (%d)\n", __FILE__, __LINE__, GetLastError(), GetLastError());
        return FALSE;
    }

    if (!doPause)
        ResumeThread(hZoomerThread);

    return TRUE;
}


BOOL StartTriggerThread()
{
    hTriggerEvent = CreateEvent(NULL, FALSE, FALSE, _T("TriggerEvent"));
    if (hTriggerEvent == NULL) 
    {
        fprintf(stderr, "CreateEvent() error (%s in line %d): %d\n", __FILE__, __LINE__, GetLastError());
        return FALSE;
    }

    hTriggerThread = (HANDLE)_beginthreadex(NULL, 0, &TriggerThread, &triggerParam, CREATE_SUSPENDED, &triggerThreadId);
    if (hTriggerThread == NULL)
    {
        fprintf(stderr, "_beginthreadex() error (%s in line: %d): %d\n", __FILE__, __LINE__, GetLastError());
        return FALSE;
    }

    if (!SetThreadPriority(hTriggerThread, THREAD_PRIORITY_NORMAL))
    {
        fprintf(stderr, "SetThreadPriority() error (%s in line: %d): 0x%x (%d)\n", __FILE__, __LINE__, GetLastError(), GetLastError());
        return FALSE;
    }

    ResumeThread(hTriggerThread);

    return TRUE;
}


double CalcItersPerSecond()
{
    __declspec(align(16)) double iterations = 0;
    __declspec(align(16)) double sec = 0;
    for (int i = 0; i < numThreads; ++i)
    {
        iterations += param[i].iterations;
        sec += param[i].deltaT;
    }
    return numThreads * iterations / sec;
}


void KillThreads()
{
    AbortAll();
}


void AbortAll()
{
    EnterCriticalSection(&critSecAbort);
    doAbort = true;
    LeaveCriticalSection(&critSecAbort);
}


void PrintSystemInfo(LPSYSTEM_INFO si)
{
    printf("Hardware information: \n");
    printf("  OEM ID: %u\n", si->dwOemId);
    printf("  Number of processors: %u\n", si->dwNumberOfProcessors);
    printf("  Page size: %u\n", si->dwPageSize);
    printf("  Processor type: %u\n", si->dwProcessorType);
    printf("  Processor level: %u\n", (DWORD) si->wProcessorLevel);
    printf("  Processor revision: 0x%04x\n", (DWORD) si->wProcessorRevision);
    printf("  Active processor mask: 0x%x\n", si->dwActiveProcessorMask);
}


void TriggerAction(int action)
{
    triggerParam.action = action;
    RestartWorkerThreads();
    if (!SetEvent(hTriggerEvent))
        fprintf(stderr, "[%s, line %d] SetEvent() failed in function TriggerAction()\n", __FILE__, __LINE__);
}
