// Copyright (c) 2008 Oliver Lau <ola@ctmagazin.de>
// Heise Zeitschriften Verlag, Hannover, Germany

#include <QtGui>
#include <QtGlobal>
#include <QString>
#include <QImageReader>
#include <QImageWriter>
#include <QFileInfo>
#include <QMessageBox>
#include <QThreadPool>

#include "imagescaler.h"

using namespace QtConcurrent;

ImageScaler::ImageScaler(int numThreads, QWidget *parent)
: QWidget(parent)
{
	QLabel* labelNumThreads = new QLabel(tr("Threads"));
    spinBoxNumThreads = new QSpinBox();
    spinBoxNumThreads->setRange(1, 64);
    spinBoxNumThreads->setSingleStep(1);

	QLabel* labelWidth = new QLabel(tr("Breite"));
    spinBoxWidth = new QSpinBox();
	spinBoxWidth->setMinimumWidth(50);
    spinBoxWidth->setRange(16, 512);
    spinBoxWidth->setSingleStep(1);
	spinBoxWidth->setValue(100);

	QLabel* labelHeight = new QLabel(tr("Hhe"));
    spinBoxHeight = new QSpinBox();
	spinBoxHeight->setMinimumWidth(50);
    spinBoxHeight->setRange(16, 512);
    spinBoxHeight->setSingleStep(1);
	spinBoxHeight->setValue(100);

	progressBar = new QProgressBar();

    selectButton = new QPushButton();
    selectButton->setText(tr("Bilddateien auswhlen ..."));
    selectButton->setMinimumSize(200, 24);

    startStopButton = new QPushButton();
    startStopButton->setMinimumSize(96, 24);
    startStopButton->setMaximumSize(96, 24);
    startStopButton->setEnabled(false);

    QHBoxLayout* hboxLayout1 = new QHBoxLayout();
    hboxLayout1->addWidget(selectButton);
    hboxLayout1->addWidget(startStopButton);
    hboxLayout1->addStretch();

    QVBoxLayout* vboxLayout1 = new QVBoxLayout();
    vboxLayout1->addWidget(labelNumThreads);
    vboxLayout1->addWidget(spinBoxNumThreads);

    QVBoxLayout* vboxLayout2 = new QVBoxLayout();
    vboxLayout2->addWidget(labelWidth);
    vboxLayout2->addWidget(spinBoxWidth);

    QVBoxLayout* vboxLayout3 = new QVBoxLayout();
    vboxLayout3->addWidget(labelHeight);
    vboxLayout3->addWidget(spinBoxHeight);

    QHBoxLayout* hboxLayout2 = new QHBoxLayout();
	hboxLayout2->addLayout(vboxLayout1);
	hboxLayout2->addStretch();
	hboxLayout2->addLayout(vboxLayout2);
	hboxLayout2->addLayout(vboxLayout3);

	QVBoxLayout* vboxLayout = new QVBoxLayout(this);
    vboxLayout->addLayout(hboxLayout1);
    vboxLayout->addLayout(hboxLayout2);
    vboxLayout->addWidget(progressBar);
    vboxLayout->addStretch();

    QObject::connect(selectButton, SIGNAL(clicked()), this, SLOT(open()));
    QObject::connect(startStopButton, SIGNAL(clicked()), this, SLOT(startStop()));
    QObject::connect(spinBoxNumThreads, SIGNAL(valueChanged(int)), this, SLOT(on_numThreadsChanged(int)));

#ifdef USE_THREADPOOL
	pool = new ThreadPool(numThreads, QThread::LowPriority);
    QObject::connect(pool, SIGNAL(progressRangeChanged(int, int)), progressBar, SLOT(setRange(int, int)));
    QObject::connect(pool, SIGNAL(progressValueChanged(int)), progressBar, SLOT(setValue(int)));
	QObject::connect(pool, SIGNAL(finished()), this, SLOT(on_scalingFinished()));
#else
    QObject::connect(&futureWatcher, SIGNAL(progressRangeChanged(int, int)), progressBar, SLOT(setRange(int, int)));
    QObject::connect(&futureWatcher, SIGNAL(progressValueChanged(int)), progressBar, SLOT(setValue(int)));
    QObject::connect(&futureWatcher, SIGNAL(finished()), this, SLOT(on_scalingFinished()));
#endif // USE_THREADPOOL

    spinBoxNumThreads->setValue(numThreads);
    running = false;
    decorateButtons();
}


ImageScaler::~ImageScaler()
{
#ifdef USE_THREADPOOL
	pool->waitForTerminated();
	if (pool != NULL)
	{
		delete pool;
		pool = NULL;
	}
#else
    futureWatcher.waitForFinished();
#endif
}


void ImageScaler::start(void)
{
	emit showMessage(tr("Skaliervorgang luft ..."));
	timer.start();
    running = true;
#ifdef USE_THREADPOOL
	for (QStringList::const_iterator i = files.begin(); i < files.end(); ++i)
	{
		pool->enqueue(new ImageScalingJob(*i, QSize(spinBoxWidth->value(), spinBoxHeight->value())));
	}
	pool->run();
#else
	scaler.setSize(QSize(spinBoxWidth->value(), spinBoxHeight->value()));
    futureWatcher.setFuture(QtConcurrent::map(files, scaler));
#endif // USE_THREADPOOL
    decorateButtons();
}


void ImageScaler::stop(void)
{
    qDebug("ImageScaler::stop()");
    running = false;
#ifdef USE_THREADPOOL
	pool->cancel();
    pool->waitForFinished(); // HACK
#else
    futureWatcher.cancel();
    futureWatcher.waitForFinished();
#endif // USE_THREADPOOL
	qDebug("ImageScaler::stop() finished");
    decorateButtons();
}


void ImageScaler::startStop(void)
{
    if (running)
        stop();
    else
        start();
}


void ImageScaler::decorateButtons(void)
{
    if (running)
    {
        startStopButton->setText(tr("Stop"));
        selectButton->setEnabled(false);
    }
    else
    {
        startStopButton->setText(tr("Start"));
        selectButton->setEnabled(true);
    }
}


void ImageScaler::on_numThreadsChanged(int numThreads)
{
#ifdef USE_THREADPOOL
	pool->setMaxThreadCount(numThreads);
#else
	QThreadPool::globalInstance()->setMaxThreadCount(numThreads);
#endif // USE_THREADPOOL
}


void ImageScaler::on_scalingFinished(void)
{
	qDebug("ImageScaler::on_scalingFinished()");
#ifdef USE_THREADPOOL
    if (pool->isCanceled())
#else
    if (futureWatcher.future().isCanceled())
#endif // USE_THREADPOOL
    {
        QMessageBox::information(this, tr("Skalieren abgebrochen"),
            tr("Der Skaliervorgang wurde nicht vollstndig abgeschlossen.\n"
            "Von einigen Bildern fehlen demnach die Thumbnails.\n"),
            QMessageBox::Ok);
    }
    else
    {
        emit showMessage(tr("Ausfhrungszeit: %1 s").arg((float) timer.elapsed() / 1e3));
    }
    stop();
    progressBar->reset();
}


void ImageScaler::open(void)
{
    files = QFileDialog::getOpenFileNames(this, tr("Bilddateien auswhlen ..."), ".");
    if (files.size() > 0)
    {
        startStopButton->setEnabled(true);
        emit showMessage(tr("%1 Dateien ausgewhlt").arg(files.size()));
    }
}
