// *****************************************************************
// Filename:  FaceRecognitionOrganizer.h
// Copyright: Pedram Azad, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:	  Pedram Azad
// Date:      02.05.2008
// *****************************************************************

#include "FaceRecognitionOrganizer.h"

#include "Image/ByteImage.h"
#include "Image/ImageProcessor.h"
#include "Image/PrimitivesDrawer.h"
#include "Image/PrimitivesDrawerCV.h"
#include "Math/FloatMatrix.h"
#include "ObjectFinder/HaarClassifierCV.h"
#include "ObjectFinder/ObjectFinder.h"
#include "ObjectFinder/CompactRegionFilter.h"
#include "Color/ColorParameterSet.h"
#include "Helpers/helpers.h"

#include "VideoCapture/OpenCVCapture.h"
#include "VideoCapture/BitmapSequenceCapture.h"
#include "VideoCapture/BitmapCapture.h"
#ifdef WIN32
#include "VideoCapture/VFWCapture.h"
#endif

#include "gui/QTWindow.h"
#include "gui/QTApplicationHandler.h"

#include "ViewDatabase.h"
#include "DatabaseEntry.h"

#ifdef WIN32
#include "SendKeys/SendKeys.h"
#endif

#include <qpushbutton.h>
#include <qcombobox.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qslider.h>
#include <qstring.h>
#include <qlabel.h>
#include <qfiledialog.h>
#include <qsettings.h>
#include <qlcdnumber.h>
#include <qinputdialog.h>
#include <qcolor.h>

#include <stdio.h>


#define	VIEWS_OUTPUT_BASE					"database"
#define OBJECT_DATABASE_CONFIGURATION_FILE	"face_database.txt"
#define CASCADE_PATH						"haarcascade_frontalface_alt2.xml"


#define H_OFFSET 100



CFaceRecognitionOrganizer::CFaceRecognitionOrganizer()
{
	m_pPersonDatabase = new CViewDatabase();
	
	m_pVideoCapture = 0;

	m_bRecording = false;
	m_nIndex = 0;
	m_nRecognizedCounter = 0;

	m_pWindow = 0;

	m_pImage = 0;
	m_pHSVImage = 0;
	m_pSegmentedImage = 0;
	m_pGrayscaleImage = 0;

	m_sPassword = "(^{F12})";
}


CFaceRecognitionOrganizer::~CFaceRecognitionOrganizer()
{
	delete m_pPersonDatabase;

	if (m_pImage)
	{
		delete m_pImage;
		delete m_pHSVImage;
		delete m_pSegmentedImage;
		delete m_pGrayscaleImage;
	}
}


void CFaceRecognitionOrganizer::SetSize(int width, int height, CByteImage::ImageType type)
{
	printf("Info:   Die aktuelle Kameraaufloesung ist %ix%i\n", width, height);

	// restore gray background if necessary
	if (m_pImage)
	{
		CByteImage background(m_pImage->width, m_pImage->height, CByteImage::eRGB24);

		QColor color = m_pWindow->paletteBackgroundColor();

		const int r = color.red();
		const int g = color.green();
		const int b = color.blue();

		const int nBytes = 3 * background.width * background.height;

		for (int i = 0; i < nBytes; i += 3)
		{
			background.pixels[i] = r;
			background.pixels[i + 1] = g;
			background.pixels[i + 2] = b;
		}

		m_pWindow->DrawImage(&background);
	}

	m_pWindow->setFixedSize(width + 420, MY_MAX(height, 480));

	m_pComboBoxPersons->move(width + 270, 23);

	m_pShowPersonButton->move(width + 20, 20);
	m_pDeleteButton->move(width + 150, 20);
	m_pRecordButton->move(width + 20, 60);
	m_pOpenPasswordDialogButton->move(width + 270, H_OFFSET + 15);

	m_pLineEditName->move(width + 270, 62);
	
	m_pCheckBoxUseColorSegmentation->move(width + 20, H_OFFSET);
	m_pCheckBoxShowSegmentedImage->move(width + 20, H_OFFSET + 30);
	
	m_pSliderColorH->move(width + 65, H_OFFSET + 70);
	m_pSliderColorHT->move(width + 65, H_OFFSET + 95);
	m_pSliderColorMinS->move(width + 65, H_OFFSET + 120);
	m_pSliderColorMaxS->move(width + 65, H_OFFSET + 145);
	m_pSliderColorMinV->move(width + 65, H_OFFSET + 170);
	m_pSliderColorMaxV->move(width + 65, H_OFFSET + 195);
	m_pSliderRecognitionThreshold->move(width + 150, H_OFFSET + 230);

	m_pLCDRecognitionThreshold->move(width + 320, H_OFFSET + 227);
	
	m_pLabelH->move(width + 20, H_OFFSET + 70);
	m_pLabelHTol->move(width + 20, H_OFFSET + 95);
	m_pLabelMinS->move(width + 20, H_OFFSET + 120);
	m_pLabelMaxS->move(width + 20, H_OFFSET + 145);
	m_pLabelMinV->move(width + 20, H_OFFSET + 170);
	m_pLabelMaxV->move(width + 20, H_OFFSET + 195);
	m_pLabelRecognitionThreshold->move(width + 20, H_OFFSET + 230);
	m_pLabelName->move(width + 375, 63);

	#ifdef WIN32
	m_pShowVideoSourceDialogButton->move(width + 20, 433);
	m_pShowVideoFormatDialogButton->move(width + 145, 433);
	m_pOpenCameraButton->move(width + 270, 433);
	m_pComboBoxVideoDrivers->move(width + 20, 393);
	#endif

	if (m_pImage)
	{
		delete m_pImage;
		delete m_pHSVImage;
		delete m_pSegmentedImage;
		delete m_pGrayscaleImage;
	}

	m_pImage = new CByteImage(width, height, type);
	m_pHSVImage = new CByteImage(width, height, type);
	m_pSegmentedImage = new CByteImage(width, height, CByteImage::eGrayScale);
	m_pGrayscaleImage = new CByteImage(width, height, CByteImage::eGrayScale);
}


bool CFaceRecognitionOrganizer::Run(int argc, char **args)
{
	// load cascade for face recognition
	CHaarClassifierCV faceFinder;
	if (!faceFinder.Init(CASCADE_PATH))
	{
		printf("Fehler: Konfigurationsdatei des Gesichtserkenners konnte nicht geladen werden\n");
		return false;
	}

	// init person database
	if (!m_pPersonDatabase->Init(OBJECT_DATABASE_CONFIGURATION_FILE))
		printf("Fehler: Datenbank konnte nicht geladen werden\n");
		
	// create object for color segmentation
	CObjectFinder objectFinder;
	CCompactRegionFilter compactRegionFilter;
	CColorParameterSet colorParameterSet;
	objectFinder.SetColorParameterSet(&colorParameterSet);
	objectFinder.SetRegionFilter(&compactRegionFilter);

	// create GUI
	CQTApplicationHandler applicationHandler(argc, args);
	applicationHandler.Reset();

	// create windows
	#ifdef WIN32
	CQTWindow captureWindow(1024, 768);
	captureWindow.hide();
	m_pVideoCapture = new CVFWCapture(captureWindow.winId(), 0);
	#else
	m_pVideoCapture = new COpenCVCapture(-1);
	#endif

	// initialize camera
	if (!m_pVideoCapture->OpenCamera())
	{
		printf("Fehler: Kamera konnte nicht geoeffnet werden\n");
		delete m_pVideoCapture;
		m_pVideoCapture = 0;
		return false;
	}

	// retrieve image information
	int width = m_pVideoCapture->GetWidth();
	int height = m_pVideoCapture->GetHeight();
	CByteImage::ImageType type = m_pVideoCapture->GetType();

	if (type == CByteImage::eGrayScale)
	{
		// color mode not available for grayscale cameras
		m_pCheckBoxUseColorSegmentation->setEnabled(false);
	}
	
	int i;

	// main window
	m_pWindow = new CQTWindow(width + 420, MY_MAX(height, 480));
		
	// combo boxes
	m_pComboBoxPersons = new QComboBox(m_pWindow);
	m_pComboBoxPersons->setFixedWidth(130);
	m_pComboBoxPersons->setFixedHeight(25);
	
	for (i = 0; i < m_pPersonDatabase->GetNumberOfClasses(); i++)
		m_pComboBoxPersons->insertItem(m_pPersonDatabase->GetDatabaseEntry(i)->m_sName.c_str());

	// buttons
	m_pShowPersonButton = new QPushButton(m_pWindow);
	m_pShowPersonButton->setText(QString("Aufnahmen anzeigen"));
	m_pShowPersonButton->setFixedSize(120, 32);
		
	m_pDeleteButton = new QPushButton(m_pWindow);
	m_pDeleteButton->setText(QString("Eintrag lschen"));
	m_pDeleteButton->setFixedSize(100, 32);
	
	m_pRecordButton = new QPushButton(m_pWindow);
	m_pRecordButton->setText(QString("Einlernmodus starten"));
	m_pRecordButton->setFixedSize(230, 32);
	
	m_pOpenPasswordDialogButton = new QPushButton(m_pWindow);
	m_pOpenPasswordDialogButton->setText(QString("Passwort setzen"));
	m_pOpenPasswordDialogButton->setFixedSize(130, 32);
	
	// edit fields
	m_pLineEditName = new QLineEdit(m_pWindow);
	m_pLineEditName->setText(QString("noname"));
	m_pLineEditName->setReadOnly(false);
	m_pLineEditName->setFixedWidth(100);
	m_pLineEditName->setFixedHeight(25);

	// check boxes
	m_pCheckBoxUseColorSegmentation = new QCheckBox("Farbsegmentierung verwenden", m_pWindow);
	m_pCheckBoxUseColorSegmentation->setChecked(false);
	m_pCheckBoxUseColorSegmentation->setFixedWidth(200);

	m_pCheckBoxShowSegmentedImage = new QCheckBox("Segmentiertes Bild anzeigen", m_pWindow);
	m_pCheckBoxShowSegmentedImage->setChecked(false);
	m_pCheckBoxShowSegmentedImage->setFixedWidth(200);

	// sliders
	m_pSliderColorH = new QSlider(0, 180, 1, 0, Qt::Horizontal, m_pWindow);
	m_pSliderColorH->setFixedWidth(180);
	m_pSliderColorH->setFixedHeight(20);

	m_pSliderColorHT = new QSlider(0, 180, 1, 25, Qt::Horizontal, m_pWindow);
	m_pSliderColorHT->setFixedWidth(180);
	m_pSliderColorHT->setFixedHeight(20);

	m_pSliderColorMinS = new QSlider(0, 255, 1, 20, Qt::Horizontal, m_pWindow);
	m_pSliderColorMinS->setFixedWidth(180);
	m_pSliderColorMinS->setFixedHeight(20);

	m_pSliderColorMaxS = new QSlider(0, 255, 1, 255, Qt::Horizontal, m_pWindow);
	m_pSliderColorMaxS->setFixedWidth(180);
	m_pSliderColorMaxS->setFixedHeight(20);

	m_pSliderColorMinV = new QSlider(0, 255, 1, 20, Qt::Horizontal, m_pWindow);
	m_pSliderColorMinV->setFixedWidth(180);
	m_pSliderColorMinV->setFixedHeight(20);
	
	m_pSliderColorMaxV = new QSlider(0, 255, 1, 255, Qt::Horizontal, m_pWindow);
	m_pSliderColorMaxV->setFixedWidth(180);
	m_pSliderColorMaxV->setFixedHeight(20);
	
	m_pSliderRecognitionThreshold = new QSlider(70, 100, 1, 85, Qt::Horizontal, m_pWindow);
	m_pSliderRecognitionThreshold->setFixedWidth(150);
	m_pSliderRecognitionThreshold->setFixedHeight(20);

	// lcd displays
	m_pLCDRecognitionThreshold = new QLCDNumber(4, m_pWindow);
	m_pLCDRecognitionThreshold->setFixedWidth(50);
	m_pLCDRecognitionThreshold->setFixedHeight(30);

	// labels
	m_pLabelH = new QLabel(m_pWindow);
	m_pLabelH->setText("H");
	m_pLabelH->setFixedWidth(40);
	m_pLabelH->setFixedHeight(20);

	m_pLabelHTol = new QLabel(m_pWindow);
	m_pLabelHTol->setText("H Tol");
	m_pLabelHTol->setFixedWidth(40);
	m_pLabelHTol->setFixedHeight(20);

	m_pLabelMinS = new QLabel(m_pWindow);
	m_pLabelMinS->setText("S Min");
	m_pLabelMinS->setFixedWidth(40);
	m_pLabelMinS->setFixedHeight(20);

	m_pLabelMaxS = new QLabel(m_pWindow);
	m_pLabelMaxS->setText("S Max");
	m_pLabelMaxS->setFixedWidth(40);
	m_pLabelMaxS->setFixedHeight(20);

	m_pLabelMinV = new QLabel(m_pWindow);
	m_pLabelMinV->setText("V Min");
	m_pLabelMinV->setFixedWidth(40);
	m_pLabelMinV->setFixedHeight(20);

	m_pLabelMaxV = new QLabel(m_pWindow);
	m_pLabelMaxV->setText("V Max");
	m_pLabelMaxV->setFixedWidth(40);
	m_pLabelMaxV->setFixedHeight(20);

	m_pLabelRecognitionThreshold = new QLabel(m_pWindow);
	m_pLabelRecognitionThreshold->setText("Schwellwert Korrelation");
	m_pLabelRecognitionThreshold->setFixedWidth(120);
	m_pLabelRecognitionThreshold->setFixedHeight(20);

	m_pLabelName = new QLabel(m_pWindow);
	m_pLabelName->setText("Name");
	m_pLabelName->setFixedWidth(40);
	m_pLabelName->setFixedHeight(20);

	#ifdef WIN32
	m_pShowVideoSourceDialogButton = new QPushButton(m_pWindow);
	m_pShowVideoSourceDialogButton->setText(QString("Video Quelle"));
	m_pShowVideoSourceDialogButton->setFixedSize(110, 32);
	
	m_pShowVideoFormatDialogButton = new QPushButton(m_pWindow);
	m_pShowVideoFormatDialogButton->setText(QString("Video Format"));
	m_pShowVideoFormatDialogButton->setFixedSize(110, 32);

	m_pOpenCameraButton = new QPushButton(m_pWindow);
	m_pOpenCameraButton->setText(QString("Kamera ffnen"));
	m_pOpenCameraButton->setFixedSize(110, 32);
	
	m_pComboBoxVideoDrivers = new QComboBox(m_pWindow);
	m_pComboBoxVideoDrivers->setFixedWidth(360);
	m_pComboBoxVideoDrivers->setFixedHeight(25);

	for (i = 0; i < 10; i++)
	{
		std::string sName;

		if (((CVFWCapture *) m_pVideoCapture)->GetDriverName(i, sName))
			m_pComboBoxVideoDrivers->insertItem(sName.c_str());
		else
			break;
	}

	connect(m_pShowVideoSourceDialogButton, SIGNAL(clicked()), this, SLOT(ShowVideoSourceDialog()));
	connect(m_pShowVideoFormatDialogButton, SIGNAL(clicked()), this, SLOT(ShowVideoFormatDialog()));
	connect(m_pOpenCameraButton, SIGNAL(clicked()), this, SLOT(OpenCamera()));
	#endif

	// signals / slots
	connect(m_pShowPersonButton, SIGNAL(clicked()), this, SLOT(ShowPerson()));
	connect(m_pRecordButton, SIGNAL(clicked()), this, SLOT(RecordButtonClicked()));
	connect(m_pDeleteButton, SIGNAL(clicked()), this, SLOT(Delete()));
	connect(m_pOpenPasswordDialogButton, SIGNAL(clicked()), this, SLOT(SetPassword()));

	// load settings (state of application when it was closed the last time)
	QSettings settings;
	settings.setPath("ct", "FaceRecognitionApp", QSettings::User);
	bool bRead, bRet;
	int nRet;

	bRet = settings.readBoolEntry("settings/useColorSegmentation", 0, &bRead);
	if (bRead) m_pCheckBoxUseColorSegmentation->setChecked(bRet);
	
	nRet = settings.readNumEntry("settings/h", 0, &bRead);
	if (bRead) m_pSliderColorH->setValue(nRet);

	nRet = settings.readNumEntry("settings/ht", 0, &bRead);
	if (bRead) m_pSliderColorHT->setValue(nRet);

	nRet = settings.readNumEntry("settings/mins", 0, &bRead);
	if (bRead) m_pSliderColorMinS->setValue(nRet);

	nRet = settings.readNumEntry("settings/maxs", 0, &bRead);
	if (bRead) m_pSliderColorMaxS->setValue(nRet);

	nRet = settings.readNumEntry("settings/minv", 0, &bRead);
	if (bRead) m_pSliderColorMinV->setValue(nRet);

	nRet = settings.readNumEntry("settings/maxv", 0, &bRead);
	if (bRead) m_pSliderColorMaxV->setValue(nRet);

	nRet = settings.readNumEntry("settings/threshold", 0, &bRead);
	if (bRead) m_pSliderRecognitionThreshold->setValue(nRet);

	QString sPassword = settings.readEntry("settings/password", QString::null, &bRead);
	if (bRead)
	{
		m_sPassword = "";
		m_sPassword += sPassword.latin1();
	}
	
	SetSize(width, height, type);
	m_pWindow->Show();
	
	#ifdef WIN32
	CSendKeys sendKeys;
	#endif

	int n = 0;

	while (!applicationHandler.ProcessEventsAndGetExit())
	{
		m_pVideoCapture->CaptureImage(&m_pImage);

		ImageProcessor::ConvertImage(m_pImage, m_pGrayscaleImage);

		bool bFoundRegion = false;
		MyRegion region;

		if (m_pCheckBoxUseColorSegmentation->isChecked())
		{
			m_pCheckBoxShowSegmentedImage->setEnabled(true);
			m_pSliderColorH->setEnabled(true);
			m_pSliderColorHT->setEnabled(true);
			m_pSliderColorMinS->setEnabled(true);
			m_pSliderColorMaxS->setEnabled(true);
			m_pSliderColorMinV->setEnabled(true);
			m_pSliderColorMaxV->setEnabled(true);

			ImageProcessor::CalculateHSVImage(m_pImage, m_pHSVImage);
			ImageProcessor::FilterHSV(m_pHSVImage, m_pSegmentedImage,
				m_pSliderColorH->value(), m_pSliderColorHT->value(),
				m_pSliderColorMinS->value(), m_pSliderColorMaxS->value(),
				m_pSliderColorMinV->value(), m_pSliderColorMaxV->value());

			ImageProcessor::Erode(m_pSegmentedImage, m_pSegmentedImage);
			ImageProcessor::Dilate(m_pSegmentedImage, m_pSegmentedImage, 51);
			ImageProcessor::Erode(m_pSegmentedImage, m_pSegmentedImage, 51);
			
			objectFinder.m_objectList.clear();
			
			objectFinder.FindObjectsInSegmentedImage(m_pSegmentedImage, 0, eSkin, (width * height) / 8, false);

			if (m_pCheckBoxShowSegmentedImage->isChecked())
				ImageProcessor::ConvertImage(m_pSegmentedImage, m_pImage);

			const int nRegions = (int) objectFinder.m_objectList.size();
			int max = -1;

			for (int i = 0; i < nRegions; i++)
			{
				const Object2DEntry &entry = objectFinder.m_objectList.at(i);

				if (entry.region.nPixels > max)
				{
					max = entry.region.nPixels;
					bFoundRegion = true;
					region = entry.region;
				}
			}
		}
		else
		{
			m_pCheckBoxShowSegmentedImage->setEnabled(false);
			m_pSliderColorH->setEnabled(false);
			m_pSliderColorHT->setEnabled(false);
			m_pSliderColorMinS->setEnabled(false);
			m_pSliderColorMaxS->setEnabled(false);
			m_pSliderColorMinV->setEnabled(false);
			m_pSliderColorMaxV->setEnabled(false);

			RegionList resultList;
			faceFinder.Find(m_pGrayscaleImage, resultList);

			const int nRegions = (int) resultList.size();
			int min = 2 * width * height;

			for (int i = 0; i < nRegions; i++)
			{
				const MyRegion &faceRegion = resultList.at(i);

				const int nSize = (faceRegion.max_x - faceRegion.min_x + 1) * (faceRegion.max_y - faceRegion.min_y + 1);

				if (nSize < min)
				{
					min = nSize;
					bFoundRegion = true;
					region = faceRegion;
				}
			}
		}

		if (bFoundRegion)
		{
			const int w = region.max_x - region.min_x + 1;
			const int h = region.max_y - region.min_y + 1;

			MyRegion region_small;
			region_small.min_x = region.min_x + int(0.1 * w);
			region_small.min_y = region.min_y + int(0.1 * h);
			region_small.max_x = region.max_x - int(0.1 * w);
			region_small.max_y = region.max_y - int(0.1 * h);

			PrimitivesDrawer::DrawRegion(m_pImage, region, 255, 0, 0);
			PrimitivesDrawer::DrawRegion(m_pImage, region_small, 0, 0, 255);

			if (!m_bRecording || n++ % 5 == 0)
			{
				CByteImage face_normalized(200, 200, CByteImage::eGrayScale);
				CViewDatabase::Normalize(m_pGrayscaleImage, &face_normalized, region_small);

				if (m_bRecording)
				{
					char szPath[1024];
					sprintf(szPath, "%s/%s_%.4i.bmp", VIEWS_OUTPUT_BASE, m_pLineEditName->text().latin1(), m_nIndex++);
					face_normalized.SaveToFile(szPath);
				}
				else
				{
					float correlation;
					int nClass, nView;

					if (m_pPersonDatabase->FindBestMatch(&face_normalized, correlation, nClass, nView))
					{
						std::string sName = m_pPersonDatabase->GetDatabaseEntry(nClass)->m_sName;
						printf("Name = %s -- Korrelation = %.2f\n", sName.c_str(), correlation);

						if (correlation > m_pSliderRecognitionThreshold->value() / 100.0f)
						{
							if (m_sName == sName)
								m_nRecognizedCounter++;
							else
								m_nRecognizedCounter = 0;

							m_sName = sName;

							if (m_nRecognizedCounter > 5)
							{
								#ifdef WIN32
								if (!m_pLineEditName->hasFocus())
									sendKeys.SendKeys(m_sPassword.c_str());
								
								#endif

								// recognized person
								char szText[1024];
								sprintf(szText, "recognized: %s (%i%%)", m_pPersonDatabase->GetDatabaseEntry(nClass)->m_sName.c_str(), int(100 * correlation + 0.5));
								PrimitivesDrawerCV::PutText(m_pImage, szText, 20, 30, 1, 1, 255, 255, 255, 2);
							}
						}
						else
						{
							m_nRecognizedCounter = 0;
							m_sName = "";
						}
					}
				}
			}
		}

		m_pWindow->DrawImage(m_pImage);
		m_pLCDRecognitionThreshold->display(m_pSliderRecognitionThreshold->value() / 100.0f);

		if (width != m_pVideoCapture->GetWidth() || height != m_pVideoCapture->GetHeight() ||
			type != m_pVideoCapture->GetType())
		{
			// different resolution or image type has been selected
			width = m_pVideoCapture->GetWidth();
			height = m_pVideoCapture->GetHeight();
			type = m_pVideoCapture->GetType();

			SetSize(width, height, type);
		}
	}

	// save settings
	settings.writeEntry("settings/useColorSegmentation", m_pCheckBoxUseColorSegmentation->isChecked());
	settings.writeEntry("settings/h", m_pSliderColorH->value());
	settings.writeEntry("settings/ht", m_pSliderColorHT->value());
	settings.writeEntry("settings/mins", m_pSliderColorMinS->value());
	settings.writeEntry("settings/maxs", m_pSliderColorMaxS->value());
	settings.writeEntry("settings/minv", m_pSliderColorMinV->value());
	settings.writeEntry("settings/maxv", m_pSliderColorMaxV->value());
	settings.writeEntry("settings/threshold", m_pSliderRecognitionThreshold->value());
	settings.writeEntry("settings/password", m_sPassword.c_str());
	
	delete m_pWindow;

	#ifndef WIN32
	// video for windows does not close camera properly
	delete m_pVideoCapture;
	#endif

	return true;
}

void CFaceRecognitionOrganizer::ShowPerson()
{
	ShowViews(m_pPersonDatabase, m_pComboBoxPersons->currentItem());
}

void CFaceRecognitionOrganizer::Delete()
{
	const int nCurrentItem = m_pComboBoxPersons->currentItem();
	
	if (nCurrentItem < 0 || nCurrentItem >= m_pPersonDatabase->GetNumberOfClasses())
		return;
	
	std::string sName = m_pPersonDatabase->GetDatabaseEntry(nCurrentItem)->m_sName.c_str();

	char szCommand[1024];

	#ifdef WIN32
	sprintf(szCommand, "del %s\\%s_*.bmp", VIEWS_OUTPUT_BASE, sName.c_str());
	#else
	sprintf(szCommand, "rm %s/%s_*.bmp", VIEWS_OUTPUT_BASE, sName.c_str());
	#endif

	system(szCommand);

	if (m_pPersonDatabase->RemoveEntry(sName.c_str()))
		printf("Info:   Eintrag mit Name '%s' wurde geloescht\n", sName.c_str());
	else
		printf("Fehler: Eintrag mit Name '%s' konnte nicht geloescht werden\n", sName.c_str());

	m_pComboBoxPersons->clear();

	for (int i = 0; i < m_pPersonDatabase->GetNumberOfClasses(); i++)
		m_pComboBoxPersons->insertItem(m_pPersonDatabase->GetDatabaseEntry(i)->m_sName.c_str());
}

void CFaceRecognitionOrganizer::RecordButtonClicked()
{
	m_bRecording = !m_bRecording;

	if (m_bRecording)
	{
		char szPath[1024];
		sprintf(szPath, "%s/%s_%.4i.bmp", VIEWS_OUTPUT_BASE, m_pLineEditName->text().latin1(), 0);

		int width, height;
		m_nIndex = CDatabaseEntry::GetNumberOfSamples(szPath, width, height);

		m_pRecordButton->setText("Einlernmodus beenden");
	}
	else
	{
		char szPath[1024];
		sprintf(szPath, "%s/%s_%.4i.bmp", VIEWS_OUTPUT_BASE, m_pLineEditName->text().latin1(), 0);

		if (m_pPersonDatabase->AddEntry(m_pLineEditName->text().latin1(), szPath) == 1)
		{
			m_pComboBoxPersons->clear();

			for (int i = 0; i < m_pPersonDatabase->GetNumberOfClasses(); i++)
				m_pComboBoxPersons->insertItem(m_pPersonDatabase->GetDatabaseEntry(i)->m_sName.c_str());
		}

		m_pRecordButton->setText("Einlernmodus starten");
	}
}

void CFaceRecognitionOrganizer::ShowViews(CViewDatabase *pDatabase, int nCurrentItem)
{
	if (nCurrentItem < 0 || nCurrentItem >= pDatabase->GetNumberOfClasses())
		return;
	
	const char *pTrainingDataFileName = pDatabase->GetDatabaseEntry(nCurrentItem)->m_sPath.c_str();
	
	// count number of rows
	CBitmapSequenceCapture capture(pTrainingDataFileName);
	if (!capture.OpenCamera())
	{
		printf("Fehler: Ansichten in '%s' konnten nicht geladen werden\n", pTrainingDataFileName);
		return;
	}

	const int image_width = 100;
	const int image_height = 100;
	
	CByteImage image(capture.GetWidth(), capture.GetHeight(), CByteImage::eGrayScale);
	CByteImage small_image(image_width, image_height, CByteImage::eGrayScale);
	CByteImage *pImage = &image;

	const int nRows = pDatabase->GetDatabaseEntry(nCurrentItem)->GetNumberOfSamples();
	const int nSamplesPerRow = 8;
	const int nMaxRows = 6;
	const int nViews = nRows / nMaxRows + (nRows % nMaxRows != 0);
	
	CQTWindow *pWindow = new CQTWindow(nSamplesPerRow * image_width, (nRows % nSamplesPerRow) == 0 ? (nRows / nSamplesPerRow) * image_height : ((nRows / nSamplesPerRow) + 1) * image_height);
	pWindow->Show();
	
	for (int i = 0; i < nRows; i++)
	{
		if (!capture.CaptureImage(&pImage))
			break;

		ImageProcessor::Resize(&image, &small_image);
		
		const int x = (i % nSamplesPerRow) * image_width;
		const int y = (i / nSamplesPerRow) * image_height;
		
		pWindow->DrawImage(&small_image, x, y);
	}
}

void CFaceRecognitionOrganizer::SetPassword()
{
	bool ok = false;

	QString sPassword = "";
	sPassword += m_sPassword.c_str();

	QString text = QInputDialog::getText(
		"FaceRecognitionApp",
		"Bitte geben Sie Ihr Passwort ein:",
		QLineEdit::Normal, sPassword, &ok, m_pWindow);

	if (ok && !text.isEmpty())
	{
		m_sPassword = "";
		m_sPassword += text.latin1();
	}
}


void CFaceRecognitionOrganizer::ShowVideoSourceDialog()
{
#ifdef WIN32
	Start();
	((CVFWCapture *) m_pVideoCapture)->ShowVideoSourceDialog();
	Stop();
#endif
}

void CFaceRecognitionOrganizer::ShowVideoFormatDialog()
{
#ifdef WIN32
	((CVFWCapture *) m_pVideoCapture)->ShowVideoFormatDialog();
#endif
}

void CFaceRecognitionOrganizer::OpenCamera()
{
#ifdef WIN32
	((CVFWCapture *) m_pVideoCapture)->SetDriverIndex(m_pComboBoxPersons->currentItem());
	m_pVideoCapture->OpenCamera();
#endif
}

int CFaceRecognitionOrganizer::ThreadMethod()
{
	#ifdef WIN32
	// for refreshing camera image while setting brightness etc. in video source dialog
	while (!GetExit())
	{
		m_pVideoCapture->CaptureImage(&m_pImage);
		m_pWindow->DrawImage(m_pImage);
	}
	#endif

	return 0;
}
