// *****************************************************************
// Filename:  ViewDatabase.cpp
// Copyright: Pedram Azad, Chair Prof. Dillmann (IAIM),
//            Institute for Computer Science and Engineering (CSE),
//            University of Karlsruhe. All rights reserved.
// Author:	  Pedram Azad
// Date:      02.05.2008
// *****************************************************************



// *****************************************************************
// includes
// *****************************************************************

#include "ViewDatabase.h"
#include "DatabaseEntry.h"

#include "DataStructures/DynamicArray.h"
#include "Image/ByteImage.h"
#include "Image/ImageProcessor.h"
#include "VideoCapture/BitmapSequenceCapture.h"
#include "Math/FloatMatrix.h"
#include "Helpers/helpers.h"
#include "Structs/Structs.h"

#include <stdio.h>
#include <math.h>
#include <vector>




CViewDatabase::CViewDatabase()
{
	m_nImageWidth = 0;
	m_nImageHeight = 0;
	
	m_pDatabaseEntries = new CDynamicArray(100);
}

CViewDatabase::~CViewDatabase()
{
	if (m_sConfigurationFilename.length() > 0)
		WriteConfigurationFile(m_sConfigurationFilename.c_str());
	
	DeleteDatabase();
}

int CViewDatabase::GetNumberOfClasses()
{
	return m_pDatabaseEntries->GetSize();
}

void CViewDatabase::DeleteDatabase()
{
	m_pDatabaseEntries->Clear();
	
	m_nImageWidth = 0;
	m_nImageHeight = 0;
}

const CDatabaseEntry* CViewDatabase::GetDatabaseEntry(int nIndex)
{
	return (CDatabaseEntry *) m_pDatabaseEntries->GetElement(nIndex);
}


int CViewDatabase::AddEntry(const char *pName, const char *pImagePath)
{
	int width, height;
	const int nSamples = CDatabaseEntry::GetNumberOfSamples(pImagePath, width, height);

	if (nSamples <= 0)
	{
		printf("Fehler: Keine Bilder gefunden\n");
		return -1;
	}

	if (m_nImageWidth == 0 && m_nImageHeight == 0)
	{
		// this is the first database entry
		m_nImageWidth = width;
		m_nImageHeight = height;
	}

	if (m_nImageWidth != width || m_nImageHeight != height)
	{
		// size mismatch
		printf("Fehler: Groessen stimmen nicht ueberein\n");
		return -2;
	}

	const int nClasses = m_pDatabaseEntries->GetSize();

	for (int i = 0; i < nClasses; i++)
	{
		CDatabaseEntry *pEntry = (CDatabaseEntry *) m_pDatabaseEntries->GetElement(i);
		
		if (pEntry->m_sName == pName)
		{
			pEntry->m_sPath = "";
			pEntry->m_sPath += pImagePath;

			pEntry->ReadImages();

			return 0;
		}
	}

	CDatabaseEntry *pEntry = new CDatabaseEntry(pName, pImagePath);
	pEntry->ReadImages();

	m_pDatabaseEntries->AddElement(pEntry);

	return 1;
}

bool CViewDatabase::RemoveEntry(const char *pName)
{
	int nClasses = m_pDatabaseEntries->GetSize();
	bool bDeleted = true;
	
	for (int i = 0; i < nClasses; i++)
	{
		const CDatabaseEntry *pEntry = (CDatabaseEntry *) m_pDatabaseEntries->GetElement(i);
				
		if (strcmp(pEntry->m_sName.c_str(), pName) == 0)
		{
			m_pDatabaseEntries->DeleteElement(i);
			bDeleted = true;
			nClasses--;
		}
	}

	return bDeleted;
}

bool CViewDatabase::ReadConfigurationFile(const char *pConfigurationFilename)
{
	DeleteDatabase();
	
	printf("\nInfo:   Konfigurationsdatei '%s' wird eingelesen\n", pConfigurationFilename);
	
	FILE *f = fopen(pConfigurationFilename, "r");
	if (!f)
	{
		printf("Fehler: Datei '%s' konnte nicht geoeffnet werden\n", pConfigurationFilename);
		return false;
	}
	
	char szTemp1[1024], szTemp2[1024], szTemp3[1024];
	std::vector<std::string> class_name_list;
	std::vector<std::string> class_path_list;
	
	while (fscanf(f, "%s%s%s", szTemp1, szTemp2, szTemp3) != EOF)
	{
		if (strcmp(szTemp2, "=") != 0)
		{
			printf("Fehler: Konfigurationsdate ist ungueltig (1)\n");
			fclose(f);
			return false;
		}
		
		if (strcmp(szTemp1, "class_entry") == 0)
		{
			if (fscanf(f, "%s", szTemp1) == EOF)
			{
				printf("Fehler: Konfigurationsdate ist ungueltig (2)\n");
				fclose(f);
				return false;
			}
			
			class_name_list.push_back(std::string(szTemp3));
			class_path_list.push_back(std::string(szTemp1));
		}
		else
		{
			printf("Fehler: Konfigurationsdate ist ungueltig (3)\n");
			fclose(f);
			return false;
		}
	}
	
	fclose(f);

	const int nClasses = class_name_list.size();
	
	if (nClasses <= 0)
	{
		printf("Info:   Konfigurationsdatei enthaelt keine Personen\n");
		return true;
	}

	for (int i = 0; i < nClasses; i++)
	{
		CDatabaseEntry *pEntry = new CDatabaseEntry(class_name_list.at(i).c_str(), class_path_list.at(i).c_str());

		if (pEntry->ReadImages())
		{
			const int width = pEntry->GetImageWidth();
			const int height = pEntry->GetImageHeight();

			if (m_nImageWidth != 0 && width != m_nImageWidth || m_nImageHeight != 0 && height != m_nImageHeight)
				delete pEntry;
			else
			{
				m_nImageWidth = width;
				m_nImageHeight = height;

				m_pDatabaseEntries->AddElement(pEntry);
			}
		}
		else
		{
			delete pEntry;
		}
	}
	
	return true;
}

bool CViewDatabase::WriteConfigurationFile(const char *pConfigurationFilename)
{
	FILE *f = fopen(pConfigurationFilename, "w");
	if (!f)
	{
		printf("error: could not open configuration file '%s' for writing\n", pConfigurationFilename);
		return false;
	}
	
	const int nClasses = m_pDatabaseEntries->GetSize();
	
	for (int i = 0; i < nClasses; i++)
	{
		const CDatabaseEntry *pEntry = (CDatabaseEntry *) m_pDatabaseEntries->GetElement(i);
		fprintf(f, "\nclass_entry =\n");
		fprintf(f, "\t%s\n", pEntry->m_sName.c_str());
		fprintf(f, "\t%s\n", pEntry->m_sPath.c_str());
	}
	
	fclose(f);
	
	return true;
}

bool CViewDatabase::Init(const char *pConfigurationFilename)
{
	const bool bRet = ReadConfigurationFile(pConfigurationFilename);

	m_sConfigurationFilename = "";
	m_sConfigurationFilename += pConfigurationFilename;

	return bRet;
}

bool CViewDatabase::FindBestMatch(const CByteImage *pInputImage, float &correlation, int &nResultClass, int &nResultView)
{
	if (m_pDatabaseEntries->GetSize() <= 0)
	{
		//printf("error: database is empty\n");
		return false;
	}

	if (pInputImage->width != m_nImageWidth || pInputImage->height != m_nImageHeight)
	{
		printf("error: size mismatch\n");
		return false;
	}

	const int nInputDimension = m_nImageWidth * m_nImageHeight;

	CFloatMatrix inputData(nInputDimension, 1);
	float *data = inputData.data;
	int i;
	
	for (i = 0; i < nInputDimension; i++)
		data[i] = pInputImage->pixels[i];
		 
	CDatabaseEntry::NormalizeIntensity(data, nInputDimension);

	const int nClasses = m_pDatabaseEntries->GetSize();

	float max = 0;

	for (i = 0; i < nClasses; i++)
	{
		const CDatabaseEntry *pEntry = (const CDatabaseEntry* ) m_pDatabaseEntries->GetElement(i);

		const int nSamples = pEntry->GetNumberOfSamples();

		for (int j = 0; j < nSamples; j++)
		{
			const float *data2 = pEntry->GetImageData(j);

			float sum = 0.0f;

			for (int k = 0; k < nInputDimension; k++)
				sum += data[k] * data2[k];

			if (sum > max)
			{
				nResultClass = i;
				nResultView = j;
				max = sum;
			}
		}
	}

	correlation = max;

	return correlation != 0;
}



// *****************************************************************
// static methods
// *****************************************************************

void CViewDatabase::Normalize(const CByteImage *pInputImage, CByteImage *pOutputImage, const MyRegion &region)
{
	if (pOutputImage->width != pOutputImage->height || pInputImage->type != CByteImage::eGrayScale || pOutputImage->type != CByteImage::eGrayScale)
		return;

	const int width = region.max_x - region.min_x + 1;
	const int height = region.max_y - region.min_y + 1;
	
	CByteImage copy_image(width, height, CByteImage::eGrayScale);
	ImageProcessor::CopyImage(pInputImage, &copy_image, &region);
	ImageProcessor::Resize(&copy_image, pOutputImage);
}
