// $Id$
// Copyright (c) 2008 Oliver Lau <oliver@ersatzworld.net>
// Alle Rechte vorbehalten.


#ifndef __ABSTRACTWAYPOINTLIST_H_
#define __ABSTRACTWAYPOINTLIST_H_

#include <vector>
#include <string>

#include "Value.h"
#include "Polygon.h"

namespace GPS {

    template<class PointType>
    class AbstractWaypointList {
    protected:
        std::string _Name;
        std::string _Comment;
        std::string _Description;
        std::string _Source;
        std::string _Link;
        std::string _Type;

        /// z.B. Nummer der Route
        UnsignedIntValue _Number; 

        /// Chronologisch sortierte Liste der Trackpunkte.
        std::vector<PointType*> samples;

    public:
        /// Konstruktor.
        AbstractWaypointList(void) { /* ... */ }

        /// Destruktor.
        virtual ~AbstractWaypointList(void) { /* ... */ }

        /// Kopieren der Membervariablen aus einem anderen AbstractWaypointList-Objekt
        inline void copyMembers(const AbstractWaypointList& o)
        {
            _Name = o._Name;
            _Comment = o._Comment;
            _Description = o._Description;
            _Source = o._Source;
            _Link = o._Link;
            _Type = o._Type;
        }

        /// Entfernung zum Start in Metern zurckgeben.
        /// @return Entfernung in Metern
        virtual double distance(void) const = 0;

        /// Ermitteln, ob die Liste Punkte enthlt.
        /// @return true, wenn die Liste leer ist.
        inline bool isEmpty(void) const
        {
            return samples.size() == 0;
        }

        /// Letzten Punkt aus der Liste entfernen.
        inline void pop(void)
        {
            if (!isEmpty())
                samples.pop_back();
        }

        /// Routen-Daten lschen.
        inline void clear(void)
        {
            samples.clear();
        }

        /// Einen Punkt an die Liste anhngen.
        /// @param pt Anzuhngender Punkt. NULL-Zeiger werden ignoriert.
        inline void append(PointType* pt)
        {
            if (pt != NULL)
                samples.push_back(pt);
        }


        /// Einen Trackpunkt aus dem Track entfernen.
        /// @param trkpt Zu entfernender Trackpunkt.
        inline void erase(typename std::vector<PointType*>::iterator trkpt)
        {
            samples.erase(trkpt);
        }


        /// Einen Punkt einfgen.
        /// @param i Stelle, an der der Punkt in die Liste eingefgt werden soll.
        /// @param t Einzufgender Trackpunkt.
        void insert(typename std::vector<PointType*>::iterator i, PointType* t)
        {
            samples.insert(i, t);
        }


        /// Punkt-Liste zurckgeben.
        /// @return Punkt-Liste
        /// @see samples
        inline std::vector<PointType*>& points(void)
        {
            return samples;
        }


        /// Durch den Punkte eingeschlossene Flche in Quadratmetern zurckgeben.
        /// @return Flche in Quadratmetern
        double area(void) const
        {
            GPS::Polygon p;
            for (typename std::vector<PointType*>::const_iterator i = samples.begin(); i != samples.end(); ++i)
                p << (*i)->toUTM();
            p.close();
            return p.area();
        }

        /// Namen zurckgeben.
        inline const std::string& name(void) const { return _Name; }

        /// Kommentar zurckgeben.
        inline const std::string& comment(void) const { return _Comment; }

        /// Beschreibung zurckgeben.
        inline const std::string& description(void) const { return _Description; }

        /// Quelle zurckgeben.
        inline const std::string& source(void) const { return _Source; }

        /// Link zurckgeben.
        inline const std::string& link(void) const { return _Link; }

        /// Typ zurckgeben.
        inline const std::string& type(void) const { return _Type; }

        /// Nummer zurckgeben.
        inline const IntValue& number(void) const { return _Number; }

        /// Namen setzen.
        inline void setName(std::string name) { _Name = name; }

        /// Kommentar setzen.
        inline void setComment(std::string comment) { _Comment = comment; }

        /// Beschreibung setzen.
        inline void setDescription(std::string description) { _Description = description; }

        /// Quelle setzen.
        inline void setSource(std::string source) { _Source = source; }

        /// Link setzen.
        inline void setLink(std::string link) { _Link = link; }

        /// Typ setzen.
        inline void setType(std::string type) { _Type = type; }

        /// Nummer setzen.
        inline void setNumber(int number) { _Number = number; }

    };

};

#endif // __ABSTRACTWAYPOINTLIST_H_


// $Id$
// Copyright (c) 2008 Oliver Lau <oliver@ersatzworld.net>
// Alle Rechte vorbehalten.

#ifndef __ELLIPSOID_H_
#define __ELLIPSOID_H_

#include <cassert>

#include <string>
#include <map>

#include "helper.h"


namespace GPS {

    class Ellipsoid;

    extern std::map<std::string, Ellipsoid> ellipsoid;

    /// Erdumfang in Metern
    const double RHO = 6378137.0;

    ///
    const double PHI = RHO * PI / 180.0;

    ///
    const double PI360th = PI / 360.0;


    /// Die Klasse Ellipsoid beschreibt einen Ellipsoid ber seine Radien.
    /// @see http://de.wikipedia.org/wiki/Erdellipsoid
    class Ellipsoid {
    private:
        /// Groe Achse in Metern
        double _Radius;

        /// Kleine Achse in Metern
        double _Radius2;

        /// Exzentrizitt (Kehrwert der Abflachung): a / (a - b)
        double _Eccentricity;

        /// (a^2 - b^2) / a^2
        double _EccentricitySquared;

        /// (a^2 - b^2) / b^2
        double _EccentricitySquared2;

    public:
        /// Konstruktor.
        explicit Ellipsoid()
        { /* ... */ };

        /// Copy-Konstruktor.
        Ellipsoid(const Ellipsoid& o);

        /// Konstruktor.
        /// @param a Groe Achse in Metern
        /// @param b Kleine Achse in Metern
        Ellipsoid(double a, double b);

        /// Konstruktor.
        /// @param name Name des Ellipsoid
        Ellipsoid(const std::string& name);

        /// Einen Ellipsoid ber seinen Namen auswhlen.
        /// Mgliche Angaben sind:
        /// - "AIRY"
        /// - "AIRY-MODIFIED"
        /// - "AUSTRALIAN"
        /// - "BESSEL-1841"
        /// - "CLARKE-1880"
        /// - "EVEREST-1830"
        /// - "EVEREST-MODIFIED"
        /// - "FISHER-1960"
        /// - "FISHER-1968"
        /// - "GRS80"
        /// - "HOUGH-1956"
        /// - "HAYFORD"
        /// - "IAU76"
        /// - "KRASSOVSKY-1938"
        /// - "NAD27"
        /// - "NWL-9D"
        /// - "SOUTHAMERICAN-1969"
        /// - "SOVIET-1985"
        /// - "WGS72"
        /// - "WGS84"
        void choose(const std::string& name);

        inline double radius(void) const { return _Radius; }
        inline double a(void) const { return _Radius; }
        inline double b(void) const { return _Radius2; }
        inline double eccentricity(void) const { return _Eccentricity; }
        inline double eccentricitySquared(void) const { return _EccentricitySquared; }
        inline double eccentricitySquared2(void) const { return _EccentricitySquared2; }
    };

};

#endif // __ELLIPSOID_H_


// $Id$
// Copyright (c) 2008 Oliver Lau <oliver@ersatzworld.net>
// Alle Rechte vorbehalten.

#ifndef __DURATION_H_
#define __DURATION_H_

#include "Timestamp.h"

#ifdef WIN32
#include <Windows.h>
#endif

namespace GPS {

    class Timestamp;

    /// Dauer einer Handlung.
    /// Auflsung der Zeitmessung in Millisekunden.
    class Duration {

    private:
        /// Zeitstempel in Millisekunden
        timestamp_t ms;

        /// Zeitstempel des Startzeitpunkts.
#ifdef WIN32
        LARGE_INTEGER t0;
        LARGE_INTEGER freq;
#else
        Timestamp t0;
#endif

    public:
        /// Konstruktor.
        Duration(void) : ms(0) { }

        /// Konstruktor.
        /// @param _MS Millisekunden
        Duration(timestamp_t _MS) : ms(_MS) { };

        /// Konstruktor.
        /// @param str Dauer als String. 
        /// Erlaubte Formate:
        ///        h:mm:ss
        ///        z.B. 6:00:48
        Duration(const char* str);

        /// Konstruktor.
        /// Dieser Konstruktor erfllt dieselbe Aufgabe wie Duration(const char*),
        /// @param str Dauer als String. 
        /// @see Duration(const char* str)
        Duration(const std::string& str)
        {
            *this = str.c_str();
        }

        /// Konstruktor.
        Duration(int days, int hours, int minutes, int seconds, int milliseconds = 0);

        /// Konstruktor.
        Duration(int hours, int minutes, int seconds);

        /// Starten der Zeitmessung.
        /// @see stop
        void start(void);

        /// Stoppen der Zeitmessung.    
        /// @see start
        void stop(void);

        /// Zeitdauern addieren.
        /// @param ts Zeitstempel, der addiert werden soll.
        Duration& operator+=(const Duration& ts);

        /// Zeitdauern addieren.
        /// @param ms Millisekunden, die addiert werden sollen.
        Duration& operator+=(timestamp_t ms);

        /// Zeitdauern subtrahieren.
        /// @param ts Zeitdauer, die abgezogen werden soll.
        Duration& operator-=(const Duration& ts);

        /// Zeitdauern subtrahieren.
        /// @param ms Millisekunden, die abgezogen werden sollen.
        Duration& operator-=(timestamp_t ms);

        /// Zeitdauern auf Gleichheit prfen.
        /// @return true, wenn Zeitdauern identisch sind.
        friend bool operator==(const Duration&, const Duration&);

        /// Zeitdauern auf Ungleichheit prfen.
        /// @return true, wenn Zeitdauern voneinander abweichen.
        friend bool operator!=(const Duration&, const Duration&);

        /// Zeitdauern vergleichen.
        /// @return true, wenn die zweite Zeitdauer kleiner oder gleich der ersten ist.
        friend bool operator>=(const Duration&, const Duration&);

        /// Zeitdauern vergleichen.
        /// @return true, wenn die zweite Zeitdauer grer als die erste ist.
        friend bool operator<=(const Duration&, const Duration&);

        /// Zeitdauern vergleichen.
        /// @return true, wenn die zweite Zeitdauer kleiner als die erste ist.
        friend bool operator>(const Duration&, const Duration&);

        /// Zeitstempel vergleichen.
        /// @return true, wenn die erste Zeitdauer kleiner als die zweite ist.
        friend bool operator<(const Duration&, const Duration&);

        /// Differenz zweier Zeitdauern berechnen.
        /// @return Differenz
        friend const Duration operator-(const Duration&, const Duration&);

        /// Differenz zweier Zeitdauern berechnen.
        /// @return Differenz
        friend const Duration operator-(const Duration&, timestamp_t);

        /// Summe zweier Zeitdauern berechnen.
        /// @return Summe
        friend const Duration operator+(const Duration&, const Duration&);

        /// Summe zweier Zeitdauern berechnen.
        /// @return Summe
        friend const Duration operator+(const Duration&, timestamp_t);

        /// Zeitdauer (in Sekunden) in Stream ausgeben.
        friend std::ostream& operator<<(std::ostream&, const Duration&);

        /// true zurckgeben, wenn Zeitdauer gleich 0
        inline bool isNull(void) const { return ms == 0; }

        /// Zeitdauer in Sekunden zurckgeben.
        /// @return Wert der Zeitdauer in Sekunden
        inline time_t seconds(void) const { return ms / 1000ULL; }

        /// Zeitdauer in Millisekunden zurckgeben.
        /// @return Wert der Zeitdauer in Millisekunden
        inline timestamp_t milliseconds(void) const { return ms; }

        /// Zeitdauer vis strftime() als String formatieren und zurckgeben.
        /// Wenn kein Format-String angegeben wird, verwendet toString() "%H:%M:%S".
        /// @param fmt Format-String
        /// @return String-Reprsentation der Zeitdauer.
        std::string toString(const char* fmt = NULL) const;

        /// Datum und Uhrzeit in Zeitstempel (Millisekunden) wandeln.
        /// @param days Tage
        /// @param hours Stunde
        /// @param minutes Minuten
        /// @param seconds Sekunden
        /// @param milliseconds Millisekunden
        /// @return Zeitdauer (Millisekunden)
        static timestamp_t toMs(int days, int hours, int minutes, int seconds, int milliseconds = 0);
    };

};

#endif // __DURATION_H_

// $Id$
// Copyright (c) 2008 Oliver Lau <oliver@ersatzworld.net>
// Alle Rechte vorbehalten.

#ifndef __POINT_H_
#define __POINT_H_

#include <string>

namespace GPS {

    class Vector;
    class Line;

    /// Die Klasse Point reprsentiert einen Punkt in einem dreidimensionalen
    /// kartesischen Koordinatensystem.
    class Point {
    protected:
        /// x-Wert
        double _X;

        /// y-Wert
        double _Y;

        /// z-Wert
        double _Z;

    public:
        /// Konstruktor.
        Point(void) : _X(0), _Y(0), _Z(0)
        { /* ... */ }

        /// Konstruktor.
        /// @param x
        /// @param y
        /// @param z
        Point(double x, double y, double z = 0.0) : _X(x), _Y(y), _Z(z)
        { /* ... */ }

        /// Destruktor.
        virtual ~Point() { /* ... */ }

        /// Einen Punkt von einem anderen subtrahieren.
        /// Das Ergebnis der Subtraktion ist ein Vektor.
        /// @param p1 Punkt, von dem p2 abgezogen werden soll
        /// @param p2 Punkt, der von p1 abgezogen werden soll
        /// @return Vektor, der die Differenz zwischen den beiden Punkten reprsentiert.
        friend Vector operator-(const Point& p1, const Point& p2);

        /// Einen Vektor von einem Punkt subtrahieren.
        /// Das Ergebnis der Subtraktion ist ein Punkt.
        /// @param p Punkt, von dem abgezogen werden soll.
        /// @param v Vektor, der abgezogen werden soll
        /// @return Punkt, der die Differenz reprsentiert.
        friend Point operator-(const Point& p, const Vector& v);

        /// Einen Vektor auf einen Punkt addieren.
        /// Das Ergebnis der Addition ist ein Punkt.
        /// @param p zu addierender Punkt
        /// @param v zu addierender Vektor
        /// @return Punkt, der die Summe reprsentiert.
        friend Point operator+(const Point& p, const Vector& v);

        /// Zwei Punkte addieren.
        /// Das Ergebnis der Addition ist ein Punkt.
        /// @param p1 zu addierender Punkt
        /// @param p2 zu addierender Punkt
        /// @return Punkt, der die Summe reprsentiert.
        friend Point operator+(const Point& p1, const Point& p2);

        /// Einen Punkt mit einem Skalar multiplizieren.
        /// Das Produkt ist ein Punkt.
        /// @param p Punkt
        /// @param c Skalar
        /// @return Punkt, der die Summe reprsentiert.
        friend Point operator*(const Point& p, double c);

        /// Einen Punkt mit einem Skalar multiplizieren.
        /// Das Produkt ist ein Punkt.
        /// @param c Skalar
        /// @param p Punkt
        /// @return Punkt, der das Produkt reprsentiert.
        friend Point operator*(double c, const Point& p);

        /// Einen Punkt durch einen Skalar teilen.
        /// Das Produkt ist ein Punkt.
        /// @param p Punkt
        /// @param c Skalar
        /// @return Quotient aus Punkt und Skalar.
        friend Point operator/(const Point& p, double c);

        /// Zwei Punkte miteinander vergleichen.
        /// @param p1 zu vergleichender Punkt
        /// @param p2 zu vergleichender Punkt
        /// @return true, wenn Punkte identisch sind, sonst false
        friend bool operator==(const Point& p1, const Point& p2);

        /// Zwei Punkte miteinander vergleichen.
        /// @param p1 zu vergleichender Punkt
        /// @param p2 zu vergleichender Punkt
        /// @return true, wenn Punkte sich unterscheiden, sonst false
        friend bool operator!=(const Point& p1, const Point& p2);

        /// Einen Vektor von einem Punkt abziehen.
        /// Das Ergebnis der Subtraktion ist ein Punkt.
        /// @param v zu subtrahierender Vektor
        /// @return Punkt, der die Differenz reprsentiert
        Point& operator-=(const Vector& v);

        /// Einen Vektor auf einen Punkt addieren.
        /// Das Ergebnis der Addition ist ein Punkt.
        /// @param v zu addierender Vektor
        /// @return Punkt, auf den die Spitze des Vektorpfeils zeigt.
        Point& operator+=(const Vector& v);

        /// Einen Punkt mit einem Skalar multiplizieren.
        /// Das Produkt ist ein Punkt.
        /// @param c Skalar
        /// @return Punkt, der die Summe reprsentiert.
        Point& operator*=(double c);

        /// Einen Punkt durch einen Skalar teilen.
        /// Das Produkt ist ein Punkt.
        /// @param c Skalar
        /// @return Quotient aus Punkt und Skalar.
        Point& operator/=(double c);

        /// Den senkrechten Abstand des Punktes zu der gedachten Geraden
        /// durch zwei andere Punkte berechnen.
        /// @param p1 Erster Punkt, durch den die gedachte Gerade verluft.
        /// @param p2 Zweiter Punkt, durch den die gedachte Gerade verluft.
        /// @return Senkrechter Abstand
        double perpendicularDistanceToLine(const Point& p1, const Point& p2) const;

        /// Den senkrechten Abstand des Punktes zu einer Geraden berechnen.
        /// @param line Gerade
        /// @return Senkrechter Abstand
        double perpendicularDistanceToLine(const Line& line) const;

        /// Den senkrechten Abstand des Punktes zu der gedachten Strecken
        /// zwischen zwei andere Punkten berechnen.
        /// @param p1 Erster Punkt, durch den die gedachte Strecke verluft.
        /// @param p2 Zweiter Punkt, durch den die gedachte Strecke verluft.
        /// @return Senkrechter Abstand
        double perpendicularDistanceToSegment(const Point& p1, const Point& p2) const;

        /// Den senkrechten Abstand des Punktes zu einer Strecke berechnen.
        /// @param line Strecke
        /// @return Senkrechter Abstand
        double perpendicularDistanceToSegment(const Line& line) const;

        inline bool isNull(void) const { return _X == 0 && _Y == 0 && _Z == 0; }

        /// X-Wert setzen.
        /// @param x X-Wert
        inline void setX(double x) { _X = x; }

        /// Y-Wert setzen.
        /// @param y Y-Wert
        inline void setY(double y) { _Y = y; }

        /// Z-Wert setzen.
        /// @param z Z-Wert
        inline void setZ(double z) { _Z = z; }

        /// X-Wert zurckgeben.
        /// @return X-Wert
        /// @see _Z
        inline double x(void) const { return _X; }

        /// Y-Wert zurckgeben.
        /// @return Y-Wert
        /// @see _Y
        inline double y(void) const { return _Y; }

        /// Z-Wert zurckgeben.
        /// @return Z-Wert
        /// @see _Z
        inline double z(void) const { return _Z; }

        /// Den Punkt in einen String wandeln.
        inline operator std::string() { return toString(); }

        /// Den Punkt in einen String wandeln.
        virtual std::string toString(void) const;

    };

};

#endif // __POINT_H_
