// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.
// $Id: mainwindow.cpp c5f95eadb0c7 2011/11/03 10:12:40 Oliver Lau <oliver@von-und-fuer-lau.de> $

#include "mainwindow.h"
#include "ui_mainwindow.h"

#include <QSettings>
#include <QtCore/QDebug>
#include <QList>
#include <QAudioDeviceInfo>
#include <qmath.h>

#include "helper.h"
#include "nui.h"
#include "flagsemaphore.h"

const QString MainWindow::Company = "von-und-fuer-lau.de";
const QString MainWindow::AppName = QObject::tr("Winkeralphabet");
#ifndef QT_NO_DEBUG
const QString MainWindow::AppVersion = "0.5 DEBUG $Date: 2011/11/03 10:12:40 $";
#else
const QString MainWindow::AppVersion = "0.5";
#endif


MainWindow::MainWindow(QWidget *parent) : QMainWindow(parent), ui(new Ui::MainWindow)
        , mAlreadyDetected(false)
        , mHelp(false)
        , mCalibrationStep(0)
{
#ifndef QT_NO_DEBUG
    mDebugWidget = new QPlainTextEdit;
    mDebugWidget->setFont(QFont("Courier New", 16, QFont::Bold));
    mDebugWidget->setWindowTitle(tr("Debug Log"));
    mDebugWidget->setGeometry(20, 20, 300, 900);
    mDebugWidget->show();
    mDebugWidget->clearFocus();
#endif

    ui->setupUi(this);

    setWindowTitle(tr("%1 %2").arg(MainWindow::AppName).arg(MainWindow::AppVersion));

    qRegisterMetaType<SkeletonList>();
    qRegisterMetaType<FlagSemaphore::Side>();

    ui->horizontalLayout->removeWidget(ui->dummyWidget1);
    ui->horizontalLayout->removeWidget(ui->dummyWidget2);
    ui->horizontalLayout->removeWidget(ui->dummyWidget3);

    m3DWidget = new ThreeDWidget;
    ui->verticalLayout->addWidget(m3DWidget);

    mPoseHelperWidget = new PoseHelperWidget;
    ui->horizontalLayout->insertWidget(0, mPoseHelperWidget);

    mDepthWidget = new DepthImageWidget;
    ui->horizontalLayout->insertWidget(0, mDepthWidget);

    mVideoWidget = new VideoWidget;
    ui->horizontalLayout->insertWidget(0, mVideoWidget);

    QObject::connect(&mNUIThread, SIGNAL(videoFrameReady(QImage)), mVideoWidget, SLOT(setFrame(const QImage&)));
    QObject::connect(&mNUIThread, SIGNAL(depthFrameReady(QImage)), mDepthWidget, SLOT(setFrame(const QImage&)));
    QObject::connect(&mNUIThread, SIGNAL(skeletonsReady(SkeletonList, SkeletonList)), m3DWidget, SLOT(setSkeletons(const SkeletonList&)));
    QObject::connect(&mNUIThread, SIGNAL(skeletonsReady(SkeletonList, SkeletonList)), this, SLOT(evaluateSkeletons(const SkeletonList&, const SkeletonList&)));
    mNUIThread.start();

    QObject::connect(&mSpeechRecognizer, SIGNAL(phraseRecognized(QString, QString, float)), this, SLOT(phraseRecognized(QString, QString, float)));

    mFlags = FlagSemaphore::FlagSemaphore::instance();
    mPoseIterator = new QMapIterator<QString, FlagSemaphore::Position>(mFlags->positions());
    nextPose();

    QObject::connect(NUI::instance(), SIGNAL(waving(int, FlagSemaphore::Side)), this, SLOT(waving(int, FlagSemaphore::Side)));

    ui->angleSlider->setMinimum(NUI_CAMERA_ELEVATION_MINIMUM);
    ui->angleSlider->setMaximum(NUI_CAMERA_ELEVATION_MAXIMUM);
    ui->angleSlider->setValue(0);
    QObject::connect(ui->angleSlider, SIGNAL(valueChanged(int)), this, SLOT(tiltChanged(int)));
    tiltChanged(ui->angleSlider->value());

    QObject::connect(ui->confidenceSlider, SIGNAL(valueChanged(int)), this, SLOT(confidenceChanged(int)));
    confidenceChanged(ui->confidenceSlider->value());

    QObject::connect(ui->checkBoxSmooth, SIGNAL(toggled(bool)), NUI::instance(), SLOT(setSmoothTransform(bool)));
    QObject::connect(ui->checkBoxWaveDetection, SIGNAL(toggled(bool)), NUI::instance(), SLOT(setWaveDetection(bool)));

    clearFocus();
    restoreAppSettings();
}


MainWindow::~MainWindow()
{
#ifndef QT_NO_DEBUG
    if (mDebugWidget)
        delete mDebugWidget;
#endif
    delete ui;
}


void MainWindow::closeEvent(QCloseEvent* event)
{
    saveAppSettings();
    QMainWindow::closeEvent(event);
#ifndef QT_NO_DEBUG
    mDebugWidget->close();
#endif
}


void MainWindow::saveAppSettings(void)
{
    QSettings settings(MainWindow::Company, MainWindow::AppName);
    settings.setValue("Kinect/tilt", NUI::instance()->tilt());
    settings.setValue("MainWindow/geometry", saveGeometry());
    settings.setValue("MainWindow/windowState", saveState());
#ifndef QT_NO_DEBUG
    settings.setValue("LogWindow/geometry", mDebugWidget->saveGeometry());
#endif
}


void MainWindow::restoreAppSettings(void)
{
    QSettings settings(MainWindow::Company, MainWindow::AppName);
    NUI::instance()->setTilt(settings.value("Kinect/tilt").toInt());
    mLastAdjustmentTime.start();
    restoreGeometry(settings.value("MainWindow/geometry").toByteArray());
    restoreState(settings.value("MainWindow/windowState").toByteArray());
#ifndef QT_NO_DEBUG
    mDebugWidget->restoreGeometry(settings.value("LogWindow/geometry").toByteArray());
#endif
}


static inline bool jointIsVisible(const Point& p)
{
    return (p.state() == NUI_SKELETON_POSITION_TRACKED) && (p.x() > 0.0f) && (p.x() < 1.0f) && (p.y() > 0.0f) && (p.y() < 1.0f);
}


void MainWindow::autoAdjustTilt(void)
{
    Q_ASSERT_X(mCalibrationStep > 0, "MainWindow::autoAdjustTilt()", "mCalibrationStep must be > 0");
    if (mCurrentSkeleton3D.isEmpty() || mCurrentSkeleton2D.isEmpty())
        return;
    switch (mCalibrationStep) {
    case 1:
        {
            m3DWidget->displayMessage(tr("Bitte auf markierte Position begeben!"));
            m3DWidget->showHotspot(true);
            mVideoWidget->showSilhouette(true);
            ++mCalibrationStep;
        }
        break;
    case 2:
        {
            const Point& hip = mCurrentSkeleton3D.at(NUI_SKELETON_POSITION_HIP_CENTER);
            const Point Hotspot(0.0f, hip.y(), NUI::HotspotDistance);
            if (hip.isInsideSphere(Hotspot, 0.24f))
                ++mCalibrationStep;
        }
        break;
    case 3:
        {
            m3DWidget->displayMessage(tr("Justieren der Neigung ... Bitte warten!"));
            ++mCalibrationStep;
            mLastAdjustmentTime.start();
        }
        break;
    case 4:
        {
            const Point& hip = mCurrentSkeleton3D.at(NUI_SKELETON_POSITION_HIP_CENTER);
            const Point Hotspot(0.0f, hip.y(), NUI::HotspotDistance);
            if (hip.isInsideSphere(Hotspot, 0.24f)) {
                if (mLastAdjustmentTime.elapsed() < 1000)
                    return;
                const Point& leftFoot = mCurrentSkeleton2D.at(NUI_SKELETON_POSITION_FOOT_LEFT);
                const Point& rightFoot = mCurrentSkeleton2D.at(NUI_SKELETON_POSITION_FOOT_RIGHT);
                const Point& head = mCurrentSkeleton2D.at(NUI_SKELETON_POSITION_HEAD);
                const long tilt = NUI::instance()->tilt();
                static const long dTilt = 3; // tilt angle is "noisy" --> tilt at least 3 degrees to make sure that angle change is above noise
                if (jointIsVisible(head)) {
                    if (jointIsVisible(rightFoot) && jointIsVisible(leftFoot)) {
                        // body is visible from head to heels --> adjustment completed
                        m3DWidget->displayMessage(tr("Fertig!"), 3000);
                        m3DWidget->showHotspot(false);
                        mVideoWidget->showSilhouette(false);
                        mCalibrationStep = 0;
                    }
                    else {
                        // feet are hidden, tilt down
                        mLastAdjustmentTime.start();
                        mNUIThread.blockSignals(true);
                        ui->angleSlider->setValue(tilt-dTilt);
                        mNUIThread.blockSignals(false);
                    }
                }
                else {
                    if (jointIsVisible(rightFoot) && jointIsVisible(leftFoot)) {
                        // head is hidden, tilt up
                        mLastAdjustmentTime.start();
                        mNUIThread.blockSignals(true);
                        ui->angleSlider->setValue(tilt+dTilt);
                        mNUIThread.blockSignals(false);
                    }
                    else {
                        mCalibrationStep = 1;
                    }
                }
            }
            else {
                mCalibrationStep = 1;
            }
        }
        break;
    }

}


void MainWindow::phraseRecognized(QString word, QString ruleName, float confidence)
{
    if (confidence > 1e-2*(float)ui->confidenceSlider->value()) {
#ifndef QT_NO_DEBUG
        mDebugWidget->appendPlainText(tr("Regel: \"%1\" (Zuverlssigkeit: %2%)").arg(ruleName).arg(100*confidence, 0, 'f', 3));
#endif
        if (ruleName == "help") {
            mHelp = true;
        }
        else if (ruleName == "thanks") {
            mHelp = false;
            mCalibrationStep = 0;
        }
        else if (ruleName == "calibrate") {
            if (mCurrentSkeleton3D.empty()) {
                m3DWidget->displayMessage(tr("Kalibrieren nur mit erkannter Person mglich!"), 3000);
            }
            else {
                mCalibrationStep = 1;
            }
        }
        else if (ruleName == "pause") {
            mNUIThread.blockSignals(true);
        }
        else if (ruleName == "continue") {
            mNUIThread.blockSignals(false);
        }
        else if (ruleName == "up") {
            mNUIThread.blockSignals(true);
            ui->angleSlider->setValue(NUI::instance()->tilt()+3);
            mNUIThread.blockSignals(false);
        }
        else if (ruleName == "down") {
            mNUIThread.blockSignals(true);
            ui->angleSlider->setValue(NUI::instance()->tilt()-3);
            mNUIThread.blockSignals(false);
        }
        else if (ruleName == "reset") {
            // ...
        }
        else if (ruleName == "quit") {
            // ...
        }
        mPoseHelperWidget->setPose(mPoseIterator->value(), mHelp);
    }
#ifndef QT_NO_DEBUG
    else
        mDebugWidget->appendPlainText(QString("#+!&?$. %1 %2%").arg(word).arg(100*confidence, 0, 'g', 3));
#endif
}


void MainWindow::waving(int userId, FlagSemaphore::Side side)
{
#ifndef QT_NO_DEBUG
    mDebugWidget->appendPlainText(tr("Spieler %1 winkt %2").arg(userId).arg((side == FlagSemaphore::LEFT)? tr("links") : tr("rechts")));
#else
    Q_UNUSED(userId);
#endif
    if (side == FlagSemaphore::RIGHT) {
        mHelp = !mHelp;
        mPoseHelperWidget->setPose(mPoseIterator->value(), mHelp);
    }
}


void MainWindow::tiltChanged(int angle)
{
    NUI::instance()->setTilt(angle);
    ui->angle->setText(QString("%1").arg(NUI::instance()->tilt()));
}


void MainWindow::confidenceChanged(int confidence)
{
    ui->confidence->setText(QString("%1%").arg(confidence));
}


void MainWindow::nextPose(void)
{
    if (mPoseIterator->hasNext()) {
        mPoseIterator->next();
        ui->log->appendPlainText(tr("Zeig mir '%1'?").arg(mPoseIterator->key()));
    }
    else {
        mPoseIterator->toFront();
    }
    mHelp = false;
    mPoseHelperWidget->setPose(mPoseIterator->value(), mHelp);
}


void MainWindow::evaluateSkeletons(const SkeletonList& skeletons, const SkeletonList& skeletonsOnImage)
{
    Q_ASSERT_X(skeletons.size() > 0, "MainWindow::evaluateSkeletons()", "SkeletonList (3D) must not be emtpy");
    Q_ASSERT_X(skeletonsOnImage.size() > 0, "MainWindow::evaluateSkeletons()", "SkeletonList (2D) must not be emtpy");

    mDepthWidget->setSkeletons(skeletonsOnImage);
    mVideoWidget->setSkeletons(skeletonsOnImage);
    const Skeleton& skel = skeletons.first();
    mCurrentSkeleton3D = skel;
    mCurrentSkeleton2D = skeletonsOnImage.first();

    if (mCalibrationStep > 0) { // calibration mode
        if (mCalibrationStep == 1)
            ui->log->clear();
        autoAdjustTilt();
    }
    else { // normal mode
        float leftHandAngle = 0.0f;
        bool leftArmCorrect = skel.armPoints(FlagSemaphore::LEFT, mPoseIterator->value().left(), &leftHandAngle);
        ui->leftAngle->setText(QString("%1").arg((int)leftHandAngle));
        float rightHandAngle = 0.0f;
        bool rightArmCorrect = skel.armPoints(FlagSemaphore::RIGHT, mPoseIterator->value().right(), &rightHandAngle);
        ui->rightAngle->setText(QString("%1").arg((int)rightHandAngle));
        const Point& distance = skel.at(NUI_SKELETON_POSITION_HIP_CENTER);
        ui->distance->setText(QString("%1").arg(int(100*distance.z())));
        ui->floorLevel->setText(QString("%1").arg(int(-100*NUI::instance()->floorLevel())));
        if (leftArmCorrect && rightArmCorrect) {
            if (!mAlreadyDetected) {
                mAlreadyDetected = true;
                mFirstDetectionTime.start();
                ui->log->appendPlainText(tr("Halten!"));
            }
            if (mFirstDetectionTime.elapsed() > 1000) {
                mAlreadyDetected = false;
                ui->log->appendPlainText(tr("Richtig!"));
                nextPose();
            }
        }
        else {
            if (mAlreadyDetected && mFirstDetectionTime.elapsed() > 200)
                ui->log->appendPlainText(tr("Konzentrier dich! Zeig mir '%1'?").arg(mPoseIterator->key()));
            mFirstDetectionTime.start();
            mAlreadyDetected = false;
        }
    }
}
