// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.
// $Id: nui.h 0fa08f2d4845 2011/10/14 13:23:44 Oliver Lau <oliver@von-und-fuer-lau.de> $

#ifndef __NUI_H_
#define __NUI_H_

#include <Windows.h>
#include <Propsys.h>
#include <dmo.h>
#include <wmcodecdsp.h>
#include <MSR_NuiApi.h>
#include <MSRKinectAudio.h>

#include "helper.h"
#include "point.h"
#include "vector.h"
#include "flagsemaphore.h"
#include "skeleton.h"

#include <QtCore/QString>
#include <QtCore/QVector>
#include <QtCore/QMutex>
#include <QtCore/QTime>
#include <QtCore/QtDebug>
#include <QtGui/QColor>
#include <QtGui/QImage>
#include <QtGui/QTransform>

#undef NUI_WITH_AUDIO
#undef NUI_ENSURE_REENTRANCE

class NUI : public QObject // Singleton
{
    Q_OBJECT
public:
    static NUI* instance(void);
    static const QColor JointColorTable[NUI_SKELETON_POSITION_COUNT];
    static const int FrameInterval = 33; // one frame per 33 ms
    static const float HotspotDistance;

    inline const QString& deviceName(void) const { return mDeviceName; }

    enum WaveDirection { NONE, LEFT, RIGHT };

    long tilt(void);
    void setTilt(long);

    void enableTracking(HANDLE event = NULL);
    void disableTracking(void);

    HANDLE openStream(NUI_IMAGE_TYPE eImageType, NUI_IMAGE_RESOLUTION eResolution, HANDLE event);
    QImage getStreamVideoFrame(HANDLE stream);
    QImage getStreamDepthFrame(HANDLE stream);
    void getSkeletons(__OUT SkeletonList& skeletons, __OUT SkeletonList& skeletonsOnImage, int trackingState = NUI_SKELETON_TRACKED);


    inline HRESULT resultCode(void) const { return mResult; }
    inline const LimbList& limbs(void) const { return mLimbs; }
    inline float floorLevel(void) const { return mMetersAboveFloor; }

public slots:
    inline void setSmoothTransform(bool enabled = true) { mSmoothTransform = enabled; }
    inline void setWaveDetection(bool enabled = true) { mWaveDetection = enabled; qDebug() << "setWaveDetection(" << enabled << ")"; }

signals:
    void waving(int userId, FlagSemaphore::Side side);

private:
    explicit NUI(QObject* parent = NULL);
    ~NUI();

#if defined(NUI_ENSURE_REENTRANCE)
    static QMutex mInstanceMutex;
#endif

#if defined(NUI_WITH_AUDIO)
    void initAudio(void);
#endif

    static QRgb depthToRGB(quint16 s);

    static NUI* mInstance;

    HRESULT mResult;
    LimbList mLimbs;

    float mMetersAboveFloor;    // meters
    bool mSmoothTransform;

    // variables for wave detection
    bool mWaveDetection;
    QTime mLastWaveDirectionChangeRight;
    QTime mLastWaveDirectionChangeLeft;
    float mLastWaveDirectionChangePositionRightX;
    float mLastWaveDirectionChangePositionLeftX;
    WaveDirection mWaveDirectionRight;
    WaveDirection mWaveDirectionLeft;
    float mLastRightHandPositionX;
    float mLastLeftHandPositionX;
    int mWaveCountRight;
    int mWaveCountLeft;

    bool detectWaving(__IN const Skeleton& skeleton,
                      __IN int userId,
                      __IN FlagSemaphore::Side side,
                      __IN __OUT QTime& lastWaveDirectionChange,
                      __IN __OUT float& lastWaveDirectionChangePositionX,
                      __IN __OUT float& lastHandPositionX,
                      __IN __OUT NUI::WaveDirection& waveDirection,
                      __IN __OUT int& waveCount);

#if defined(NUI_WITH_AUDIO)
    IMediaObject* mDMO;
    ISoundSourceLocalizer* mSoundSourceLocalizer;
#endif

    class Watchdog {
    public:
        ~Watchdog() {
            if (NUI::mInstance)
                delete NUI::mInstance;
        }
    };

    static Watchdog mWatchdog;
    friend class Watchdog;

    QString mDeviceName;
};


#endif // __NUI_H_
