// Copyright (c) 2011 Oliver Lau <oliver@von-und-fuer-lau.de>
// All rights reserved.
// $Id: point.cpp cf169bb4ae86 2011/10/06 17:09:09 Oliver Lau <oliver@von-und-fuer-lau.de> $

#include <math.h>

#include "point.h"
#include "vector.h"
#include "line.h"
#include "nui.h"


template <typename T>
static inline T sqr(T x) {
    return x * x;
}


bool Point::isInsideSphere(const Point& c, float r) const
{
    return sqr(c.x()-mX) + sqr(c.y()-mY) + sqr(c.z()-mZ) < sqr(r);
}


float Point::perpendicularDistanceToSegment(const Point& p1, const Point& p2) const
{
    const Vector vec(p1, p2);
    const Vector ref(*this - p1);
    const float hypotenuse = ref.length();
    const float A = ref * vec;
    if (A < 0)
        return ref.length();
    const float B = sqr(vec);
    if (B < A)
        return (*this - p2).length();
    return sqrt(sqr(hypotenuse) - sqr(A/vec.length()));
}


float Point::perpendicularDistanceToSegment(const Line& line) const
{
    return perpendicularDistanceToSegment(line.p1(), line.p2());
}


float Point::perpendicularDistanceToLine(const Point& p1, const Point& p2) const
{
    const Vector vec(p1, p2);
    const Vector ref(*this - p1);
    const float hypotenuse = ref.length();
    const Vector unified = vec.normalized();
    const float A = ref * unified;
    return sqrt(sqr(hypotenuse) - sqr(A));
}


float Point::perpendicularDistanceToLine(const Line& line) const
{
    return perpendicularDistanceToLine(line.p1(), line.p2());
}


Point& Point::operator-=(const Vector& v)
{
    mX -= v.u();
    mY -= v.v();
    mZ -= v.w();
    return *this;
}


Point& Point::operator+=(const Vector& v)
{
    mX += v.u();
    mY += v.v();
    mZ += v.w();
    return *this;
}


Point& Point::operator*=(float c)
{
    mX *= c;
    mY *= c;
    mZ *= c;
    return *this;
}


Point& Point::operator/=(float c)
{
    mX /= c;
    mY /= c;
    mZ /= c;
    return *this;
}


Vector operator-(const Point& p1, const Point& p2)
{
    return Vector(p1.x() - p2.x(), p1.y() - p2.y(), p1.z() - p2.z());
}


Point operator+(const Point& p, const Vector& v)
{
    return Point(p.x() + v.u(), p.y() + v.v(), p.z() + v.w());
}


Point operator-(const Point& p, const Vector& v)
{
    return Point(p.x() - v.u(), p.y() - v.v(), p.z() - v.w());
}


Point operator*(const Point& p, float c)
{
    return Point(p.x() * c, p.y() * c, p.z() * c);
}


Point operator*(float c, const Point& p)
{
    return Point(p.x() * c, p.y() * c, p.z() * c);
}


Point operator/(const Point& p, float c)
{
    return Point(p.x() / c, p.y() / c, p.z() / c);
}


Point operator+(const Point& p1, const Point& p2)
{
    return Point(p1.x() + p2.x(), p1.y() + p2.y(), p1.z() + p2.z());
}


bool operator==(const Point& p1, const Point& p2)
{
    return p1.x() == p2.x() && p1.y() == p2.y();
}


bool operator!=(const Point& p1, const Point& p2)
{
    return !(p1 == p2);
}


#ifndef QT_NO_DEBUG_STREAM
QDebug operator<<(QDebug dbg, const Point &p) {
    dbg.nospace() << "Point(" <<
            p.x() << ", " <<
            p.y() << ", " <<
            p.z() << ", '";
    switch (p.state()) {
    case NUI_SKELETON_POSITION_TRACKED:
        dbg.nospace() << "TRACKED";
        break;
    case NUI_SKELETON_POSITION_NOT_TRACKED:
        dbg.nospace() << "NOT TRACKED";
        break;
    case NUI_SKELETON_POSITION_INFERRED:
        dbg.nospace() << "INFERRED";
        break;
    default:
        dbg.nospace() << "<invalid value>";
        break;
    }
    dbg.nospace() << "')";
    return dbg.space();
}
#endif
