package com.linkesoft.eierlaufen;

import android.hardware.Sensor;
import android.hardware.SensorEvent;
import android.hardware.SensorEventListener;
import android.hardware.SensorManager;
import android.util.Log;

public class Egg implements SensorEventListener {

	// alle Werte in SI-Einheiten (m, s), Winkel in rad

	// Konstanten
	// Tiefpassfilter
	 final float alpha = 0.9f;
	 // Haftreibung
	 final float staticfriction=0.06f;
	 final float slidingfriction=0.8f*staticfriction;
	 // Schwellwert fr Schrittzaehlung, relativ zur Erdbeschleunigung
	 final float stepLimit=0.04f;
	 // Schwierigkeitsgrad (Zeitverlangsamung, 1=Echtzeit, kaum zu schaffen)
	 final float difficulty=0.2f;
	 
	 final float gsquare=SensorManager.GRAVITY_EARTH*SensorManager.GRAVITY_EARTH;

	 // Ei-Variablen
	private float x,y; // Position des virtuellen Eis auf der Gerteoberflaeche	
	private float vx,vy; // Geschwindigkeit des Eis 
	private float phi; // Winkel in Radiant zur Nordrichtung	
	private float startPhi; // Winkel bei Start
	
	// letzte Messwerte
	private float[] aold=new float[3]; // fr gleitenden Durchschnitt
	private float[] acurrent=new float[3]; // fr Magnetfeld
	private long lastTimeStampAccelerationNS; // in ns

	private final EggView eggView;
	
	public boolean update; // aktualisiere EggView

	// Schrittzaehler
	private boolean stepUp; 
		
	public StepCounterUpdateHandler stepCounterHandler;

	public static interface StepCounterUpdateHandler {
		public void onStep();
	}

	public Egg(EggView eggView)
	{
		this.eggView = eggView;
		reset();
	}
	
	public void reset()
	{
		phi=0;
		aold=new float[3];
		acurrent=new float[3];
		lastTimeStampAccelerationNS=0;
		stepUp=false;
		resetEgg();
	}
	
	public void resetEgg()
	{
		x=y=0;
		vx=vy=0;
		startPhi=phi;
		updateEgg();
	}
	
	private void onAccelerate(float[] a, long timestamp) {
		acurrent=a.clone();
		// bestimme Zeitintervall seit letzter Messung
		float dt = 0;
		if (lastTimeStampAccelerationNS != 0)
			dt = (timestamp - lastTimeStampAccelerationNS) * 1e-9f * difficulty;
		lastTimeStampAccelerationNS = timestamp;

		// Tiefpassfilter
		for (int i = 0; i < a.length; i++)
			a[i] = alpha * aold[i] + (1f - alpha) * a[i];
		aold = a.clone();
		float ax = a[0];
		float ay = a[1];
		float az = a[2];

		final float asquare = ax * ax + ay * ay;
		// Haftreibung haelt den Koerper in Ruhe
		if (vx == 0 && vy == 0 && asquare < staticfriction * staticfriction * (gsquare - asquare))
			return;

		// Gleitreibung verringert resultierende Kraft
		// (Gleitreibung < Haftreibung)
		if (asquare < gsquare) // nicht freier Fall?
		{
			float arsquare = slidingfriction * slidingfriction * (gsquare - asquare);
			ax -= (float) (Math.sqrt(arsquare / asquare) * ax);
			ay -= (float) (Math.sqrt(arsquare / asquare) * ay);
		}

		// zurckgelegte Strecke des Eis
		x -= vx * dt + ax * dt * dt / 2f;
		y += vy * dt + ay * dt * dt / 2f;
		// neue Geschwindigkeit
		vx = vx + ax * dt;
		vy = vy + ay * dt;

		// Schrittbestimmung
		if (stepUp) {
			Log.v("Sensor","stepUp az="+az+" limit="+((1f - stepLimit) * SensorManager.GRAVITY_EARTH));
			if (az < (1f - stepLimit) * SensorManager.GRAVITY_EARTH) {
				if (stepCounterHandler != null) {
					stepCounterHandler.onStep();
					stepUp = false;
				}
			}

		} else {
			Log.v("Sensor","stepDown az="+az+" limit="+((1f + stepLimit) * SensorManager.GRAVITY_EARTH));
			if (az > (1f + stepLimit) * SensorManager.GRAVITY_EARTH)
				stepUp = true;
		}

		if(update)
			updateEgg();
	}

	private void onMagneticField(float[] values, long timestamp) {
		if(acurrent[0]==0 && acurrent[1]==0 && acurrent[2]==0)
			return; // noch keine Beschleunigungswerte
		// Berechne Rotationsmatrix aus Werten 
		// von Beschleunigungs-(Gravitations-) und Magnetsensor
		float rotationMatrix[] = new float[3 * 3];
		if (SensorManager.getRotationMatrix(rotationMatrix, null, acurrent, values)) {
			float orientation[] = new float[3];
			SensorManager.getOrientation(rotationMatrix, orientation);
			phi = orientation[0]; // -PI..PI
			if(startPhi==0)
			{
				startPhi=phi;
				return;
			}
			Log.v("Egg", "Current compass reading " + phi / Math.PI * 180);
			if(update)
				updateEgg();
		}
	}
	private void onGyroscope(float[] values, long timestamp) {
		// hier kann alternativ der Neigungswinkel ueber Aenderungen 
		// der Drehgeschwindigkeit bestimmt werden
	}

	private void updateEgg() {		
		eggView.setCoordinates(x,y,startPhi-phi);		
	}
	
// SensorEventListener API
	public void onSensorChanged(SensorEvent event) {
		if (event.sensor.getType() == Sensor.TYPE_ACCELEROMETER) {
			onAccelerate(event.values, event.timestamp);
		} else if (event.sensor.getType() == Sensor.TYPE_MAGNETIC_FIELD) {
			onMagneticField(event.values, event.timestamp);
		} else if (event.sensor.getType() == Sensor.TYPE_GYROSCOPE) {
			onGyroscope(event.values, event.timestamp);
		}
	}
	public void onAccuracyChanged(Sensor sensor, int accuracy) {	
	}
}
