package com.linkesoft.bbingo;

import java.util.Collections;
import java.util.List;

import android.app.Activity;
import android.app.AlertDialog;
import android.app.AlertDialog.Builder;
import android.content.DialogInterface;
import android.content.Intent;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.media.AudioManager;
import android.media.SoundPool;
import android.os.Bundle;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;

import com.linkesoft.bbingo.BBingoDB.WordList;

/**
 * Hauptklasse-Activity der B-Bingo-Anwendung,
 * zeigt ein Gitter von 5x5 Buttons zur Auswahl von Buzzwoertern.
 * 
 * Siehe {@link http://de.wikipedia.org/wiki/Buzzword-Bingo}
 * 
 * @author Andreas Linke
 *
 */
public class Main extends Activity {
	private static final int SIZE = 5;
	
	private String currentWordlist=null;

	private BBingoDB db;
	private SoundPool soundPool; // Erfolgsfanfare
	private int fanfareSoundID; // ID der geladenen Sound-Resource
	private int streamID; // ID des gerade gespielten Sounds (zum Stoppen)
	
/**
 * Startmethode, wird beim ersten Start der Activity aufgerufen,
 * auch bei Aenderung der Orientierung (horizontal/vertikal).
 * Definiere GUI-Layout und verknuepfe Event-Handler.	
 */
	@Override
	public void onCreate(Bundle savedInstanceState) {
		super.onCreate(savedInstanceState);
		setContentView(R.layout.main); // aus XML
		// verknuepfe Buttons mit Aktionen	
		for (int y = 0; y < SIZE; y++)
			for (int x = 0; x < SIZE; x++) {
				Button button = (Button) findViewById(getButtonID(x, y));				
				button.setOnClickListener(onButtonClick); 
			}
		db=new BBingoDB(this); // oeffne Datenbank, wird in onDestroy wieder geschlossen
		
	    // lade Soundschnipsel fuer Gewinn-Fanfare
	    soundPool = new SoundPool(1, AudioManager.STREAM_MUSIC, 0);
		fanfareSoundID=soundPool.load(this, R.raw.fanfare, 1); // Verweis auf fanfare.mp3 im res/raw Ordner des Projekts		
	}
	
	/**
	 * Spiele vorher im soundPool geladenen Sound ab.
	 */
	private void playFanfare() {
		AudioManager audioManager = (AudioManager) getSystemService(AUDIO_SERVICE);
		// berechne relative Lautstaerke, basierend auf vom Benutzer
		// eingestellter Lautstaerke und Maximum (int-Werte!)
		float volume = (float) audioManager.getStreamVolume(AudioManager.STREAM_MUSIC) / 
						(float) audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
		// spiele den in onCreate geladenen Sound
		streamID = soundPool.play(fanfareSoundID, volume, volume, 1, 0, 1f);
	}
	
	/**
	 * Wird nach onCreate und nach dem Beenden einer anderen Activity aufgerufen.
	 * Lade die aktuelle Wortliste. Wenn sie nicht mit der aktuellen Liste uebereinstimmt,
	 * mische und setze alle Buttons neu.
	 */
	@Override
	protected void onResume() {
		super.onResume();
		loadWordList();
	}

	/**
	 * Hole aktuelle Wortlist aus der DB.
	 * Falls noch keine Liste gespeichert wurde,
	 * wird auf eine eingebaute Standard-Liste zurueckgegriffen.
	 */
	private void loadWordList()
	{
		// 
		WordList wordlist = db.getWordList(Prefs.getID(this));
		if (wordlist == null) {
			// keine Wortliste zur ID gefunden, nimm erste
			long id = db.getFirstWordListID();
			if (id == 0) {
				// keine Wortliste gespeichert
				id = db.createDefaultWordList(this); // erzeuge Standard-Liste
			}
			Prefs.setID(this, id);
			wordlist = db.getWordList(id);
		}
		setTitle("B-Bingo: "+wordlist.title); // zeige Titel der aktuelle Wortliste im Titel der Activity
		if (currentWordlist == null || !currentWordlist.equals(wordlist.words)) {
			// Wortliste hat sich geaendert, mische und setze Button-Titel entsprechend
			currentWordlist = wordlist.words;
			shuffleWords(false); // keine Animation, da schon beim Layout animiert wurde
		}
	}
	
	@Override
	protected void onDestroy() {
		db.close(); // jede Activity hat eine eigene Datenbank-Verbindung, die beim Beenden geschlossen werden muss
		super.onDestroy();
	}
/**
 * Button-Klick-Handler fuer alle Buttons	
 */
	final View.OnClickListener onButtonClick=new View.OnClickListener() {
		@Override
		public void onClick(View v) {
			toggleButtonState((Button) v);
			checkForBingo();
		}
	};
	
/**
 * Anlegen von Menues (die per Menue-Taste aufgerufen werden)	
 */
	@Override
	public boolean onCreateOptionsMenu(android.view.Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.main, menu);
		return super.onCreateOptionsMenu(menu);
	};

/**
 * Auswahl eines Menue-Eintrags	
 */
	@Override
	public boolean onOptionsItemSelected(MenuItem item) {
		switch (item.getItemId()) {
		/*
		case R.id.edit: {
				Intent intent = new Intent(this, EditWordList.class);
				intent.putExtra(EditWordList.ID, Prefs.getID(this)); // uebergebe aktuelle ID
				startActivity(intent);
			}
			break;
			*/
		case R.id.lists:
			startActivityForResult(new Intent(this, WordLists.class), R.id.lists);
			break;
		case R.id.prefs:
	    	startActivity(new Intent(this,PrefsActivity.class));
			break;
		}
		return super.onOptionsItemSelected(item);
	}

/**
 * Wird nach Beenden einer per <code>startActivityForResult</code> gestarteten 
 * Activity gerufen	
 */
	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data) {
		if (requestCode == R.id.lists && resultCode == RESULT_OK) {
			// eine andere Liste wurde ausgewaehlt
			long id = data.getExtras().getLong(WordLists.ID);
			Prefs.setID(this, id);
		}
		super.onActivityResult(requestCode, resultCode, data);
	}

/**
 * Liefere numerische Button-ID per Reflection aus den Konstanten der R-Klasse
 * @param x Spaltennummer 0..4
 * @param y Zeilennummer 0..4
 * @return
 */
	private int getButtonID(int x, int y) {
		try {
			// lookup id by reflection
			// button id is a static int defined in R.id, e.g. R.id.Button01
			String buttonid = "Button" + y + "" + (x + 1); // e.g. "Button01" 
			return R.id.class.getField(buttonid).getInt(null);
		} catch (Exception e) {
			// reflection lookup could throw exceptions e.g. if the button is
			// not found
			throw new RuntimeException("Internal error", e);
		}
	}

	private boolean isButtonSet(Button button) {
		return button.getTag() != null;
	}

/**
 * Aendere Zustand eines Buttons von nicht gesetzt auf gesetzt 
 * und umgekehrt. Dabei wird die Button-Farbe entsprechend angepasst. 
 * @param button der zu aendernde Button
 */
	private void toggleButtonState(Button button) {
		if (isButtonSet(button)) {
			button.getBackground().clearColorFilter();
			button.setTag(null);
		} else {
			// Button-Hintergrund-Farbe wird ueber einen Filter geaendert 
			button.getBackground().setColorFilter(Color.GREEN,PorterDuff.Mode.SRC_ATOP);
			button.setTag("x"); // verwende die frei tag-Property zum Speichern des Zustands
		}
	}
	
/**
 * Pruefe auf vollstaendige ausgewaehlte Button-Reihe 
 * vertikal, horizontal und die beiden Diagonalen 
 */
	private void checkForBingo() {
		boolean bingo = false;
		int x, y;
		// vertikal
		for (x = 0; x < SIZE && !bingo; x++) {
			for (y = 0; y < SIZE; y++) {
				Button button = (Button) findViewById(getButtonID(x, y));
				if (!isButtonSet(button))
					break;
			}
			bingo = (y == SIZE);
		}
		// horizontal
		for (y = 0; y < SIZE && !bingo; y++) {
			for (x = 0; x < SIZE; x++) {
				Button button = (Button) findViewById(getButtonID(x, y));
				if (!isButtonSet(button))
					break;
			}
			bingo = (x == SIZE);
		}
		// diagonal
		for (x = 0; x < SIZE && !bingo; x++) {
			Button button = (Button) findViewById(getButtonID(x, x));
			if (!isButtonSet(button))
				break;
		}
		if (x == SIZE)
			bingo = true;
		for (x = 0; x < SIZE && !bingo; x++) {
			Button button = (Button) findViewById(getButtonID(x, SIZE - 1 - x));
			if (!isButtonSet(button))
				break;
		}
		if (x == SIZE)
			bingo = true;

		if (bingo) {
			// Gewonnen, zeige ein einfaches Popup und spiele Sound
			if(!Prefs.getNoSound(this))
				playFanfare(); // asynchron
			// Die Methoden geben jeweils wieder das Builder-Objekt zurueck
			// und lassen sich so leicht verketten
			Builder dlg=new AlertDialog.Builder(this)
					.setTitle(R.string.BBingoWonTitle)
					.setMessage(R.string.BBingoWon)
					.setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {						
						@Override
						public void onClick(DialogInterface dialog, int which) {
							soundPool.stop(streamID); // stoppe Sound, falls er noch gespielt wird
						}
					})
					.setNeutralButton(R.string.New, new DialogInterface.OnClickListener() {						
						@Override
						public void onClick(DialogInterface dialog, int which) {
							soundPool.stop(streamID); // stoppe Sound
							shuffleWords(true);	
						}
					});
			dlg.show();
		}
	}
/**
 * Worte der Wortliste mischen und auf die Buttons verteilen
 */
	private void shuffleWords(boolean animate)
	{
		List<String> words=BBingoDB.stringToList(currentWordlist);
		// Liste darf nicht leer sein
		if(words.size()==0)	
			words.add(""); 
		// Liste mischen und den Buttons zuweisen
		Collections.shuffle(words);
		int i=0;
		for (int x = 0; x < SIZE; x++) 
			for (int y = 0; y < SIZE; y++)
			{
				Button button=(Button) findViewById(getButtonID(x, y));
				if(isButtonSet(button))
					toggleButtonState(button); // setze Button state zurueck
				if(i>=words.size())
					{
					// overflow
					Collections.shuffle(words);
					i=0;
					}
				button.setText(words.get(i++)); 
			}		
		if(animate) {
			ViewGroup mainViewGroup = (ViewGroup) findViewById(R.id.LinearLayoutMain);
			for (int j = 0; j < mainViewGroup.getChildCount(); j++) {
				ViewGroup rowViewGroup = (ViewGroup) mainViewGroup.getChildAt(j);
				rowViewGroup.startLayoutAnimation();
			}	
		}
			
	}
}