using System;
using System.Windows.Forms;

namespace Gesundheitskarte
{
    public class eGK
    {
        private IntPtr hCard;
        private WinSCard.SCARD_IO_REQUEST pciSend, pciRecv;



        /// <summary>
        /// Der Konstruktor
        /// </summary>
        /// <param name="hKarte">Kartenhandle</param>
        /// <returns></returns>
        public eGK(IntPtr hKarte)
        {
            hCard = hKarte;

            // Strukturen initialisieren
            pciSend.dwProtocol = (uint)SCardProtocolIdentifiers.T1;
            pciSend.cbPciLength = sizeof(uint) * 2;
            pciRecv.dwProtocol = (uint)SCardProtocolIdentifiers.T1;
            pciRecv.cbPciLength = sizeof(uint) * 2;
        }



        /// <summary>
        /// Die ausgewaehlte PIN (passwordReference) ueberschreiben
        /// </summary>
        /// <param name="pin">passwordreference</param>
        /// <param name="PinStatus">der ermittelte Transport-PIN-Status</param>
        /// <param name="oldPIN">alte PIN</param>
        /// <param name="newPIN">neue PIN</param>
        /// <returns>bei Erfolg true, sonst false</returns>
        public bool setPIN(byte pinID, PinStatus status, byte[] newPIN, byte[] oldPIN)
        {
            byte[] empfang = new byte[128];
            uint lReturn;
            int dwCnt = 0;

            byte[] BODY = {    0x00,      // CLASS-Byte nach ISO7816
                               0x24,      // INS = CHANGE REFERENCE DATA
                               0x00,      // P1 = 0 
                               pinID};    // P2 = passwordReference 
            byte[] PIN_CHANGE;
            byte[] oldSecret;

            // Kommando-APDU entsprechend PIN-Status aufbereiten
            switch (status)
            {

                // Transport_PIN_Leerpin_1: P1 = 1, kein oldSecret
                case PinStatus.Transport_PIN_Leerpin_1:
                    
                    BODY[2] = 0x01;

                    PIN_CHANGE = new byte[4 + 1 + 8];
                    Array.Copy(BODY, PIN_CHANGE, 4);
                    PIN_CHANGE[4] = 8; // Laengenangabe

                    EncodePin(newPIN, PIN_CHANGE, 5);

                    break;

                // Transport_PIN_Leerpin_2: P1 = 0, oldSecret = 20FF FFFF FFFF FFFF
                case PinStatus.Transport_PIN_Leerpin_2:

                    PIN_CHANGE = new byte[4 + 1 + 8 + 8];
                    oldSecret = new byte[0];
                    Array.Copy(BODY, PIN_CHANGE, 4);

                    // Laengenangabe des DATA-Feldes
                    PIN_CHANGE[4] = 16;

                    // Altes Passwort in einem Format–2–PIN Block
                    EncodePin(oldSecret, PIN_CHANGE, 5);

                    // Neues Passwort in einem Format–2–PIN Block (Beschreibung in N81 in Teil 1 der Spez.)
                    EncodePin(newPIN, PIN_CHANGE, 13);

                    break;

                // Transport_PIN_0000: P1 = 0, oldSecret = 2400 00FF FFFF FFFF
                case PinStatus.Transport_PIN_0000:

                    PIN_CHANGE = new byte[4 + 1 + 8 + 8];
                    oldSecret = new byte[4] { 0x30, 0x30, 0x30, 0x30 };
                    Array.Copy(BODY, PIN_CHANGE, 4);

                    PIN_CHANGE[4] = 8 + 8; // Laengenangabe
                    EncodePin(oldSecret, PIN_CHANGE, 5);

                    // Neues Passwort in einem Format–2–PIN Block (Beschreibung in N81 in Teil 1 der Spez.)
                    EncodePin(newPIN, PIN_CHANGE, 13);
            
                    break;

                // regulaeres Passwort: P1 = 0, oldSecret = aktuelle PIN
                case PinStatus.RetryCounter:

                    PIN_CHANGE = new byte[4 + 1 + 8 + 8];
                    Array.Copy(BODY, PIN_CHANGE, 4);

                    PIN_CHANGE[4] = 8 + 8; // Laengenangabe

                    // Altes Passwort in einem Format–2–PIN Block (Beschreibung in N81 in Teil 1 der Spez.)
                    EncodePin(oldPIN, PIN_CHANGE, 5);

                    // Neues Passwort in einem Format–2–PIN Block (Beschreibung in N81 in Teil 1 der Spez.)
                    EncodePin(newPIN, PIN_CHANGE, 13);

                    break;
                
                    
                default:    // kein unterstuetzer PIN-Status
                    return false;
            }

            // APDU an eGK senden
            dwCnt = empfang.Length;
            lReturn = WinSCard.SCardTransmit(
                                    hCard,
                                    ref pciSend,
                                    PIN_CHANGE,
                                    PIN_CHANGE.Length,
                                    IntPtr.Zero,
                                    empfang,
                                    ref dwCnt);
            
            // Aufruf von SCardTransmit fehlerfrei?
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardTransmit(Read Record)");
                return false;
            }

            // Antwort_APDU im Erfolgsfalls
            if ((empfang[dwCnt - 2] == 0x90) && (empfang[dwCnt - 1] == 0x00))
            {
                MessageBox.Show("Neue PIN wurde erfolgreich auf der eGK abgelegt.", "Erfolg", MessageBoxButtons.OK, MessageBoxIcon.Information);
                return true;
            }


            // Antwort_APDU bei blockierter PIN-Verifiaktionsmethode
            if ((empfang[dwCnt - 2] == 0x63) && (empfang[dwCnt - 1] == 0x83))
            {
                MessageBox.Show("PIN konnte nicht gesetzt werden!! Verwendete Verifikationsmethode ist blockiert.", "PIN-Fehler",
                                MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            // Antwort_APDU bei falscher oldPIN
            if ((empfang[dwCnt - 2] == 0x63) && ((empfang[dwCnt - 1] & 0x0F0) == 0xC0))
            {
                int fehlbedZaehler = (int)empfang[dwCnt - 1] & 0x0F; // unterstes Nibble enthaelt den Fehlbedienungszaehler
                string ausgabe = "Die neue PIN konnte nicht gesetzt werden, da die alte PIN nicht akzeptiert wurde!!!\n\n";

                if (fehlbedZaehler != 0)
                    ausgabe += " Es sind nur noch " + fehlbedZaehler + " Versuche möglich.";
                else
                    ausgabe += " Es sind keine weiteren Versuche mehr möglich.";
                MessageBox.Show(ausgabe, "PIN-Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
                return false;
            }

            // Antwort_APDU im "allgemeinen" Fehlerfall
            string fehler = "Fehler beim Setzen der PIN!\n\nFehler =" + string.Format("{0:X2}:{1:X2}", empfang[0], empfang[1]);
            MessageBox.Show(fehler, "PIN-Fehler", MessageBoxButtons.OK, MessageBoxIcon.Error);
            return false;
        }



        /// <summary>
        /// PIN in einen Format-2-PIN Block umwandeln
        /// </summary>
        /// <param name="pin">passwordreference (PIN) als Byte-Folge</param>
        /// <param name="target">Zielbereich</param>
        /// <param name="startIndex">"Startpunkt" der Ablage des Blocks im Zielbereich</param>
        /// <returns></returns>
        private void EncodePin(byte[] pin, byte[] target, int startIndex)
        {
            // Passwort in einen Format–2–PIN Block packen (Beschreibung in N81 in Teil 1 der Spez.)
            target[startIndex] = (byte)(0x20 + pin.Length);  // Erste Nibble MUSS 2 sein, zweites gibt Anzahl der PIN-Nibbles an

            // Ablage der PIN-Oktette
            byte octet = 0;
            for (int i = 0; i < 14; ++i)
            {
                byte nibble;
                if (i < pin.Length)
                    nibble = (byte)(pin[i] - 0x30);
                else
                    nibble = 0xf;
                if (i % 2 == 0)
                    octet = (byte)(nibble << 4);
                else
                {
                    octet |= nibble;
                    target[startIndex + 1 + i / 2] = octet;
                }
            }
        }



        /// <summary>
        /// Status der ausgewaehlten PIN (passwordReference) ermitteln
        /// </summary>
        /// <param name="pin">passwordreference</param>
        /// <param name="PinStatus">der ermittelte PIN-Status</param>
        /// <param name="fehlbedZaehler">der Wert des Fehlbedienungszaehlers</param>
        /// <param name="ausgabe">Textausgabe des PIN-Status</param>
        /// <returns>bei Erfolg true, sonst false</returns>
        public bool getPinStatus(byte pin, ref PinStatus status, ref int fehlbedZaehler, ref String ausgabe)
        {
            byte[] empfang = new byte[128];
            uint lReturn;
            int dwCnt = 0;

            byte[] PIN_STATUS = {    0x80,      // Class-Byte ist proprietaer (nicht nach ISO7816)
                                     0x20,      // INS = GET PIN STATUS
                                     0x00,      // P1 = 0
                                     pin};      // P2 = passwordReference
                                     
            dwCnt = empfang.Length;
            lReturn = WinSCard.SCardTransmit(
                                    hCard,
                                    ref pciSend,
                                    PIN_STATUS,
                                    PIN_STATUS.Length,
                                    IntPtr.Zero,
                                    empfang,
                                    ref dwCnt);

            // Aufruf von SCardTransmit fehlerfrei?
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardTransmit(Read Record)");
                return false;
            }

            // Antwort_APDU mit der Nachricht "kein Transportschutz"
            // dann steht im niederwertigen Nibble der Wert des 
            // Fehlbedienungszaehlers dieser PIN
            if ((empfang[dwCnt - 2] == 0x63) && ((empfang[dwCnt - 1] & 0x0F0) == 0xC0))
            {
                status = PinStatus.RetryCounter;
                fehlbedZaehler = (int)empfang[dwCnt - 1] & 0x0F; // unterstes Nibble enthaelt den Fehlbedienungszaehler
                ausgabe = "reguläres Passwort, Fehlbedienungszähler = " + fehlbedZaehler;
                return true;
            }

            // Antwort_APDU mit der Nachricht "Passwort wurde nicht gefunden"
            if ((empfang[dwCnt - 2] == 0x6A) && (empfang[dwCnt - 1] == 0x88))
            {
                ausgabe = "Passwortobjekt nicht gefunden";
                status = PinStatus.PasswordNotFound;
                fehlbedZaehler = 0;
                return false;
            }

            // Antwort_APDU mit der Nachricht "Passwort nicht noetig"
            if ((empfang[dwCnt - 2] == 0x90) && (empfang[dwCnt - 1] == 0x00))
            {
                ausgabe = "kein Passwort nötig";
                status = PinStatus.NoError;
                fehlbedZaehler = 0;
                return false;
            }

            // Antwort_APDU mit der Nachricht "Passwort mit Transportschutz"
            // dann steht im niederwertigen Nibble der Wert die Art des 
            // verwendeten Transportschutzes
            if ((empfang[dwCnt - 2] == 0x62) && ((empfang[dwCnt - 1] & 0x0F0) == 0xC0))
            {
                switch (empfang[dwCnt - 1] & 0x07)
                {
                    case 0x01: status = PinStatus.Transport_PIN_Zufallszahl;
                        ausgabe = "Transportschutzmethode = Zufallszahl";
                        break;
                    case 0x02: status = PinStatus.Transport_PIN_abgeleitet;
                        ausgabe = "Transportschutzmethode = abgeleitete PIN";
                        break;
                    case 0x04: status = PinStatus.Transport_PIN_Leerpin_2;
                        ausgabe = "Transportschutzmethode = Leerpin 2";
                        break;
                    case 0x05: status = PinStatus.Transport_PIN_festerWert;
                        ausgabe = "Transportschutzmethode = fester Wert";
                        break;
                    case 0x07: status = PinStatus.Transport_PIN_Leerpin_1;
                        ausgabe = "Transportschutzmethode = Leerpin 1";
                        break;
                    case 0x0F: status = PinStatus.Transport_PIN_0000;
                        ausgabe = "Transportschutzmethode mit PIN=0000";
                        break;
                }
                
                fehlbedZaehler = 0; 
                return true;
            }

            return true;
        }



        /// <summary>
        /// Auslesen eines Records des EFs mit der uebergebenen SID
        /// </summary>
        /// <param name="SID">Short-File-ID</param>
        /// <param name="recordNr">Nummer des Rekords</param>
        /// <param name="empfang">Rekordinhalt</param>
        /// <returns>bei Erfolg true, sonst false</returns>
        public bool readRecord(byte SID, byte recordNr, byte[] empfang)
        {
            uint lReturn;
            int dwCnt = 0;

            // APDU aufbauen
            byte[] READ_RECORD = {   0x00,                      // CLASS-Byte nach ISO 7614
                                     0xb2,                      // INS = Read Record
                                     recordNr,                  // P1 = Record-Nummer
                                     (byte)((SID <<3) + 0x04),  // P2 = (Short-ID auf Bits 8-5) + 0x04 => nutze Listenelement P1
                                     0x00};                     // keine Laengenangabe

            // APDU senden
            dwCnt = empfang.Length;
            lReturn = WinSCard.SCardTransmit(
                                    hCard,
                                    ref pciSend,
                                    READ_RECORD,
                                    READ_RECORD.Length,
                                    IntPtr.Zero,
                                    empfang,
                                    ref dwCnt);

            // Aufruf von SCardTransmit fehlerfrei?
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardTransmit(Read Record)");
                return false;
            }


            // Antwort_APDU "Rekord nicht gefunden" == (0x6A:0x83)
            if ((empfang[dwCnt - 2] == 0x6A) && (empfang[dwCnt - 1] == 0x83))
            {                
                return false;
            }



            // Antwort_APDU OK != (0x90:0x00) ?
            if ((empfang[dwCnt - 2] != 0x90) || (empfang[dwCnt - 1] != 0x00))
            {
                String ausgabe = "Karte antwortet bei READ RECORD mit " + String.Format("{0:X02}", empfang[dwCnt - 2]) +
                                 ":" + String.Format("{0:X02}", empfang[dwCnt - 1]) + "";
                MessageBox.Show(ausgabe);
                return false;
            }


            return true;
        }



        /// <summary>
        /// In das Wurzelverzeichnis der Karte wechseln
        /// </summary>
        /// <returns>bei Erfolg true, sonst false</returns>
        public bool select_MF()
        {
            uint lReturn;
            int dwCnt = 0;

            byte[] empfang = new byte[2];

            // APDU vorbereiten
            // ----------------
            // 0x00   CLA ==> CLASS-Byte nach ISO 7614
            // 0xA4   INS ==> Anweisung SELECT
            // 0x04   P1  ==> DF wird ueber den Namen bestimmt => wenn Data leer, dann MF
            // 0x0C,  P2  ==> keine Daten in der Antwort
            // Wenn kein Data-Feld vorhanden ist, wird MF gewaehlt
            byte[] SELECT_AID = new byte[] { 0x00, 0xA4, 0x04, 0x0C};


            // APDU an die eGK senden
            dwCnt = empfang.Length;
            lReturn = WinSCard.SCardTransmit(
                            hCard,
                            ref pciSend,
                            SELECT_AID,
                            SELECT_AID.Length,
                            IntPtr.Zero,
                            empfang,
                            ref dwCnt);

            // Aufruf von SCardTransmit fehlerfrei?
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardTransmit(SELECT_MF)");
                return false;
            }

            // Antwort_APDU OK == (0x90:0x00)
            if ((empfang[0] != 0x90) || (empfang[1] != 0x00))
                return false;

            return true;

        }



        /// <summary>
        /// Die Anwendung mit der uebergebenen AID auswaehlen
        /// </summary>
        /// <param name="AID">Application-ID der Anwendung</param>
        /// <returns>bei Erfolg true, sonst false</returns>
        public bool select_AI(byte[] AID)
        {
            uint lReturn;
            int dwCnt = 0;

            byte[] empfang = new byte[2];


            // =============================
            // Select DF ueber AID 
            // =============================   
         
            // leere APDU erzeugen
            byte[] SELECT_AID = new byte[5+AID.Length];


            // APDU vorbereiten
            // ----------------
            // 0x00   CLA ==> CLASS-Byte nach ISO 7614
            // 0xA4   INS ==> Anweisung SELECT
            // 0x04   P1  ==> DF wird ueber den Namen bestimmt
            // 0x0C,  P2  ==> keine Daten in der Antwort
            Array.Copy(new byte[] { 0x00, 0xA4, 0x04, 0x0C}, SELECT_AID, 4); 
            
            // Laenge der AID
            SELECT_AID[4] = (byte)AID.Length;

            // Ab Position 5 wird die AID erwartet
            Array.Copy( AID, 0, SELECT_AID, 5, AID.Length); 

            // APDU an die eGK senden
            dwCnt = empfang.Length;
            lReturn = WinSCard.SCardTransmit(
                            hCard,
                            ref pciSend,
                            SELECT_AID,
                            SELECT_AID.Length,
                            IntPtr.Zero,
                            empfang,
                            ref dwCnt);

            // Aufruf von SCardTransmit fehlerfrei?
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardTransmit(SELECT_HCA)");
                return false;
            }

            // Antwort_APDU OK == (0x90:0x00)
            if ((empfang[0] != 0x90) || (empfang[1] != 0x00))
            {
                // Hier keine Fehlerausgabe ueber Messagebox, da nicht
                // jede Anwendung implementiert werden muss, und somit
                // das Fehlen eines DFs kein schwerwiegender Fehler 
                // sein muss
                return false;
            }

            return true;

        }


     
        /// <summary>
        /// Auslesen eines transparenten EF, Adressierung mit Short-ID
        /// </summary>
        /// <param SID="SID">Short-File-ID</param>
        /// <param empfang="j">Inhalt der Datei</param>
        /// <returns>bei Erfolg true, sonst false</returns>
        public bool readBinary(byte SID, byte[] empfang)
        {
            uint lReturn;
            int dwCnt = 0;


            // =============================
            // transparentes EF auslesen
            // =============================
            byte[] READ_BINARY = {   0x00,                 // CLASS-Byte nach ISO 7614
                                     0xb0,                 // Instruktion READ BINARY
                                     (byte)(0x80 + SID),   // P1 = SID des EFs 
                                      0,                    // Offset
                                      0x00};                //

            dwCnt = empfang.Length;
            lReturn = WinSCard.SCardTransmit(
                                    hCard,
                                    ref pciSend,
                                    READ_BINARY,
                                    READ_BINARY.Length,
                                    IntPtr.Zero,
                                    empfang,
                                    ref dwCnt);

            // Aufruf von SCardTransmit fehlerfrei?
            if (lReturn != 0)
            {
                MessageBox.Show("Fehler bei SCardTransmit(Read Binary)");
                return false;
            }

            // Antwort_APDU OK == (0x90:0x00)
            if ((empfang[dwCnt - 2] != 0x90) || (empfang[dwCnt - 1] != 0x00))
            {

                String ausgabe = "Karte antwortet bei READ BINARY mit " + String.Format("{0:X02}", empfang[dwCnt - 2]) + 
                                 ":" + String.Format("{0:X02}", empfang[dwCnt - 1]) + "";
                MessageBox.Show(ausgabe);
                return false;
            }



            // ===============================================================
            // folgende Schritte (Restbloecke)
            // werden so lange ausgefuert, bis keine Daten mehr gesendet werden
            // ===============================================================
            byte[] READ_BINARY_REST = {     0x00, // CLASS-Byte nach ISO 7614
                                            0xb0, // Instruktion READ BINARY
                                            0x01, // P1 = Block-Nummer (wird inkrementiert)
                                            0x00, // P2 
                                            0x00}; // keine Laengenangabe

            byte i = 1;
            int adresse = dwCnt - 2;
            byte[] rest = new byte[1024];

            while (dwCnt > 2) // So lange, bis nur noch der Trailer empfangen wird
            {

                READ_BINARY_REST[2] = i++;  // Blockzaehler erhoehen

                dwCnt = rest.Length;
                lReturn = WinSCard.SCardTransmit(
                                        hCard,
                                        ref pciSend,
                                        READ_BINARY_REST,
                                        READ_BINARY_REST.Length,
                                        IntPtr.Zero,
                                        rest,
                                        ref dwCnt);

                // Aufruf von SCardTransmit fehlerfrei?
                if (lReturn != 0)
                {
                    MessageBox.Show("Fehler bei SCardTransmit(Read Binary)");
                    return false;
                }

                // gerade empfangenen Datenblock hinten anfuegen
                Array.Copy(rest, 0, empfang, adresse, dwCnt - 2);
                adresse += dwCnt - 2;
            }

            return true;
        }

    }
}
