//
//  Splash.cpp
//  ==========
//
//	Splash-Screen
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


#include "stdafx.h"
#include "resource.h"
#include "Shellapi.h"

const int VER_MAJOR = 1;
const int VER_MINOR = 0;

const TCHAR *szTitle = _T("Genetic Application Framework");
const TCHAR *szSubTitle = _T("Copyright  2013 Thorsten Radde, released under BSD-License");
const TCHAR *szHyperlink = _T("http://www.IdealSoftware.com");
const TCHAR *szDetails = _T("Reporting + PDF Library");


// =========================================================================
//								Globals
// =========================================================================
HBITMAP		m_hBitmap;					// Splash screen Bitmap
int			m_nWidth;
int			m_nHeight;

HCURSOR		m_hCursor;
HFONT		m_fontTitle;
HFONT		m_fontSubTitle;
HFONT		m_fontHyperLink;
HFONT		m_fontDetails;

RECT		m_rcTitle;
RECT		m_rcSubTitle;
RECT		m_rcHyperLink;
RECT		m_rcDetails;

bool		m_bHyperlinkHovered;
HWND		m_hWndSplash;
HWND		m_hWndParent;


#define NEW_DIB_FORMAT(lpbih) (lpbih->biSize != sizeof(BITMAPCOREHEADER))
#define BYTESPERLINE(Width, BPP) ((WORD)((((DWORD)(Width) * (DWORD)(BPP) + 31) >> 5)) << 2)

// ===========================================================================
//								DIBNumColors()
// ===========================================================================
WORD DIBNumColors(BYTE *lpbi)
{
    WORD                bits;
    LPBITMAPINFOHEADER  lpbih = (LPBITMAPINFOHEADER)lpbi;
    LPBITMAPCOREHEADER  lpbch = (LPBITMAPCOREHEADER)lpbi;

    if (NEW_DIB_FORMAT(lpbih)) {
      if (lpbih->biClrUsed != 0)
        return (WORD)lpbih->biClrUsed;
      bits = lpbih->biBitCount;
    }
    else
      bits = lpbch->bcBitCount;

    if (bits > 8) 
      return 0; /* Since biClrUsed is 0, we dont have a an optimal palette */
    else
      return (WORD)(1 << bits);
}


// ===========================================================================
//								ColorTableSize()
// ===========================================================================
DWORD ColorTableSize(BYTE *lpDib)
{
    LPBITMAPINFOHEADER lpbih = (LPBITMAPINFOHEADER)lpDib;
    
    if (NEW_DIB_FORMAT(lpbih))
    {
     #ifndef BI_BITFIELDS
        #define BI_BITFIELDS 3
     #endif
       if (((LPBITMAPINFOHEADER)(lpbih))->biCompression == BI_BITFIELDS)
          return (sizeof(DWORD) * 3) + (DIBNumColors (lpDib) * sizeof (RGBQUAD));
       else
          return (DIBNumColors (lpDib) * sizeof (RGBQUAD));
    }
    else
      return (DIBNumColors (lpDib) * sizeof (RGBTRIPLE));
}


// ===========================================================================
//                            GetDIBInfoHeaderSize
// ===========================================================================
DWORD GetDIBInfoHeaderSize(BYTE *lpDib)
{
   return ((BITMAPINFOHEADER *)lpDib)->biSize;
}


// ===========================================================================
//                               GetDIBBitsAddr
//
// Zeiger auf die eigentlichen Bitmap-Bits (Bilddaten) holen
// ===========================================================================
BYTE *GetDIBBitsAddr(BYTE *lpDib)
{
   if (lpDib == NULL)
	   return NULL;

   return lpDib + GetDIBInfoHeaderSize(lpDib) + ColorTableSize(lpDib);
}


// ===========================================================================
//                               ReadDIB()
// ===========================================================================
BYTE *ReadDIB(BYTE *source)
{
	BITMAPINFOHEADER	*pbih;
	BYTE				*lpDib;
	DWORD				dwDibSize;

	pbih = (BITMAPINFOHEADER *)source;
	int width			= pbih->biWidth;
	int height			= pbih->biHeight;
	int bit_count		= pbih->biBitCount;
	int bytes_per_line	= BYTESPERLINE(width, bit_count);

	dwDibSize = sizeof(BITMAPINFOHEADER);
	dwDibSize += bytes_per_line * height;

	lpDib = (BYTE *)malloc(dwDibSize);
	if (lpDib == NULL)
		return NULL ;

	memcpy(lpDib, source, dwDibSize);
 
	return lpDib;
}


// =========================================================================
//								LoadBitmap()
//
// Loads image from Resource
// =========================================================================
HBITMAP LoadBitmap(HINSTANCE hInstance, UINT res_id)
{
	HGLOBAL hglb;
	HBITMAP hbm = NULL;

	HRSRC hrsc = FindResource(hInstance, MAKEINTRESOURCE(res_id), RT_BITMAP);
	
	if (hrsc == NULL)
		return NULL;
	
	if ((hglb = ::LoadResource(hInstance, hrsc)) == NULL)
		return NULL;

	LPBITMAPINFOHEADER lpBitmap = (LPBITMAPINFOHEADER)LockResource(hglb);
	if (lpBitmap == NULL)
		return NULL;

	// Use ReadDIB() for RLE Bitmap
	lpBitmap = (LPBITMAPINFOHEADER)ReadDIB((BYTE *)lpBitmap);

	if (lpBitmap)
	{
		m_nWidth = (int)lpBitmap->biWidth;
		m_nHeight = (int)lpBitmap->biHeight;
		HDC hDCScreen = ::GetDC(NULL);
		hbm = ::CreateCompatibleBitmap(hDCScreen, m_nWidth, m_nHeight);
		::ReleaseDC(NULL, hDCScreen);
		HDC hDCGlyphs = CreateCompatibleDC(NULL);

		if (hbm != NULL)
		{
			HBITMAP hbmOld = (HBITMAP)::SelectObject(hDCGlyphs, hbm);

			LPBYTE lpBits = GetDIBBitsAddr((BYTE *)lpBitmap);

			StretchDIBits(hDCGlyphs, 0, 0, m_nWidth, m_nHeight, 0, 0, m_nWidth, m_nHeight,
				lpBits, (LPBITMAPINFO)lpBitmap, DIB_RGB_COLORS, SRCCOPY);
			
			SelectObject(hDCGlyphs, hbmOld);
		}

		DeleteDC(hDCGlyphs);
		free(lpBitmap);
	}

	// free copy of bitmap info struct and resource itself
	::FreeResource(hglb);

	return hbm;
}


// =========================================================================
//							DestroySplash()
// =========================================================================
void DestroySplash()
{
	ReleaseCapture();
	EnableWindow(m_hWndParent, TRUE);
	SetFocus(m_hWndParent);
	DeleteObject(m_hBitmap);
	DeleteObject(m_fontTitle);
	DeleteObject(m_fontSubTitle);
	DeleteObject(m_fontHyperLink);
}


// =========================================================================
//								Paint()
//
// Paints all necessary text into the bitmap and then blits the
// bitmap to the DC.
// =========================================================================
void Paint(HDC hDC)
{
	// paint the bitmap
	HDC hdcMem = CreateCompatibleDC(hDC);
	HBITMAP	hBitmapOld = SelectBitmap(hdcMem, m_hBitmap);
	BitBlt(hDC, 0, 0, m_nWidth, m_nHeight, hdcMem, 0, 0, SRCCOPY);
	SelectObject(hdcMem, hBitmapOld);
	DeleteDC(hdcMem);

	HFONT hOldFont = SelectFont(hDC, m_fontTitle);
	SetBkMode(hDC, TRANSPARENT);
	DrawText(hDC, szTitle, (int)_tcslen(szTitle), &m_rcTitle, DT_LEFT | DT_SINGLELINE | DT_TOP);

	SelectFont(hDC, m_fontSubTitle);
	DrawText(hDC, szSubTitle, (int)_tcslen(szSubTitle), &m_rcSubTitle, DT_LEFT | DT_SINGLELINE | DT_TOP);

	SelectFont(hDC, m_fontDetails);
	DrawText(hDC, szDetails, (int)_tcslen(szDetails), &m_rcDetails, DT_LEFT | DT_SINGLELINE | DT_TOP);

	SelectObject(hDC, m_fontHyperLink);
	SetTextColor(hDC, m_bHyperlinkHovered ? RGB (0, 128, 0) : RGB (0, 0, 255));
	DrawText(hDC, szHyperlink, (int)_tcslen(szHyperlink), &m_rcHyperLink, DT_LEFT | DT_SINGLELINE | DT_TOP);

	SelectObject(hDC, hOldFont);
}


// =========================================================================
//								SetHoverState()
// =========================================================================
void SetHoverState(BOOL state)
{
	if (state)
	{
		if (!m_bHyperlinkHovered)
			InvalidateRect(m_hWndSplash, NULL, FALSE);

		m_bHyperlinkHovered = true;
		SetCursor(m_hCursor);
	}
	else
	{
		if (m_bHyperlinkHovered)
			InvalidateRect(m_hWndSplash, NULL, FALSE);

		m_bHyperlinkHovered = false;
		SetCursor(LoadCursor(NULL, IDC_ARROW));
	}
}


// =========================================================================
//							LButtonDown()
// =========================================================================
void LButtonDown(WPARAM fwkeys, WORD x, WORD y)
{
	POINT point;

	point.x = x;
	point.y = y;

	EnableWindow(m_hWndParent, TRUE);
	PostMessage(m_hWndSplash, WM_CLOSE, 0, 0);

	if (PtInRect (&m_rcHyperLink, point))
	{
		ShellExecute (NULL, _T("open"), szHyperlink, NULL, NULL, SW_SHOWNORMAL);
	}
}


// =========================================================================
//								MouseMove()
// =========================================================================
void MouseMove(WORD x, WORD y)
{
	POINT point;

	point.x = x;
	point.y = y;

	SetHoverState(PtInRect (&m_rcHyperLink, point));
}


// =========================================================================
//								Keydown()
// =========================================================================
void Keydown()
{
	PostMessage(m_hWndSplash, WM_CLOSE, 0, 0);
}


// =========================================================================
//								SplashWndProc()
// =========================================================================
LRESULT CALLBACK SplashWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	PAINTSTRUCT ps;
	HDC hdc;

	switch (message)
	{
	case WM_PAINT:
		hdc = BeginPaint(hWnd, &ps);
		Paint(hdc);
		EndPaint(hWnd, &ps);
		break;

	case WM_LBUTTONDOWN:
		LButtonDown(wParam, GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
		break;

	case WM_MOUSEMOVE:
		MouseMove(GET_X_LPARAM(lParam), GET_Y_LPARAM(lParam));
		break;

	case WM_KEYDOWN:
	case WM_SYSCHAR:
		Keydown();
		break;

	case WM_TIMER:
		PostMessage(hWnd, WM_CLOSE, 0, 0);
		break;

	case WM_DESTROY:
		DestroySplash();
		break;

	default:
		return DefWindowProc(hWnd, message, wParam, lParam);
	}

	return 0;
}


// =========================================================================
//									Splash()
//
// Loads the Splash-Bitmap from Resource and initializes required
// GDI objects.
// Compute rectangles for text to paint.
// =========================================================================
void Splash(HWND hWndParent, HINSTANCE hInstance, bool with_timer)
{
	m_hWndParent = hWndParent;

	// Load Bitmap
	m_hBitmap = LoadBitmap(hInstance, IDB_BITMAP_DNA);

	// Create GDI Objects
	m_hCursor = LoadCursor(NULL, IDC_HAND);
	if (!m_hCursor)
		m_hCursor = LoadCursor(hInstance, MAKEINTRESOURCE(IDC_MYHAND));

	m_fontTitle = CreateFont(28, 0, 0, 0, FW_NORMAL, FALSE, FALSE, 0, ANSI_CHARSET,
		OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
		DEFAULT_PITCH | FF_SWISS, _T("Arial"));

	m_fontSubTitle = CreateFont(16, 0, 0, 0, FW_NORMAL, FALSE, FALSE, 0, ANSI_CHARSET,
		OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
		DEFAULT_PITCH | FF_SWISS, _T("Arial"));

	m_fontHyperLink = CreateFont(20, 0, 0, 0, FW_NORMAL, FALSE, TRUE, 0, ANSI_CHARSET,
		OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
		DEFAULT_PITCH | FF_SWISS, _T("Arial"));

	m_fontDetails = CreateFont(14, 0, 0, 0, FW_BOLD, FALSE, FALSE, 0, ANSI_CHARSET,
		OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, DEFAULT_QUALITY,
		DEFAULT_PITCH | FF_SWISS, _T("Arial"));

	// Set the coordinates for painting
	HDC hdcDesktop = GetDC(NULL);

	SIZE size;
	HFONT hOldFont = SelectFont(hdcDesktop, m_fontHyperLink);
	m_rcHyperLink.left = 8;
	m_rcHyperLink.top = 240;
	GetTextExtentPoint32(hdcDesktop, szHyperlink, (int)_tcslen(szHyperlink), &size);
	m_rcHyperLink.right = m_rcHyperLink.left + size.cx;
	m_rcHyperLink.bottom = m_rcHyperLink.top + size.cy;

	SelectObject(hdcDesktop, m_fontTitle);
	m_rcTitle.left = 8;
	m_rcTitle.top = 2;
	GetTextExtentPoint32(hdcDesktop, szTitle, (int)_tcslen(szTitle), &size);
	m_rcTitle.right = m_rcTitle.left + size.cx;
	m_rcTitle.bottom = m_rcTitle.top + size.cy;

	SelectObject(hdcDesktop, m_fontSubTitle);
	m_rcSubTitle.left = 8;
	m_rcSubTitle.top = m_rcTitle.bottom;
	GetTextExtentPoint32(hdcDesktop, szSubTitle, (int)_tcslen(szSubTitle), &size);
	m_rcSubTitle.right = m_rcSubTitle.left + size.cx;
	m_rcSubTitle.bottom = m_rcSubTitle.top + size.cy;

	SelectObject(hdcDesktop, m_fontDetails);
	m_rcDetails.left = 8;
	m_rcDetails.top = m_rcHyperLink.bottom + 2;
	GetTextExtentPoint32(hdcDesktop, szDetails, (int)_tcslen(szDetails), &size);
	m_rcDetails.right = m_rcDetails.left + size.cx;
	m_rcDetails.bottom = m_rcDetails.top + size.cy;

	SelectObject(hdcDesktop, hOldFont);
	ReleaseDC(NULL, hdcDesktop);
	m_bHyperlinkHovered = false;

	WNDCLASSEX wcex;
	wcex.cbSize			= sizeof(WNDCLASSEX);
	wcex.style			= CS_HREDRAW | CS_VREDRAW;
	wcex.lpfnWndProc	= SplashWndProc;
	wcex.cbClsExtra		= 0;
	wcex.cbWndExtra		= 0;
	wcex.hInstance		= hInstance;
	wcex.hIcon			= NULL;
	wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
	wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW + 1);
	wcex.lpszMenuName	= NULL;
	wcex.lpszClassName	= _T("GaSplash");
	wcex.hIconSm		= NULL;
	RegisterClassEx(&wcex);

	m_hWndSplash = CreateWindow(wcex.lpszClassName, NULL, WS_POPUP,
								CW_USEDEFAULT, 0, CW_USEDEFAULT, 0, NULL, NULL, wcex.hInstance, NULL);

	HDC hdc = GetDC(NULL);
	int width = GetDeviceCaps(hdc, HORZRES);
	int height = GetDeviceCaps(hdc, VERTRES);
	ReleaseDC(NULL, hdc);
	MoveWindow(m_hWndSplash, (width - m_nWidth) / 2, (height - m_nHeight) / 2, m_nWidth, m_nHeight, TRUE);
	EnableWindow(hWndParent, FALSE);
	ShowWindow(m_hWndSplash, TRUE);
	UpdateWindow(m_hWndSplash);
	SetFocus(m_hWndSplash);
	SetCapture(m_hWndSplash);

	if (with_timer)
		SetTimer(m_hWndSplash, 1, 3500, NULL);
}
