//
//	ga_app.h
//	========
//
//	Genetic Application Framework for Windows
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


// ================================================================================================================
//												Defines
// ================================================================================================================
#define MAX_LOADSTRING 100


// ================================================================================================================
//												Prototypes
// ================================================================================================================
void InitApplication(HINSTANCE hInstance);

HWND GaCreateWindow(WNDCLASSEX &wcex, const TCHAR *szTitle);
LRESULT CALLBACK MainWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK MeanWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
LRESULT CALLBACK PopulationWndProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam);
BOOL DispatchAllWindowMessages();


// ================================================================================================================
//												enums
// ================================================================================================================

// the currently selected algorithm
enum EAlgo
{
	enAlgoGenetic,			// genetic
	enAlgoBruteForce,		// brute force
};


// Visualization speed
enum EVisualize
{
	enVisualizeOff		= -1,
	enVisualizeFastest	= 0,
	enVisualize100ms	= 100,
	enVisualize200ms	= 200,
	enVisualize1000ms	= 1000,
};


// ================================================================================================================
//												class CTimer
//
// Class for measuring elapsed time
// ================================================================================================================
class CTimer
{
protected:
	__int64		m_nStart;
	__int64		m_nStop;
	__int64		m_nFreq;

public:
	CTimer()
	{
		m_nStart = 0;
		m_nStop  = 0;
		QueryPerformanceFrequency((LARGE_INTEGER *)&m_nFreq);
	}

	void Start()
	{
		QueryPerformanceCounter((LARGE_INTEGER *)&m_nStart);
	}

	void Stop()
	{
		QueryPerformanceCounter((LARGE_INTEGER *)&m_nStop);
	}

	// copies string with elapsed time in seconds to s
	void GetElapsedTime(TCHAR *s)
	{
		_stprintf_s(s, 128, _T("Elapsed Time: %g seconds"), ((double)(m_nStop - m_nStart) / (double)m_nFreq));
	}
};


// ================================================================================================================
//												class CGaAppVisualize
//
// Class for visualizing the genetic algorithm's state
// ================================================================================================================
template<class TGenome>
class CGaAppVisualize
{
protected:
	HWND			m_hWndMain;				// window handle of main window
	HWND			m_hWndMeanValue;		// window handle of mean value window
	HWND			m_hWndPopulation;		// window handle of population window
	list<double>	m_listMeanValues;		// history of mean values, for painting the graph
	list<double>	m_listBestFitnesses;	// history of best fitness values, for painting the graph
	double			m_dblMinValue;				// minimum value of history, for scaling the graph
	double			m_dblMaxValue;				// maximum value of history, for scaling the graph
	CPopulation<TGenome> *m_pPopulation;	// the current population of the genetic algorithm
	int				m_nDelay;				// visualization speed

public:
	CGaAppVisualize()
	{
		m_hWndMain			= NULL;
		m_hWndMeanValue		= NULL;
		m_hWndPopulation	= NULL;
		m_nDelay			= 0;

		Init();
	}

	void Init()
	{
		m_listMeanValues.clear();
		m_listBestFitnesses.clear();
		m_dblMinValue = DBL_MAX;
		m_dblMaxValue = 0;
		m_pPopulation = NULL;

		if (m_hWndMeanValue)
		{
			InvalidateRect(m_hWndMeanValue, NULL, TRUE);
			UpdateWindow(m_hWndMeanValue);
		}

		if (m_hWndPopulation)
		{
			InvalidateRect(m_hWndPopulation, NULL, FALSE);
			UpdateWindow(m_hWndPopulation);
		}
	}

	void SetWndMain(HWND val) { m_hWndMain = val; }
	void SetWndMeanValue(HWND val) { m_hWndMeanValue = val; }
	void SetWndPopulation(HWND val) { m_hWndPopulation = val; }
	void SetDelay(int msec) { m_nDelay = msec; }

	list<double>	&GetMeanValues() { return m_listMeanValues; }
	list<double>	&GetBestFitnesses() { return m_listBestFitnesses; }
	double			GetMinValue() const { return m_dblMinValue; }
	double			GetMaxValue() const { return m_dblMaxValue; }

	CPopulation<TGenome>	*GetPopulation() { return m_pPopulation; }

	// This is the callback for the genetic algorithm.
	// It is called for each new generation.
	void Visualize(const TGenome &solution, bool solution_changed, int generation, double mean_value, CPopulation<TGenome> *population)
	{
		m_listMeanValues.push_back(mean_value);
		m_listBestFitnesses.push_back(solution.GetFitness());

		if (mean_value < m_dblMinValue)
			m_dblMinValue = mean_value;

		if (mean_value > m_dblMaxValue)
			m_dblMaxValue = mean_value;

		if (solution.GetFitness() < m_dblMinValue)
			m_dblMinValue = solution.GetFitness();

		if (solution.GetFitness() > m_dblMaxValue)
			m_dblMaxValue = solution.GetFitness();

		if (m_hWndMeanValue)
		{
			InvalidateRect(m_hWndMeanValue, NULL, TRUE);
			UpdateWindow(m_hWndMeanValue);
		}

		m_pPopulation = population;
		if (m_hWndPopulation)
		{
			InvalidateRect(m_hWndPopulation, NULL, FALSE);
			UpdateWindow(m_hWndPopulation);
		}

		if (solution_changed && m_hWndMain)
		{
			InvalidateRect(m_hWndMain, NULL, TRUE);
			UpdateWindow(m_hWndMain);
		}

		// Keep the Windows Message Queue alive
		if (m_nDelay == 0)
		{
			DispatchAllWindowMessages();		// keep Windows message queue alive
		}
		else
		{
			int d = 0;
			while (d < m_nDelay)
			{
				DispatchAllWindowMessages();	// keep Windows message queue alive
				Sleep(10);
				d += 10;
			}
		}
	}
};


// ================================================================================================================
//												class CGaAppInterface
//
// We have the problem that window procedures (WNDPROC) can not be class members (static members do not help).
// So window procedures have no access to instances of classes derived from CGaApplication.
// Therefore we provide this interface class.
//
// The pointer to the instance of this interface class is stored in the global variable
// "CGaAppInterface *gpGaAppInterface".
// ================================================================================================================
class CGaAppInterface
{
public:
	virtual void MainWndCreate() {}
	virtual bool MainWndCommand(int wmId, int wmEvent) { return false; }
	virtual void MainWndPaint(HWND hWnd, HDC hdc) {}
	virtual void MainWndDestroy() {}

	virtual void MeanWndPaint(HWND hWnd, HDC hdc) {}
	virtual void PopWndPaint(HWND hWnd, HDC hdc) {}
};

extern CGaAppInterface *gpGaAppInterface;


// ================================================================================================================
//												class CGaApplication
//
// Base class for genetic applications
// ================================================================================================================
template<class TGenome, class TVisualization = CVisualization<TGenome>, class TSelection = CTournamentSelection<TGenome>>
class CGaApplication
{
protected:		// members
	HINSTANCE	m_hInstance;						// current instance
	HWND		m_hWndMain;							// window handle of main window
	HWND		m_hWndMeanValue;					// window handle of mean value window
	HWND		m_hWndPopulation;					// window handle of population window
	TCHAR		m_szTitle[MAX_LOADSTRING];			// the main window title bar text
	EAlgo		m_enCurrentAlgo;					// the current algorithm running (for painting the main window)
	EVisualize	m_enVisualization;					// visualization speed
	bool		m_bBruteForceDone;					// true, if Brute-Force algorithm has finished
	bool		m_bCancel;							// true, if Brute-Force algorithm shall be cancelled
	CTimer		m_Timer;							// Timer

	// Instance of the genetic algorithm for the problem
	CGeneticAlgo<TGenome, TVisualization, TSelection> m_GeneticAlgo;

public:
	CGaApplication()
	{
		m_enCurrentAlgo		= enAlgoGenetic;		// the current algorithm running (for painting the main window)
		m_enVisualization	= enVisualizeFastest;	// visualization speed
	}

	virtual void Init(HINSTANCE hInstance)
	{
		m_hInstance = hInstance;

		WNDCLASSEX wcex;
		wcex.cbSize			= sizeof(WNDCLASSEX);
		wcex.style			= CS_HREDRAW | CS_VREDRAW;
		wcex.lpfnWndProc	= MainWndProc;
		wcex.cbClsExtra		= 0;
		wcex.cbWndExtra		= 0;
		wcex.hInstance		= hInstance;
		wcex.hIcon			= LoadIcon(hInstance, MAKEINTRESOURCE(IDI_APPLICATION));
		wcex.hCursor		= LoadCursor(NULL, IDC_ARROW);
		wcex.hbrBackground	= (HBRUSH)(COLOR_WINDOW + 1);
		wcex.lpszMenuName	= MAKEINTRESOURCE(IDC_MENU);
		wcex.lpszClassName	= _T("GaMainWindow");
		wcex.hIconSm		= LoadIcon(wcex.hInstance, MAKEINTRESOURCE(IDI_SMALL));
		RegisterClassEx(&wcex);

		LoadString(m_hInstance, IDS_APP_TITLE, m_szTitle, MAX_LOADSTRING);
		m_hWndMain = GaCreateWindow(wcex, m_szTitle);

		wcex.style			= CS_HREDRAW | CS_VREDRAW | CS_NOCLOSE;
		wcex.lpfnWndProc	= MeanWndProc;
		wcex.lpszClassName	= _T("GaMeanWindow");
		wcex.lpszMenuName	= NULL;
		RegisterClassEx(&wcex);
		m_hWndMeanValue = GaCreateWindow(wcex, _T("Mean Values"));

		wcex.lpfnWndProc	= PopulationWndProc;
		wcex.lpszClassName	= _T("GaPopulationWindow");
		wcex.hbrBackground	= (HBRUSH)GetStockObject(NULL_BRUSH);
		RegisterClassEx(&wcex);
		m_hWndPopulation = GaCreateWindow(wcex, _T("Population"));

		HMENU hMenu = GetMenu(m_hWndMain);
		CheckMenuRadioItem(hMenu, ID_VISUALIZE_OFF, ID_VISUALIZE_1000MSDELAY, ID_VISUALIZE_FASTEST, MF_BYCOMMAND);

		// get size of the screen to arrange windows
		HDC hdc = GetDC(NULL);
		int width = GetDeviceCaps(hdc, HORZRES);
		int height = GetDeviceCaps(hdc, VERTRES);
		ReleaseDC(NULL, hdc);

		// arrange windows
		MoveWindow(m_hWndMain, 0, 0, width * 2 / 3, height, TRUE);
		MoveWindow(m_hWndMeanValue, width * 2 / 3, 0, width / 3, height / 2, TRUE);
		MoveWindow(m_hWndPopulation, width * 2 / 3, height / 2, width / 3, height / 2, TRUE);
		SetFocus(m_hWndMain);

		m_GeneticAlgo.GetVisualization().SetWndMain(m_hWndMain);
		m_GeneticAlgo.GetVisualization().SetWndMeanValue(m_hWndMeanValue);
		m_GeneticAlgo.GetVisualization().SetWndPopulation(m_hWndPopulation);

		Splash(m_hWndMain, hInstance, true);
	}


	// Solves the problem using the Genetic Algorithm.
	//
	// In derived classes, override this function and set parameters, like:
	//		m_GeneticAlgo.SetPopulationSize(512)
	//		m_GeneticAlgo.SetCrossoverProbability(45);
	//		m_GeneticAlgo.SetMutationProbability(50);
	//		m_GeneticAlgo.SetElitismPercentage(10);
	//		m_GeneticAlgo.SetStableGenerations(300);
	//		m_GeneticAlgo.SetMaxGenerations(1000);
	//
	// Then call this base class method.
	virtual void CalcGenetic()
	{
		m_enCurrentAlgo = enAlgoGenetic;
		if (m_enVisualization == enVisualizeOff)
			m_GeneticAlgo.SetEnableVisualization(false);
		else
		{
			m_GeneticAlgo.SetEnableVisualization(true);
			m_GeneticAlgo.SetVisualizationDelay((int)m_enVisualization);
		}
		m_Timer.Start();
		m_GeneticAlgo.Compute();
		m_Timer.Stop();
		InvalidateRect(m_hWndMain, NULL, TRUE);
		UpdateWindow(m_hWndMain);
	}


	// Override this method to implement brute force code.
	// NOTE: Call DispatchAllWindowMessages() and check m_bCancel periodically to keep the application alive.
	virtual void DoCalcBruteForce() {}


	// This method calls DoCalcBruteForce()
	void CalcBruteForce()
	{
		m_enCurrentAlgo = enAlgoBruteForce;
		m_bBruteForceDone = false;
		m_bCancel = false;
		m_Timer.Start();
		DoCalcBruteForce();
		m_Timer.Stop();
		m_bBruteForceDone = true;
		InvalidateRect(m_hWndMain, NULL, TRUE);
		UpdateWindow(m_hWndMain);
	}


	// ------------------------------- User Interface Callbacks -------------------------------
	virtual void MainWndCreate() {}
	virtual void MainWndPaint(HWND hWnd, HDC hdc) {}
	virtual void PopWndPaint(HWND hWnd, HDC hdc) {}

	// paint the mean value window
	virtual void MeanWndPaint(HWND hWnd, HDC hdc)
	{
		if (m_enVisualization == enVisualizeOff)
			return;

		list<double> &values = m_GeneticAlgo.GetVisualization().GetMeanValues();

		RECT rcClient;
		GetClientRect(hWnd, &rcClient);
		double client_dx = rcClient.right - rcClient.left;
		double client_dy = rcClient.bottom - rcClient.top;
		double x_scale = client_dx / values.size();

		double min = m_GeneticAlgo.GetVisualization().GetMinValue();
		double value_delta = m_GeneticAlgo.GetVisualization().GetMaxValue() - min;
		double y_scale = client_dy / value_delta;

		HPEN hpen, hpenOld;
		hpen = CreatePen(PS_SOLID, 1, RGB(0, 0, 255));
		hpenOld = (HPEN)SelectObject(hdc, hpen);
		MoveToEx(hdc, 0, 0, NULL);
		double x = 0;
		for (list<double>::iterator it = values.begin(); it != values.end(); it++)
		{
			LineTo(hdc, (int)x, rcClient.bottom - (int)((*it - min) * y_scale));
			x += x_scale;
		}
		SelectObject(hdc, hpenOld);
		DeleteObject(hpen);

		list<double> &v = m_GeneticAlgo.GetVisualization().GetBestFitnesses();
		hpen = CreatePen(PS_SOLID, 1, RGB(0, 128, 0));
		hpenOld = (HPEN)SelectObject(hdc, hpen);
		MoveToEx(hdc, 0, 0, NULL);
		x = 0;
		for (list<double>::iterator it = v.begin(); it != v.end(); it++)
		{
			LineTo(hdc, (int)x, rcClient.bottom - (int)((*it - min) * y_scale) + 3);
			x += x_scale;
		}

		SelectObject(hdc, hpenOld);
		DeleteObject(hpen);
	}


	virtual bool MainWndCommand(int wmId, int wmEvent)
	{
		// Parse the menu selections:
		HMENU hMenu = GetMenu(m_hWndMain);

		switch (wmId)
		{
		case IDM_ABOUT:
			Splash(m_hWndMain, m_hInstance);
			return true;

		case IDM_EXIT:
			DestroyWindow(m_hWndMain);
			return true;

		case ID_FILE_GENETIC:
		    EnableMenuItem(hMenu, ID_FILE_GENETIC, MF_GRAYED);
		    EnableMenuItem(hMenu, ID_FILE_BRUTEFORCE, MF_GRAYED);
			CalcGenetic();
		    EnableMenuItem(hMenu, ID_FILE_GENETIC, MF_ENABLED);
		    EnableMenuItem(hMenu, ID_FILE_BRUTEFORCE, MF_ENABLED);
			return true;

		case ID_FILE_BRUTEFORCE:
		    EnableMenuItem(hMenu, ID_FILE_GENETIC, MF_GRAYED);
		    EnableMenuItem(hMenu, ID_FILE_BRUTEFORCE, MF_GRAYED);
			CalcBruteForce();
		    EnableMenuItem(hMenu, ID_FILE_GENETIC, MF_ENABLED);
		    EnableMenuItem(hMenu, ID_FILE_BRUTEFORCE, MF_ENABLED);
			return true;

		case ID_VISUALIZE_OFF:
			CheckMenuRadioItem(hMenu, ID_VISUALIZE_OFF, ID_VISUALIZE_1000MSDELAY, ID_VISUALIZE_OFF, MF_BYCOMMAND);
			m_enVisualization = enVisualizeOff;
			m_GeneticAlgo.SetEnableVisualization(false);
			return true;

		case ID_VISUALIZE_FASTEST:
			CheckMenuRadioItem(hMenu, ID_VISUALIZE_OFF, ID_VISUALIZE_1000MSDELAY, ID_VISUALIZE_FASTEST, MF_BYCOMMAND);
			m_enVisualization = enVisualizeFastest;
			m_GeneticAlgo.SetEnableVisualization(true);
			m_GeneticAlgo.SetVisualizationDelay((int)m_enVisualization);
			return true;

		case ID_VISUALIZE_100MSDELAY:
			CheckMenuRadioItem(hMenu, ID_VISUALIZE_OFF, ID_VISUALIZE_1000MSDELAY, ID_VISUALIZE_100MSDELAY, MF_BYCOMMAND);
			m_enVisualization = enVisualize100ms;
			m_GeneticAlgo.SetEnableVisualization(true);
			m_GeneticAlgo.SetVisualizationDelay((int)m_enVisualization);
			return true;

		case ID_VISUALIZE_200MSDELAY:
			CheckMenuRadioItem(hMenu, ID_VISUALIZE_OFF, ID_VISUALIZE_1000MSDELAY, ID_VISUALIZE_200MSDELAY, MF_BYCOMMAND);
			m_enVisualization = enVisualize200ms;
			m_GeneticAlgo.SetEnableVisualization(true);
			m_GeneticAlgo.SetVisualizationDelay((int)m_enVisualization);
			return true;

		case ID_VISUALIZE_1000MSDELAY:
			CheckMenuRadioItem(hMenu, ID_VISUALIZE_OFF, ID_VISUALIZE_1000MSDELAY, ID_VISUALIZE_1000MSDELAY, MF_BYCOMMAND);
			m_enVisualization = enVisualize1000ms;
			m_GeneticAlgo.SetEnableVisualization(true);
			m_GeneticAlgo.SetVisualizationDelay((int)m_enVisualization);
			return true;
		}

		return false;
	}


	virtual void MainWndDestroy()
	{
		m_GeneticAlgo.SetCancel();
		m_bCancel = true;
	}
};
