//
//	rucksack.h
//	==========
//
//	Software License Agreement (BSD License)
//	----------------------------------------
//	Copyright (c) 2013 Thorsten Radde (thorstenr@idealsoftware.com). All rights reserved.
//	Source code: www.IdealSoftware.com
//
//	Redistribution and use in source and binary forms, with or without modification,
//	are permitted provided that the following conditions are met:
//
//	* Redistributions of source code must retain the above copyright notice, this 
//	  list of conditions and the following disclaimer.
//
//	* Redistributions in binary form must reproduce the above copyright notice, this
//	  list of conditions and the following disclaimer in the documentation and/or
//	  other materials provided with the distribution.
//
//	* Neither the names Thorsten Radde or IDEAL Software GmbH, nor the names of contributors
//	  may be used to endorse or promote products derived from this software without
//	  specific prior written permission of Thorsten Radde.
//
//	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
//	ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
//	WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
//	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
//	ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
//	(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
//	LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
//	ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
//	(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
//	SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.


// Size (Volume) of the Rucksack
const int nMaxSize = 1000;

// Number of bits in a genome (32 or 64)
#define NUM_BITS	32

#if NUM_BITS == 32
	typedef unsigned int TGENOME;
#else
	#ifdef _WINDOWS_
		typedef unsigned __int64 TGENOME;		// Windows platform, Visual C++
	#else
		typedef unsigned int64_t TGENOME;		// Non-Windows platform, GCC
	#endif
#endif


// A single item for the Rucksack
class CItem
{
public:
	int m_nValue;		// value of the item
	int	m_nSize;		// size of the item
};


// Array of available items
CItem Items[] =
{
#if NUM_BITS == 32
	// 32 items = 2 ^ 32 combinations
	189,   52,
	 25,   12,
	133,   12,
	422,  167,
	 29,  150,
	523,  280,
	 33,   19,
	127,   35,
	340,  125,
	 43,   29,
	 47,   63,
	 52,   17,
	 97,   33,
	101,   45,
	108,   53,
	  7,   33,
	550,  180,
	116,   88,
	167,  140,
	 47,   31,
	212,   94,
	 13,    5,
	287,   57,
	320,   60,
	 27,  350,
	400,   99,
	179,  220,
	487,   89,
	 39,   22,
	199,   12,
	 39,   23,
	570,  143,
#else
	// 64 items = 2 ^ 64 combinations
	411,	97,
	747,	13,
	921,	49,
	954,	49,
	874,	99,
	941,	86,
	127,	60,
	871,	36,
	974,	96,
	564,	82,
	577,	48,
	364,	59,
	839,	58,
	83,		41,
	751,	43,
	194,	6,
	302,	83,
	320,	15,
	328,	25,
	322,	37,
	815,	65,
	544,	74,
	576,	26,
	660,	4,
	192,	32,
	350,	91,
	756,	20,
	359,	24,
	919,	65,
	473,	81,
	341,	78,
	752,	18,
	391,	41,
	649,	22,
	748,	67,
	932,	60,
	322,	54,
	790,	76,
	7,		29,
	915,	22,
	944,	9,
	520,	78,
	716,	69,
	587,	69,
	223,	88,
	385,	95,
	643,	49,
	143,	37,
	924,	83,
	836,	38,
	574,	35,
	141,	16,
	516,	81,
	66,		99,
	412,	88,
	257,	64,
	338,	15,
	320,	69,
	563,	68,
	149,	94,
	686,	32,
	91,		70,
	775,	22,
	61,		45,
#endif
};


// pre-computed bitmasks for fast crossover
// each entry in the array is a cross-over position
TGENOME Bitmasks[NUM_BITS];

void InitBitmasks()
{
	for (int n = 0; n < NUM_BITS; n++)
	{
		unsigned int mask = 0;
		unsigned int bit = 1;
		for (int i = 0; i <= n; i++)
		{
			mask |= bit;
			bit <<= 1;
		}

		Bitmasks[n] = mask;
	}
}


// Definition of the genome for the Rucksack-Problem
class CRsGenome
{
public:
	typedef int TFitnessValueType;

protected:
	TGENOME				m_nGenome;		// the genome
	TFitnessValueType	m_nFitness;		// fitness value

	// Invert a bit
	void InvertBit(int pos)
	{
		TGENOME mask = (TGENOME)1 << pos;

		if (m_nGenome & mask)
			m_nGenome &= ~mask;		// from 1 to 0
		else
			m_nGenome |= mask;		// from 0 to 1
	}

	void MakeValid()
	{
		// if the solution is invalid, make it valid
		while (m_nGenome > 0 && ComputeFitness() == 0)
		{
			bool flipped = false;
			while (!flipped)
			{
				int n = gen_rand32() % NUM_BITS;
				TGENOME mask = (TGENOME)1 << n;
				if (m_nGenome & mask)
				{
					m_nGenome &= ~mask;		// zero bit
					flipped = true;
				}
			}
		}
	}

public:
	static const size_t	m_nNumChromosomes = NUM_BITS;

	CRsGenome()
	{
		Reset();
	}

	void Reset()
	{
		m_nGenome	= 0;
		m_nFitness	= 0;
	}

	TGENOME			GetGenome() const { return m_nGenome; }
	void			SetGenome(const TGENOME genome) { m_nGenome = genome; }

	TFitnessValueType	GetFitness() const { return m_nFitness; }
	void				SetFitness(const TFitnessValueType val) { m_nFitness = val; }

	// initializes the genome with random chromosomes
	void Randomize()
	{
		m_nGenome = gen_rand32();

	#if NUM_BITS == 64
		m_nGenome |= (TGENOME)gen_rand32() << 32;
	#endif

		MakeValid();
	}

	// computes the fitness-value, the bigger the better
	int ComputeFitness()
	{
		m_nFitness	= 0;
		int size	= 0;
		TGENOME mask = 1;

		for (int i = 0; i < NUM_BITS; i++)
		{
			if (m_nGenome & mask)
			{
				// we use the sum of all item's values as fitness value
				m_nFitness	+= Items[i].m_nValue;
				size		+= Items[i].m_nSize;

				// but we may not exceed the maximum volume of the rucksack
				if (size > nMaxSize)
				{
					// limit exceeded, invalid solution
					return 0;
				}
			}

			mask <<= 1;
		}

		return m_nFitness;
	}

	// Crossover
	void Crossover(const CRsGenome &father, const CRsGenome &mother)
	{
		// compute crossover point, 1 <= n <= NUMBITS - 1
		int n = gen_rand32() % (NUM_BITS - 1) + 1;

		m_nGenome  = father.GetGenome() & Bitmasks[n];
		m_nGenome |= mother.GetGenome() & ~Bitmasks[n];
		MakeValid();
	}

	// Mutation
	void Mutate()
	{
		InvertBit(gen_rand32() % NUM_BITS);
	}
};
