//
//  SKTView.m
//  SceneKitTest
//
//  Created by Richard Kurz on 05.02.13.
//  Copyright (c) 2013 Nobody. No rights reserved.
//

#import "SKTView.h"


static inline CGFloat toRadians(CGFloat degrees) {return degrees * (M_PI / 180.0);}

static CATransform3D lookAt(SCNVector3 goal, CGFloat distance, CGFloat azimuthDegree, CGFloat altitudeDegree)
{
  CGFloat azimuth = toRadians(azimuthDegree);
  CGFloat altitude = toRadians(altitudeDegree);
  
  CATransform3D rotate = CATransform3DConcat(CATransform3DMakeRotation(altitude, -1.0, 0.0, 0.0),
                                             CATransform3DMakeRotation(azimuth, 0.0, 1.0, 0.0));

  CATransform3D translate = CATransform3DMakeTranslation(goal.x + distance * cos(altitude) * sin(azimuth),
                                                         goal.y + distance * sin(altitude),
                                                         goal.z + distance * cos(altitude) * cos(azimuth));
  
  return CATransform3DConcat(rotate, translate);
}


@implementation SKTView


- (void)awakeFromNib
{
  self.backgroundColor = [NSColor cyanColor];
  self.autoenablesDefaultLighting = YES;
  self.allowsCameraControl = YES;
  
  SCNNode* node;
  SCNScene* szenenGraph = [SCNScene scene];

  self.scene = szenenGraph;

  [szenenGraph.rootNode addChildNode:[self createAxisNode]];

  SCNCamera* ersteKamera = [SCNCamera camera];
  ersteKamera.zFar = 1000.0;
  ersteKamera.xFov = 53.0; // ~36mm
  ersteKamera.yFov = 42.0;
  node = [SCNNode node];
  node.camera = ersteKamera;
  node.name = @"Nahaufnahme";
  node.transform = lookAt(SCNVector3Make(0.0, 0.0, 0.0), 35.0, 45.0, 20.0);
  [szenenGraph.rootNode addChildNode:node];

  SCNCamera* zweiteKamera = [SCNCamera camera];
  zweiteKamera.zFar = 1000.0;
  zweiteKamera.xFov = 53.0; // ~36mm
  zweiteKamera.yFov = 42.0;
  node = [SCNNode node];
  node.camera = zweiteKamera;
  node.name = @"Überblick";
  node.transform = lookAt(SCNVector3Make(0.0, 0.0, 0.0), 55.0, 45.0, 20.0);
  [szenenGraph.rootNode addChildNode:node];

  SCNLight* ambientLight = [SCNLight light];
  ambientLight.type = SCNLightTypeAmbient;
  ambientLight.color = [NSColor colorWithDeviceWhite:0.1 alpha:1.0];
  node = [SCNNode node];
  node.light = ambientLight;
  [szenenGraph.rootNode addChildNode:node];

  SCNLight* omniLight = [SCNLight light];
  omniLight.type = SCNLightTypeOmni;
  omniLight.color = [NSColor colorWithDeviceWhite:0.8 alpha:1.0];
  node = [SCNNode node];
  node.name =@"Glühbirne";
  node.light = omniLight;
  node.position = SCNVector3Make(-30, 70, 60);
  [szenenGraph.rootNode addChildNode:node];
  
  SCNLight* spotLight = [SCNLight light];
  spotLight = [SCNLight light];
  spotLight.type = SCNLightTypeSpot;
  spotLight.color = [NSColor blackColor];
  spotLight.castsShadow = YES;
  spotLight.shadowColor = [NSColor blackColor];
  node = [SCNNode node];
  node.name = @"Spotlicht";
  node.light = spotLight;
  node.transform = lookAt(SCNVector3Make(0.0, 0.0, 0.0), 40, 10, 70);
  [node addChildNode:[self createAxisNode]];
  [szenenGraph.rootNode addChildNode:node];
  
  SCNFloor* bodenGeometry = [SCNFloor floor];
  bodenGeometry.firstMaterial.diffuse.contents = [NSImage imageNamed: @"floor1-256"];
  node = [SCNNode nodeWithGeometry: bodenGeometry];
  node.name = @"boden";
  node.renderingOrder = -99;
  node.position = SCNVector3Make(0, -10, 0);
  [szenenGraph.rootNode addChildNode: node];

  SCNMaterial* ringMaterial = [SCNMaterial material];
  ringMaterial.diffuse.contents = [NSImage imageNamed: @"wood1-512"];
  ringMaterial.specular.contents = [NSColor whiteColor];
  ringMaterial.shininess = 1.0;
  
  SCNTorus* ringGeometry = [SCNTorus torusWithRingRadius:7 pipeRadius:2.5];
  ringGeometry.firstMaterial = ringMaterial,
  node = [SCNNode nodeWithGeometry:ringGeometry];
  node.name = @"Ring";
  [szenenGraph.rootNode addChildNode:node];
    
  NSURL* loadURL = [[NSBundle mainBundle] URLForResource:@"teekanne" withExtension:@"dae"];
  SCNScene* loadedScene = [SCNScene sceneWithURL:loadURL options:nil error:nil];
  node = [loadedScene.rootNode childNodeWithName:@"teekanne" recursively:YES];
  node.opacity = 0.0;
  [szenenGraph.rootNode addChildNode:node];
  
  node = [SCNNode node];
  node.opacity = 0.0;
  node.name = @"Banner";
  [szenenGraph.rootNode addChildNode:node];

  SCNNode* backgroundNode = [SCNNode nodeWithGeometry:[self createTriangleStripGeometry]];
  backgroundNode.geometry.firstMaterial.diffuse.contents = [NSColor yellowColor];
  backgroundNode.name = @"background";
  backgroundNode.scale = SCNVector3Make(10.0, 2.0, 1.0);
  [node addChildNode:backgroundNode];

  SCNText* textGeometry = [SCNText textWithString: @"Game Over" extrusionDepth: 10.0];
  textGeometry.font = [NSFont fontWithName: @"Arial Black" size: 30.0];
  textGeometry.chamferRadius = 5.0;
  textGeometry.firstMaterial.diffuse.contents = [NSColor redColor];
  textGeometry.firstMaterial.specular.contents = [NSColor whiteColor];
  textGeometry.firstMaterial.shininess = 1.0;
  SCNNode* textNode = [SCNNode nodeWithGeometry: textGeometry];
  textNode.name = @"game over";
  textNode.pivot = CATransform3DConcat(CATransform3DMakeScale(10.0, 10.0, 10.0),
                                       CATransform3DMakeTranslation(90.0, 20.0, 0.0));
  [node addChildNode:textNode];

  [SCNTransaction begin];
  [SCNTransaction setAnimationDuration: 2.25];
  
  node = [self.scene.rootNode childNodeWithName:@"Ring" recursively:YES];
  node.rotation = SCNVector4Make(1, 0, 0, M_PI);
  
  [SCNTransaction setCompletionBlock:^{
    [self animateLight];
  }];
  [SCNTransaction commit];
}


- (void)keyDown:(NSEvent*)theEvent
{
  [SCNTransaction begin];
  [SCNTransaction setAnimationDuration:1.0];
  
  self.pointOfView = [self.scene.rootNode childNodeWithName:@"Überblick" recursively:YES];
  
  [SCNTransaction commit];
}


- (void)mouseDown:(NSEvent*)theEvent
{
  NSArray* hitArray = [self hitTest:[self convertPoint:[theEvent locationInWindow] fromView:nil] options:nil];
  SCNHitTestResult* hitResult = nil;
  
  for(SCNHitTestResult* result in hitArray)
  {
    BOOL ishidden = NO;
    
    for (SCNNode* node = result.node; node; node = node.parentNode)
    {
      if (node.hidden)
      {
        ishidden = YES;
        break;
      }
    }
    if (!ishidden)
    {
      hitResult = result;
      break;
    }
  }
  
  if (hitResult)
  {
    CABasicAnimation* scaleAnimation = [CABasicAnimation animationWithKeyPath:@"scale"];
    scaleAnimation.autoreverses = YES;
    SCNVector3 scale = hitResult.node.scale;
    scaleAnimation.toValue = [NSValue valueWithSCNVector3:SCNVector3Make(scale.x * 1.2, scale.y * 1.2, scale.z * 1.2)];
    
    CABasicAnimation* glowAnimation = [CABasicAnimation animationWithKeyPath:@"contents"];
    glowAnimation.autoreverses = YES;
    glowAnimation.toValue = [NSColor greenColor];

    NSUInteger materialIndex = hitResult.geometryIndex % hitResult.node.geometry.materials.count;
    SCNMaterial* material = [hitResult.node.geometry.materials objectAtIndex:materialIndex];
    SCNMaterial* glowMaterial = [material copy];
    
    [SCNTransaction begin];
    [SCNTransaction setAnimationDuration: 0.25];
    
    [hitResult.node addAnimation:scaleAnimation forKey:@"scale"];
    [hitResult.node.geometry replaceMaterialAtIndex:materialIndex withMaterial: glowMaterial];
    [glowMaterial.emission addAnimation:glowAnimation forKey:@"glow"];
   
    [SCNTransaction setCompletionBlock:^{
      [hitResult.node.geometry replaceMaterialAtIndex:materialIndex withMaterial: material];
    }];
    [SCNTransaction commit];
  }
  
  [super mouseDown:theEvent];
}


- (SCNNode*)createAxisNode
{
  static SCNGeometry* axisGeometry;
  static dispatch_once_t once;
  
  dispatch_once(&once, ^{
    
    NSArray* sources = @[
      [SCNGeometrySource geometrySourceWithVertices: (SCNVector3[]){
        {.x =  0.0, .y =  0.0, .z =  0.0},
        {.x = 10.0, .y =  0.0, .z =  0.0},
        {.x =  0.0, .y = 10.0, .z =  0.0},
        {.x =  0.0, .y =  0.0, .z = 10.0}
      } count:4]];
    
    NSArray* elements = @[
      [SCNGeometryElement geometryElementWithData:[NSData dataWithBytes:(short[]){0, 1}
                                                                 length:sizeof(short[2])]
                                    primitiveType:SCNGeometryPrimitiveTypeLine
                                   primitiveCount:1
                                    bytesPerIndex:sizeof(short)],
      [SCNGeometryElement geometryElementWithData:[NSData dataWithBytes:(short[]){0, 2}
                                                                 length:sizeof(short[2])]
                                    primitiveType:SCNGeometryPrimitiveTypeLine
                                   primitiveCount:1
                                    bytesPerIndex:sizeof(short)],
      [SCNGeometryElement geometryElementWithData:[NSData dataWithBytes:(short[]){0, 3}
                                                                 length:sizeof(short[2])]
                                    primitiveType:SCNGeometryPrimitiveTypeLine
                                   primitiveCount:1
                                    bytesPerIndex:sizeof(short)]];
    
    axisGeometry = [SCNGeometry geometryWithSources:sources elements:elements];
    
    SCNMaterial* xAxisMaterial = [SCNMaterial material];
    xAxisMaterial.lightingModelName = SCNLightingModelConstant;
    xAxisMaterial.diffuse.contents = [NSColor redColor];
    
    SCNMaterial* yAxisMaterial = [SCNMaterial material];
    yAxisMaterial.lightingModelName = SCNLightingModelConstant;
    yAxisMaterial.diffuse.contents = [NSColor greenColor];
    
    SCNMaterial* zAxisMaterial = [SCNMaterial material];
    zAxisMaterial.lightingModelName = SCNLightingModelConstant;
    zAxisMaterial.diffuse.contents = [NSColor blueColor];
    
    axisGeometry.materials = @[xAxisMaterial, yAxisMaterial, zAxisMaterial];
  });
  
  return [SCNNode nodeWithGeometry: axisGeometry];
}


- (SCNGeometry*)createTriangleStripGeometry
{
  NSArray* sources = @[
    [SCNGeometrySource geometrySourceWithVertices: (SCNVector3[]){
      {.x = -1.0, .y =  1.0, .z = 0.0},
      {.x = -1.0, .y = -1.0, .z = 0.0},
      {.x =  1.0, .y =  1.0, .z = 0.0},
      {.x =  1.0, .y = -1.0, .z = 0.0}
    } count:4],
    [SCNGeometrySource geometrySourceWithNormals:(SCNVector3[]){
      {.x = 0.0, .y = 0.0, .z = 1.0},
      {.x = 0.0, .y = 0.0, .z = 1.0},
      {.x = 0.0, .y = 0.0, .z = 1.0},
      {.x = 0.0, .y = 0.0, .z = 1.0}
    } count:4],
    [SCNGeometrySource geometrySourceWithTextureCoordinates:(CGPoint[]){
      {.x = 0.0, .y = 0.0},
      {.x = 0.0, .y = 1.0},
      {.x = 1.0, .y = 0.0},
      {.x = 1.0, .y = 1.0}
    } count:4]];
  
  NSData* triangleStripData = [NSData dataWithBytes:(short[]){0, 1, 2, 3}
                                             length:sizeof(short[4])];
  
  NSArray* elements = @[
   [SCNGeometryElement geometryElementWithData:triangleStripData
                                 primitiveType:SCNGeometryPrimitiveTypeTriangleStrip
                                primitiveCount:2
                                 bytesPerIndex:sizeof(short)]];
  
  SCNGeometry* geometry = [SCNGeometry geometryWithSources:sources elements:elements];
  
  geometry.firstMaterial = [SCNMaterial material];
  geometry.firstMaterial.doubleSided = YES;
  
  return geometry;
}


- (void)animateLight
{
  [SCNTransaction begin];
  [SCNTransaction setAnimationDuration: 2.25];
  
  SCNNode* node = [self.scene.rootNode childNodeWithName:@"Spotlicht" recursively:YES];
  node.light.color = [NSColor whiteColor];
  
  node = [self.scene.rootNode childNodeWithName:@"Glühbirne" recursively:YES];
  node.light.color = [NSColor colorWithDeviceWhite:0.4 alpha:1.0];
  
  [SCNTransaction setCompletionBlock:^{
    [self animatePosition];
  }];
  [SCNTransaction commit];
}


- (void)animatePosition
{
  [SCNTransaction begin];
  [SCNTransaction setAnimationDuration: 2.25];

  self.pointOfView = [self.scene.rootNode childNodeWithName:@"Überblick" recursively:YES];

  SCNNode* kanne = [self.scene.rootNode childNodeWithName:@"teekanne" recursively:YES];
  
  CABasicAnimation* animation = [CABasicAnimation animationWithKeyPath:@"position"];
  SCNNode* teil = [kanne childNodeWithName:@"deckel" recursively:NO];
  animation.toValue = [NSValue valueWithSCNVector3:teil.position];
  animation.fromValue = [NSValue valueWithSCNVector3:SCNVector3Make(teil.position.x, teil.position.y + 50.0, teil.position.z)];
  [teil addAnimation: animation forKey:@"fumpD"];

  animation = [CABasicAnimation animationWithKeyPath:@"position"];
  teil = [kanne childNodeWithName:@"korpus" recursively:NO];
  animation.toValue = [NSValue valueWithSCNVector3:teil.position];
  animation.fromValue = [NSValue valueWithSCNVector3:SCNVector3Make(teil.position.x, teil.position.y - 50.0, teil.position.z)];
  [teil addAnimation: animation forKey:@"fumpK"];

  animation = [CABasicAnimation animationWithKeyPath:@"position"];
  teil = [kanne childNodeWithName:@"henkel" recursively:NO];
  animation.toValue = [NSValue valueWithSCNVector3:teil.position];
  animation.fromValue = [NSValue valueWithSCNVector3:SCNVector3Make(teil.position.x - 50.0, teil.position.y, teil.position.z)];
  [teil addAnimation: animation forKey:@"fumpH"];

  animation = [CABasicAnimation animationWithKeyPath:@"position"];
  teil = [kanne childNodeWithName:@"ausguss" recursively:NO];
  animation.toValue = [NSValue valueWithSCNVector3:teil.position];
  animation.fromValue = [NSValue valueWithSCNVector3:SCNVector3Make(teil.position.x + 150.0, teil.position.y, teil.position.z)];
  [teil addAnimation: animation forKey:@"fumpA"];

  kanne.opacity = 1.0;

  [SCNTransaction setCompletionBlock:^{
    [self animateGroup];
  }];
  [SCNTransaction commit];
}


- (void)animateGroup
{
  [SCNTransaction begin];
  [SCNTransaction setAnimationDuration: 1.5];
  
  SCNNode* node = [self.scene.rootNode childNodeWithName:@"teekanne" recursively:YES];
  node.scale = SCNVector3Make(1.2, 1.2, 1.2);
  node.rotation = SCNVector4Make(0.0, 0.0, 1.0, toRadians(110));
  
  [SCNTransaction setCompletionBlock:^{
    
    [SCNTransaction begin];
    [SCNTransaction setAnimationDuration: 1.5];
    
    SCNNode* node = [self.scene.rootNode childNodeWithName:@"teekanne" recursively:YES];
    node.position = SCNVector3Make(node.position.x + 25, node.position.y + 9, node.position.z);
    
    node = [self.scene.rootNode childNodeWithName:@"deckel" recursively:YES];
    node.position = SCNVector3Make(node.position.x + 20, node.position.y + 60, node.position.z);
    node.rotation = SCNVector4Make(0.0, 0.0, 1.0, toRadians(-110));
    
    node = [self.scene.rootNode childNodeWithName:@"Ring" recursively:YES];
    node.position = SCNVector3Make(node.position.x - 25, node.position.y, node.position.z);
    
    node = [self.scene.rootNode childNodeWithName:@"Banner" recursively:YES];
    node.opacity = 1.0;
    
    self.pointOfView = [self.scene.rootNode childNodeWithName:@"Nahaufnahme" recursively:YES];
    self.pointOfView.transform = lookAt(SCNVector3Make(0.0, 0.0, 0.0), 35.0, 0.0, 20.0);
    
    [SCNTransaction setCompletionBlock:^{
      [SCNTransaction begin];
      [SCNTransaction setAnimationDuration: 1.0];
      SCNNode* node = [self.scene.rootNode childNodeWithName:@"Glühbirne" recursively:YES];
      node.light.color = [NSColor colorWithDeviceWhite:0.9 alpha:1.0];
      [SCNTransaction commit];
    }];
    [SCNTransaction commit];
  }];
  [SCNTransaction commit];
}


@end
