// Copyright (c) 2014-2015 Oliver Lau <ola@ct.de>, Heise Zeitschriften Verlag
// All rights reserved.

#ifndef __PUBSUB_H_
#define __PUBSUB_H_

// #define USE_VECTOR

#ifdef USE_VECTOR
#include <vector>
#include <algorithm>
#else
#include <list>
#endif
#include "Variant.h"

class Publisher;

class Subscriber {
public:
  virtual void notify(
    const Variant &message = Variant(),
    Publisher *sender = nullptr) = 0;
};

#ifdef USE_VECTOR
typedef std::vector<Subscriber*> SubscriberList;
#else
typedef std::list<Subscriber*> SubscriberList;
#endif

class Publisher {
public:
  Publisher(void) : isBlocked(false) { /* ... */ }

  void subscribe(Subscriber *subscriber) {
    subscribers.push_back(subscriber); // O(1)
  }

  void unsubscribe(Subscriber *subscriber) {
#ifdef USE_VECTOR
    auto pos = std::find_if(
      subscribers.begin(),
      subscribers.end(),
      [&](Subscriber *s) { return s == subscriber; }
    );
    if (pos != subscribers.end())
      subscribers.erase(pos);
#else
    subscribers.remove(subscriber); // O(n)
#endif
  }

  void broadcast(const Variant &message,
    Subscriber *ignoredSubscriber = nullptr) 
  {
    if (isBlocked)
      return;
    SubscriberList::iterator s;
    for (s = subscribers.begin(); s != subscribers.end(); ++s)
      if (*s != ignoredSubscriber)
        (*s)->notify(message, this);
  }

  void block(bool blocked) {
    isBlocked = blocked;
  }

private:
  SubscriberList subscribers;
  bool isBlocked;
};

#endif // __PUBSUB_H_
