﻿using FeedReader.Model;
using System;
using System.Collections.Generic;
using System.Runtime.Serialization;
using System.Text;
using System.Threading.Tasks;
using Windows.Storage;
using Windows.Storage.Streams;
using Windows.Web.Syndication;
using System.IO;
using System.Threading;
namespace FeedReader
{
    class FeedClient
    {
        private Uri uri;

        public FeedClient(Uri uri)
        {
            this.uri = uri;
        }

        internal async Task<Feed> LoadDataAsync(bool updateCache = false, CancellationToken token = new CancellationToken())
        {
            if (updateCache)
            {
                await UpdateCacheAsync(token).ConfigureAwait(false);
            }

            return await ReadFromCacheAsync(token).ConfigureAwait(false); ;
        }

        internal async Task RefreshCacheAsync(Action<Feed> callback, CancellationToken token)
        {
            try
            {
                await UpdateCacheAsync(token).ConfigureAwait(false); ;
                var feed = await ReadFromCacheAsync(token).ConfigureAwait(false); ;
                if (callback != null)
                {
                    callback(feed);
                }
            }
            catch (OperationCanceledException)
            {
                // because this method is not awaited do not rethrow.
            }
            catch (ObjectDisposedException)
            {
                // because this method is not awaited do not rethrow.
            }
        }

        private async Task UpdateCacheAsync(CancellationToken token)
        {
            token.ThrowIfCancellationRequested();
            try
            {
                var client = new SyndicationClient();
                // Although most HTTP servers do not require User-Agent header, 
                // others will reject the request or return a different response 
                // if this header is missing. 
                // Use SetRequestHeader() to add custom headers.
                client.SetRequestHeader("User-Agent", "Mozilla/5.0 (compatible; MSIE 10.0; Windows NT 6.2; WOW64; Trident/6.0)");


                var feed = await client.RetrieveFeedAsync(this.uri).AsTask().ConfigureAwait(false);
                var feedData = Feed.Import(feed);
                await WriteToCacheAsync(feedData, token);
            }
            catch(OperationCanceledException)
            {
                throw;
            }
            catch (ObjectDisposedException)
            {
                throw;
            }
            catch (Exception)
            {
            }
        }

        private async Task<Feed> ReadFromCacheAsync(CancellationToken token)
        {
            token.ThrowIfCancellationRequested();
            try
            {
                var folder = ApplicationData.Current.LocalFolder;
                var file = await folder.GetFileAsync("feed.xml").AsTask().ConfigureAwait(false);
                using (var randomStream = await file.OpenReadAsync().AsTask().ConfigureAwait(false))
                {
                    using (var stream = randomStream.AsStream())
                    {
                        var serializer = new DataContractSerializer(typeof(Feed));
                        return serializer.ReadObject(stream) as Feed;
                    }
                }
            }
            catch (OperationCanceledException)
            {
                throw;
            }
            catch (ObjectDisposedException)
            {
                throw;
            }
            catch (Exception)
            {
                return null;
            }
        }
        private async Task WriteToCacheAsync(Feed feedData, CancellationToken token)
        {
            token.ThrowIfCancellationRequested();
            try
            {
                var folder = ApplicationData.Current.LocalFolder;
                var file = await folder.CreateFileAsync("feed.xml", CreationCollisionOption.ReplaceExisting).AsTask().ConfigureAwait(false);
                using (var stream = await file.OpenStreamForWriteAsync().ConfigureAwait(false))
                {
                    var serializer = new DataContractSerializer(typeof(Feed));
                    serializer.WriteObject(stream, feedData);
                }
            }
            catch (OperationCanceledException)
            {
                throw;
            }
            catch (ObjectDisposedException)
            {
                throw;
            }
            catch (Exception)
            {
            }
        }
    }
}
