﻿using FeedReaderUAP.Common;
using FeedReaderUAP.Model;
using System;
using System.Collections.Generic;
using System.Collections.ObjectModel;
using System.IO;
using System.Linq;
using System.Runtime.InteropServices.WindowsRuntime;
using System.Threading;
using System.Threading.Tasks;
using Windows.Foundation;
using Windows.Foundation.Collections;
using Windows.UI.Core;
using Windows.UI.Xaml;
using Windows.UI.Xaml.Controls;
using Windows.UI.Xaml.Controls.Primitives;
using Windows.UI.Xaml.Data;
using Windows.UI.Xaml.Input;
using Windows.UI.Xaml.Media;
using Windows.UI.Xaml.Navigation;

// The Items Page item template is documented at http://go.microsoft.com/fwlink/?LinkId=234233

namespace FeedReaderUAP
{
    /// <summary>
    /// A page that displays a collection of item previews.  In the Split Application this page
    /// is used to display and select one of the available groups.
    /// </summary>
    public sealed partial class MainPage : Page
    {
        private readonly FeedClient _client;
        private NavigationHelper navigationHelper;
        private ObservableDictionary defaultViewModel = new ObservableDictionary();
        private CancellationTokenSource _cts = new CancellationTokenSource();

        /// <summary>
        /// This can be changed to a strongly typed view model.
        /// </summary>
        public ObservableDictionary DefaultViewModel
        {
            get { return this.defaultViewModel; }
        }

        /// <summary>
        /// NavigationHelper is used on each page to aid in navigation and 
        /// process lifetime management
        /// </summary>
        public NavigationHelper NavigationHelper
        {
            get { return this.navigationHelper; }
        }

        public MainPage()
        {
            this.InitializeComponent();
            NavigationCacheMode = NavigationCacheMode.Required;
            _client = new FeedClient(new Uri("http://www.heise.de/newsticker/heise-atom.xml", UriKind.Absolute));
            this.navigationHelper = new NavigationHelper(this);
            this.navigationHelper.LoadState += this.LoadState;
        }

        /// <summary>
        /// Populates the page with content passed during navigation.  Any saved state is also
        /// provided when recreating a page from a prior session.
        /// </summary>
        /// <param name="sender">
        /// The source of the event; typically <see cref="Common.NavigationHelper"/>
        /// </param>
        /// <param name="e">Event data that provides both the navigation parameter passed to
        /// <see cref="Frame.Navigate(Type, Object)"/> when this page was initially requested and
        /// a dictionary of state preserved by this page during an earlier
        /// session.  The state will be null the first time a page is visited.</param>
        private async void LoadState(object sender, LoadStateEventArgs e)
        {
            _cts.Cancel();
            try
            {
                _cts = new CancellationTokenSource();
                var token = _cts.Token;

                this.ShowLoading();

                // if page not cached
                if (!this.DefaultViewModel.ContainsKey("Items"))
                {
                    // load cached items
                    var feed = await _client.LoadDataAsync(false, token);
                    if (feed != null)
                    {
                        this.DefaultViewModel["Items"] = new ObservableCollection<FeedItem>(feed.Items);
                    }
                }

                // Update the items cache;
                await Task.Run(() => _client.RefreshCacheAsync(OnCachRefreshed, token), token);
            }
            catch (OperationCanceledException)
            {
                // loading was canceled
                HideLoading();
            }
            catch (ObjectDisposedException)
            {
                // loading was canceled
                HideLoading();
            }
        }

        private async void ShowLoading()
        {
            this.DefaultViewModel["IsBusy"] = true;
        }

        private async void HideLoading()
        {
            this.DefaultViewModel["IsBusy"] = false;
        }

        private void OnCachRefreshed(Feed feed)
        {
            // marshal to ui thread
            if (this.Dispatcher.HasThreadAccess)
            {
                this.MergeFeedDataAsync(feed);
            }
            else
            {
                this.Dispatcher.RunAsync(CoreDispatcherPriority.Normal, delegate { this.MergeFeedDataAsync(feed); });
            }

        }

        private void MergeFeedDataAsync(Feed feed)
        {
            if (feed == null)
            {
                HideLoading();
                return;
            }

            // if loading canceled, do nothing.
            if (_cts.IsCancellationRequested)
            {
                HideLoading();
                return;
            }

            var existingItems = this.DefaultViewModel.ContainsKey("Items") ? this.DefaultViewModel["Items"] as ObservableCollection<FeedItem> : null;
            if (existingItems == null)
            {
                this.DefaultViewModel["Items"] = new ObservableCollection<FeedItem>(feed.Items);
            }
            else
            {
                // get items to remove
                var itemsToRemove = existingItems.Where(i => !feed.Items.Contains(i)).ToList();
                foreach (var item in itemsToRemove)
                {
                    existingItems.Remove(item);
                }

                var index = 0;
                foreach (var item in feed.Items)
                {
                    if (existingItems.Contains(item))
                    {
                        // move existing items
                        var currentIndex = existingItems.IndexOf(item);
                        if (currentIndex != index)
                        {
                            existingItems.Move(currentIndex, index);
                        }
                    }
                    else
                    {
                        // insert new items
                        existingItems.Insert(index, item);
                    }

                    index++;
                }
            }

            HideLoading();
        }

        private void HandleItemLoaded(object sender, RoutedEventArgs e)
        {
            var control = sender as WebView;
            if (control == null)
            {
                return;
            }

            var model = control.DataContext as FeedItem;
            if (model == null)
            {
                return;
            }
            control.NavigateToString(model.SummaryHtml);
        }

        private void HandleItemClicked(object sender, ItemClickEventArgs e)
        {
            var item = e.ClickedItem as FeedItem;
            if (item == null)
            {
                return;
            }

            Frame.Navigate(typeof(WebViewPage), item.ItemUri.OriginalString);
        }

        #region NavigationHelper registration

        /// The methods provided in this section are simply used to allow
        /// NavigationHelper to respond to the page's navigation methods.
        /// 
        /// Page specific logic should be placed in event handlers for the  
        /// <see cref="Common.NavigationHelper.LoadState"/>
        /// and <see cref="Common.NavigationHelper.SaveState"/>.
        /// The navigation parameter is available in the LoadState method 
        /// in addition to page state preserved during an earlier session.

        protected override void OnNavigatedTo(NavigationEventArgs e)
        {
            navigationHelper.OnNavigatedTo(e);
        }

        protected override void OnNavigatedFrom(NavigationEventArgs e)
        {
            navigationHelper.OnNavigatedFrom(e);
        }

        #endregion

    }
}
