#!/usr/bin/perl
#
# Copyright (C) 2015  Heise Medien GmbH & Co. KG
#
# Author: Kai Wasserbäch <kaw@ct.de>
#
#
# Generate QR codes. Either for an URL given on the CLI or listed in an input
# file. In the latter case multiple URLs might be given, one per line of the
# input file.
#
# Notes:
#  - we use the lowest possible error correction level to minimize the size of
#    the QR code
#  - the default values for DPI and dimensions where chosen to produce an
#    15 cm × 15 cm image.
#  - output format is a grey-scale PNG, with black "1" modules and transparent
#    "0" modules. Use a background that maximizes contrast.
#  - border margin is set to 0! If you don't have a homogenously coloured
#    background which is larger than the code, set a margin of at least 3.
#

use strict;
use warnings;
use utf8;
binmode STDOUT, ":encoding(utf-8)";
binmode STDERR, ":encoding(utf-8)";

use Digest::MD5 qw/md5_hex/;
use File::Path qw/make_path/;
use File::Spec::Functions qw/catdir catfile curdir rel2abs/;
use Getopt::Long::Descriptive;
use Imager::QRCode;
use POSIX qw/ceil/;
use Regexp::Common qw/URI/;
use Switch;
use Try::Tiny;

our $sOutPath = rel2abs(catdir((curdir(), 'qrcodes')));
our $oOpt;
my $oUsage;

# methods
sub trim {
	my $sIn = shift;

	return undef  unless($sIn);

	$sIn =~ s/^\s+|\s+$//g;
	return $sIn;
}

sub getDotSize {
	my $iLevel = shift;

	my $hDots = {
		 1 => 21,
		 2 => 25,
		 3 => 29,
		 4 => 33,
		 5 => 37,
		 6 => 41,
		 7 => 45,
		 8 => 49,
		 9 => 53,
		10 => 57
	};

	return undef  unless($iLevel && exists($hDots->{$iLevel}));

	my $iDots = $hDots->{$iLevel};
	if($oOpt->dimension < $iDots) {
		printf(STDERR 'Your dimensions are too small, you need at least %d '
		  . "pixels\n". $iDots);
		return undef;
	} elsif($oOpt->dimension == $iDots) {
		warn('Your dots will be very small, you might want to increase the '
		  . 'dimensions.');
	}

	return ceil($oOpt->dimension / $iDots);
}

sub getQrLevel {
	my $iLength = shift;

	return undef  unless($iLength);

	my $hLevels = {
		L => [25, 47, 77, 114, 154, 195, 224, 279, 335, 395],
		M => [20, 38, 61, 90, 122, 154, 178, 221, 262, 311],
		Q => [16, 29, 47, 67, 87, 108, 125, 157, 189, 221],
		H => [10, 20, 35, 50, 64, 84, 93, 122, 143, 174]
	};

	my @aOptions = @{$hLevels->{$oOpt->ecc}};
	for(my $i=0; $i < scalar(@aOptions); $i++) {
		return $i+1  if($iLength <= $aOptions[$i]);
	}

	printf(STDERR 'Your string is too long (%d characters) for the selected '
	  . "error correction level (%s).\n", $iLength, $oOpt->ecc);
	return undef;
}

sub genQr {
	my $sUrl = shift;

	$sUrl = trim($sUrl);
	unless($sUrl =~ m/$RE{URI}{HTTP}{-scheme => 'https?'}/) {
		printf(STDERR "»%s« doesn't look like a valid URL!\n",
		  $sUrl ? $sUrl : '<UNDEF>');
		return 0;
	}

	my $sFilePath = catfile(($sOutPath), sprintf('qrcode_%s.png',
	  md5_hex($sUrl)));
	if(-e $sFilePath) {
		printf(STDERR 'It seems like the QR code for <%s> was already generated '
		  . "in %s, skipping...\n", $sUrl, $sFilePath);
		return 0;
	}

	my $iQrLvl = getQrLevel(length($sUrl));
	unless($iQrLvl) {
		printf(STDERR "Cannot determine QR level. URL <%s> is %d characters long.\n",
		  $sUrl, length($sUrl));
		return 0;
	}

	my $iSize = getDotSize($iQrLvl);
	unless($iSize) {
		printf(STDERR 'Failed to determine dot size for QR level %d and %d × %d '
		  . "pixels at %d DPI.\n", $iQrLvl, $oOpt->dimension, $oOpt->dimension,
		  $oOpt->resolution);
		return 0;
	}

	my @aLightC = split(',', $oOpt->lightcolour);
	my @aDarkC  = split(',', $oOpt->darkcolour);
	my $oImager = Imager::QRCode->new(
		size			=> $iSize,
		margin			=> $oOpt->margin,
		version			=> $iQrLvl,
		level			=> $oOpt->ecc,
		casesensitive		=> 1,
		lightcolor		=> Imager::Color->new($aLightC[0], $aLightC[1],
		  $aLightC[2], $aLightC[3]),
		darkcolor		=> Imager::Color->new($aDarkC[0], $aDarkC[1],
		  $aDarkC[2], $aDarkC[3])
	);
	my $oTmpImg = $oImager->plot($sUrl);
	my $oOutImg;
	unless($oOpt->no_greyscale) {
		$oOutImg = $oTmpImg->convert(preset => 'grey');
	} else {
		$oOutImg = $oTmpImg;
	}
	$oOutImg->settag(name => 'i_xres', value => $oOpt->resolution);
	$oOutImg->settag(name => 'i_yres', value => $oOpt->resolution);
	my $bSuccess = try {
		$oOutImg->write(file => $sFilePath);
		1;
	} catch {
		printf(STDERR "Failed to write QR code for url <%s> to %s: %s\n", $sUrl,
		  $sFilePath, $_);
	};

	return 0  unless($bSuccess);

	printf("Successfully written URL <%s> QR-encoded to %s (level=%d, "
	  . "dot-size=%d)\n", $sUrl, $sFilePath, $iQrLvl, $iSize);
	return 1;
}


# options
#
# Validation regex for our RGBA values; each element must be in the range of
# 0 and 255, we need four elements.
my $oValidNums = qr/([0-9]|[1-9][0-9]|1[0-9][0-9]|2[0-4][0-9]|25[0-5])/;
my $oColourRegex = qr/^($oValidNums,){3}$oValidNums$/;
($oOpt, $oUsage) = describe_options(
	'%c %o [<URL>]', (
	['Generate QR codes for one or more URLs with some customization options.'],
	['Copyright (C) 2015  Heise Medien GmbH & Co. KG'],
	['Author: Kai Wasserbäch <kaw@heise.de>'],
	[],[],
	['file|f=s'	=>	'Read multiple URLs from this file instead of using <URL> as input.'],
	['dimension|d=i',	'Dimension in pixels', {default => 1775}],
	['resolution|r=i',	'Resolution in DPI', {default => 300}],
	['ecc|e=s',		'Error correction level for QR code, valid options are L, M, Q or H', {default => 'L', regex => qr/^(L|M|Q|H)$/}],
	['margin|m=i',		'Margin around the QR code.', {default => 0}],
	['lightcolour|l=s',	'Light colour for "0" modules in the code. Colour needs to be given as a comma-seperated RGBA colour.', {default => '255,255,255,0', regex => $oColourRegex}],
	['darkcolour|a=s',	'Dark colour for "1" modules in the code. Colour needs to be given as a comma-seperated RGBA colour.', {default => '0,0,0,255', regex => $oColourRegex}],
	['no-greyscale|n', =>	'Don\'t force the colour space to greyscale.'],
	['help|h'	=>	'Show this help']
	),
	{show_defaults => 1}
);

print($oUsage->text), exit  if($oOpt->help);
if(!$oOpt->file && !$ARGV[0]) {
	print(STDERR "ERROR: Neither file parameter nor <URL> parameter were given!\n\n");
	print($oUsage->text);
	exit;
}

# main flow
make_path($sOutPath);

if($oOpt->file) {
	open(my $oFh, '<', $oOpt->file) or die(sprintf('Cannot open %s: %s', $oOpt->file, $!));
	while(my $sRow = <$oFh>) {
		genQr($sRow);
	}
} else {
	genQr($ARGV[0]);
}
