package com.linkesoft.ctlogocardboard;

import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.FloatBuffer;

import android.graphics.Color;
import android.opengl.GLES20;

/**
 * A two-dimensional triangle for use as a drawn object in OpenGL ES 2.0.
 */
public class Triangle {

    // Vertices für ein einzelnes Dreieck
    private static final float triangleCoords[] = {
            -0.2f,0,-0.2f,
            0.2f,0,-0.2f,
            0,0,0.2f
    };

    private static final String VERTEX_SHADER =
            "uniform mat4 uMVPMatrix;" + // Model-View-Projection-Matrix
            "attribute vec4 aPosition;" + // Vertices
            "void main() {" +
            "  gl_Position = uMVPMatrix * aPosition;" + // endgültige Position
            "}";

    private static final String FRAGMENT_SHADER =
            "uniform vec4 uColor;" + // Farbe
            "void main() {" +
            "  gl_FragColor = uColor;" + // wird einfach durchgereicht
            "}";

    // Konstanten
    private static final int COORDS_PER_VERTEX = 3;
    private static final int BYTES_PER_FLOAT=4;

    // Handles für OpenGL
    private final int uMVPMatrix;
    private final int aPosition;
    private final int uColor;

    // Puffer für Vertices
    private final FloatBuffer vertexBuffer;

    // Initiale Farbe des Dreiecks (wird über Trigger geändert)
    public int color=Color.GREEN;

    /**
     * OpenGL-Initialisierung
     */
    public Triangle() {
        // float-Array mit Vertices in ByteBuffer kopieren
        vertexBuffer = ByteBuffer.allocateDirect(triangleCoords.length * BYTES_PER_FLOAT).order(ByteOrder.nativeOrder()).asFloatBuffer();
        vertexBuffer.put(triangleCoords);
        vertexBuffer.position(0);

        // erzeuge Shader und OpenGL-Programm
        int glProgram = GLES20.glCreateProgram();
        // Vertex-Shader
        int vertexShader = GLES20.glCreateShader(GLES20.GL_VERTEX_SHADER);
        GLES20.glShaderSource(vertexShader, VERTEX_SHADER);
        GLES20.glCompileShader(vertexShader);
        GLES20.glAttachShader(glProgram, vertexShader);
        // Fragment-Shader
        int fragmentShader = GLES20.glCreateShader(GLES20.GL_FRAGMENT_SHADER);
        GLES20.glShaderSource(fragmentShader, FRAGMENT_SHADER);
        GLES20.glCompileShader(fragmentShader);
        GLES20.glAttachShader(glProgram, fragmentShader);

        GLES20.glLinkProgram(glProgram);
        GLES20.glUseProgram(glProgram);

        // initialisiere Handles zu OpenGL-Speicherbereichen
        uMVPMatrix = GLES20.glGetUniformLocation(glProgram, "uMVPMatrix");
        aPosition = GLES20.glGetAttribLocation(glProgram, "aPosition");
        uColor = GLES20.glGetUniformLocation(glProgram, "uColor");
        GLES20.glEnableVertexAttribArray(aPosition);
    }

    /**
     * Zeichne einen Frame in OpenGL ES
     *
     * @param mvpMatrix - Model-View-Projection-Matrix
     */
    public void draw(float[] mvpMatrix) {
        GLES20.glClear(GLES20.GL_COLOR_BUFFER_BIT|GLES20.GL_DEPTH_BUFFER_BIT);
        GLES20.glClearColor(0.1f,0.1f,0.2f,1f); // Hintergrundfarbe
        // Vertices des Dreiecks
        GLES20.glVertexAttribPointer(aPosition, COORDS_PER_VERTEX,GLES20.GL_FLOAT, false,0, vertexBuffer);

        // Farbe des Dreiecks
        float rgba[]=new float[4];
        rgba[0]=Color.red(color)/255f;
        rgba[1]=Color.green(color)/255f;
        rgba[2]=Color.blue(color)/255f;
        rgba[3]=Color.alpha(color)/255f;
        GLES20.glUniform4fv(uColor, 1, rgba, 0);

        // Model-View-Projection-Matrix
        GLES20.glUniformMatrix4fv(uMVPMatrix, 1, false, mvpMatrix, 0);

        // zeichne Dreieck
        GLES20.glDrawArrays(GLES20.GL_TRIANGLE_STRIP, 0, triangleCoords.length / COORDS_PER_VERTEX);

    }

}
