package com.linkesoft.bbingo;

import android.Manifest;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.DialogInterface;
import android.content.Intent;
import android.content.IntentFilter;
import android.content.pm.PackageManager;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.PorterDuff;
import android.media.AudioManager;
import android.media.SoundPool;
import android.os.Build;
import android.os.Bundle;
import android.provider.Settings;
import android.support.annotation.NonNull;
import android.support.v4.content.ContextCompat;
import android.support.v7.app.AlertDialog;
import android.support.v7.app.AppCompatActivity;
import android.text.InputType;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.View;
import android.view.ViewGroup;
import android.view.WindowManager;
import android.widget.Button;
import android.widget.EditText;
import android.widget.ImageView;
import android.widget.Toast;

import com.google.android.gms.common.ConnectionResult;
import com.google.android.gms.common.GoogleApiAvailability;

import java.util.Collections;
import java.util.List;

/**
 * Hauptklasse-Activity der B-Bingo-Anwendung,
 * zeigt ein Gitter von 5x5 Buttons zur Auswahl von Buzzwörtern.
 *
 * Siehe https://de.wikipedia.org/wiki/Buzzword-Bingo
 *
 * @author Andreas Linke
 *
 */
public class MainActivity extends AppCompatActivity {

    private static final int GRID_SIZE = 5;

    private String currentWordlist=null;

    private BBingoDB db;
    private SoundPool soundPool; // Erfolgsfanfare
    private int fanfareSoundID; // ID der geladenen Sound-Resource
    private int streamID; // ID des gerade gespielten Sounds (zum Stoppen)

    private static final int WORD_LISTS_REQUEST = 1;
    private static final int PERMISSIONS_REQUEST = 2;

    private NearbyConnections nearbyConnections;
    private final BroadcastReceiver nearbyConnectionsUpdateReceiver = new BroadcastReceiver() {
        @Override
        public void onReceive(Context context, Intent intent) {
            if(intent.getAction().equals(NearbyConnections.CONNECTIONS_UPDATED)) {
                onNearbyConnectionUpdate();
            } else if(intent.getAction().equals(NearbyConnections.WINNER_FOUND)) {
                String winnerName = intent.getStringExtra(NearbyConnections.USER_NAME);
                onNearbyWinnerFound(winnerName);
            } else if(intent.getAction().equals(NearbyConnections.SELECT_WORDLIST)){
                BBingoDB.WordList wordList = new BBingoDB.WordList();
                wordList.title = intent.getStringExtra(BBingoDB.TITLE);
                wordList.words = intent.getStringExtra(BBingoDB.WORDS);
                String senderUserName = intent.getStringExtra(NearbyConnections.USER_NAME);
                selectWordList(wordList,senderUserName);
            }
        }
    };

    /**
     * Startmethode, wird beim ersten Start der Activity aufgerufen,
     * auch bei Änderung der Orientierung (horizontal/vertikal).
     * Definiere GUI-Layout und verknüpfe Event-Handler.
     */
    @Override
    public void onCreate(Bundle savedInstanceState) {
        super.onCreate(savedInstanceState);
        setContentView(R.layout.main); // aus XML
        // verknüpfe Buttons mit Aktionen
        for (int y = 0; y < GRID_SIZE; y++)
            for (int x = 0; x < GRID_SIZE; x++) {
                Button button = (Button) findViewById(getButtonID(x, y));
                button.setOnClickListener(onButtonClick);
            }
        db=new BBingoDB(this); // oeffne Datenbank, wird in onDestroy wieder geschlossen

        // lade Soundschnipsel fuer Gewinn-Fanfare

        //noinspection deprecation
        soundPool = new SoundPool(1, AudioManager.STREAM_MUSIC, 0); // ab Lollipop soundPool = new SoundPool.Builder().build();
        fanfareSoundID=soundPool.load(this, R.raw.fanfare, 1); // Verweis auf fanfare.mp3 im res/raw Ordner des Projekts
        loadWordList();
    }

    /**
     * Spiele vorher im soundPool geladenen Sound ab.
     */
    private void playFanfare() {
        AudioManager audioManager = (AudioManager) getSystemService(AUDIO_SERVICE);
        // berechne relative Lautstaerke, basierend auf vom Benutzer
        // eingestellter Lautstaerke und Maximum (int-Werte!)
        float volume = (float) audioManager.getStreamVolume(AudioManager.STREAM_MUSIC) /
                (float) audioManager.getStreamMaxVolume(AudioManager.STREAM_MUSIC);
        // spiele den in onCreate geladenen Sound
        streamID = soundPool.play(fanfareSoundID, volume, volume, 1, 0, 1f);
    }

    /**
     * Wird nach onCreate und nach dem Beenden einer anderen Activity aufgerufen.
     * Lade die aktuelle Wortliste. Wenn sie nicht mit der aktuellen Liste übereinstimmt,
     * mische und setze alle Buttons neu.
     */
    @Override
    protected void onResume() {
        super.onResume();
        // register for broadcasts from nearby connections
        IntentFilter filter = new IntentFilter();
        filter.addAction(NearbyConnections.CONNECTIONS_UPDATED);
        filter.addAction(NearbyConnections.WINNER_FOUND);
        filter.addAction(NearbyConnections.SELECT_WORDLIST);
        registerReceiver(nearbyConnectionsUpdateReceiver,filter);
        if(Prefs.getNearby(this))
            startConnectingToNearbyDevices();
        onNearbyConnectionUpdate();
    }

    @Override
    protected void onPause() {
        unregisterReceiver(nearbyConnectionsUpdateReceiver);
        stopConnectingToNearbyDevices();
        super.onPause();
    }

    /**
     * Hole aktuelle Wortliste aus der DB.
     * Falls noch keine Liste gespeichert wurde,
     * wird auf eine eingebaute Standard-Liste zurückgegriffen.
     */
    private void loadWordList()
    {
        //
        BBingoDB.WordList wordlist = db.getWordList(Prefs.getID(this));
        if (wordlist == null) {
            // keine Wortliste zur ID gefunden, nimm erste
            long id = db.getFirstWordListID();
            if (id == 0) {
                // keine Wortliste gespeichert
                id = db.createDefaultWordList(this); // erzeuge Standard-Liste
            }
            Prefs.setID(this, id);
            wordlist = db.getWordList(id);
        }
        setTitle("B-Bingo: "+wordlist.title); // zeige Titel der aktuelle Wortliste im Titel der Activity
        if (currentWordlist == null || !currentWordlist.equals(wordlist.words)) {
            // Wortliste hat sich geaendert, mische und setze Button-Titel entsprechend
            currentWordlist = wordlist.words;
            shuffleWords(false); // keine Animation, da schon beim Layout animiert wurde
        }
    }

    @Override
    protected void onDestroy() {
        db.close(); // jede Activity hat eine eigene Datenbank-Verbindung, die beim Beenden geschlossen werden muss
        super.onDestroy();
    }
    /**
     * Button-Klick-Handler fuer alle Buttons
     */
    final View.OnClickListener onButtonClick=new View.OnClickListener() {
        @Override
        public void onClick(View v) {
            toggleButtonState((Button) v);
            checkForBingo();
        }
    };

    /**
     * Anlegen von Menüs
     */
    @Override
    public boolean onCreateOptionsMenu(android.view.Menu menu) {
        MenuInflater inflater = getMenuInflater();
        inflater.inflate(R.menu.main, menu);
        return super.onCreateOptionsMenu(menu);
    }

    @Override
    public boolean onPrepareOptionsMenu(Menu menu) {
        // Checkboxen im Menü
        menu.findItem(R.id.nosound).setChecked(Prefs.getNoSound(this));
        // prüfe ob Google Play Services verfügbar sin
        boolean hasPlayServices = GoogleApiAvailability.getInstance().isGooglePlayServicesAvailable(this)== ConnectionResult.SUCCESS;
        if(!hasPlayServices) {
            Log.e(getClass().getSimpleName(), "Google Play Services not available: "+
                    "dpi: "+getResources().getDisplayMetrics().densityDpi +
                    " Android "+Build.VERSION.RELEASE +
                    " Arch " + Build.CPU_ABI// os.arch
            );
        }
        menu.findItem(R.id.nearby).setChecked(Prefs.getNearby(this)).setVisible(hasPlayServices);
        menu.findItem(R.id.sendToNearbyPeers).setEnabled(nearbyConnections!=null && nearbyConnections.hasConnectedPeers()).setVisible(hasPlayServices);
        return super.onPrepareOptionsMenu(menu);
    }

    /**
     * Auswahl eines Menü-Eintrags
     */
    @Override
    public boolean onOptionsItemSelected(MenuItem item) {
        switch (item.getItemId()) {
            case R.id.lists:
                startActivityForResult(new Intent(this, WordListsActivity.class), WORD_LISTS_REQUEST);
                break;
            case R.id.nosound:
                Prefs.setNoSound(this,!Prefs.getNoSound(this)); // toggle
                if(Prefs.getNoSound(this))
                    soundPool.stop(streamID); // stoppe Sound sofort
                break;
            case R.id.nearby:
                Prefs.setNearby(this,!Prefs.getNearby(this)); // toggle
                if(Prefs.getNearby(this))
                    askUserNameForNearbyConnections();
                else
                    stopConnectingToNearbyDevices();
                break;
            case R.id.sendToNearbyPeers:
                BBingoDB.WordList wordlist = db.getWordList(Prefs.getID(this));
                if(nearbyConnections!=null && wordlist != null)
                    nearbyConnections.sendWordList(wordlist);
                break;
        }
        return super.onOptionsItemSelected(item);
    }

    /**
     * Wird nach Beenden einer per <code>startActivityForResult</code> gestarteten
     * Activity gerufen
     */
    @Override
    protected void onActivityResult(int requestCode, int resultCode, Intent data) {
        if (requestCode == WORD_LISTS_REQUEST && resultCode == RESULT_OK) {
            // eine andere Liste wurde ausgewählt
            long id = data.getExtras().getLong(WordListsActivity.ID);
            Prefs.setID(this, id);
            loadWordList();
        }
        super.onActivityResult(requestCode, resultCode, data);
    }

    // Nearby Connection Start/Stop

    private void startConnectingToNearbyDevices() {
        // prüfe/frage Berechtigung für COARSE_LOCATION ab Android 6
        if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M) {
            String permissionLocation = Manifest.permission.ACCESS_COARSE_LOCATION;
            String permissionStorage = Manifest.permission.WRITE_EXTERNAL_STORAGE; // für File payloads
            if (ContextCompat.checkSelfPermission(this, permissionLocation) != PackageManager.PERMISSION_GRANTED
                    || ContextCompat.checkSelfPermission(this, permissionStorage) != PackageManager.PERMISSION_GRANTED) {
                requestPermissions(new String[]{permissionLocation, permissionStorage}, PERMISSIONS_REQUEST);
                return;
            }
        }
        // Ortungsdienste müssen aktiviert sein
        // komfortable/aufwendigere Lösung mit LocationSettingsRequest.Builder
        if(!isLocationServicesEnabled()) {
            Toast.makeText(this,R.string.EnableLocationServices,Toast.LENGTH_LONG).show();
            Prefs.setNearby(this,false);
            return;
        }

        // starte Nearby Connections discovery/advertisement
        if (nearbyConnections != null)
            nearbyConnections.disconnect();
        // Bildschirm soll anbleiben
        getWindow().addFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON);
        nearbyConnections = new NearbyConnections(getApplicationContext(), Prefs.getUserName(this));

    }
    private void askUserNameForNearbyConnections() {
        // frage nach Benutzernamen
        final EditText nameEditText = new EditText(this);
        nameEditText.setText(Prefs.getUserName(this));
        nameEditText.setHint(R.string.UserName);
        nameEditText.setInputType(InputType.TYPE_CLASS_TEXT | InputType.TYPE_TEXT_FLAG_CAP_WORDS); // erster Buchstabe automatisch groß

        new AlertDialog.Builder(this)
                .setTitle(R.string.ConnectWithNearbyDevicesAs)
                .setView(nameEditText)
                .setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                    public void onClick(DialogInterface dialog, int whichButton) {
                        String userName = nameEditText.getText().toString();
                        Prefs.setUserName(MainActivity.this, userName);
                        startConnectingToNearbyDevices();

                    }
                })
                .setOnCancelListener(new DialogInterface.OnCancelListener() {
                    @Override
                    public void onCancel(DialogInterface dialog) {
                        // abgebrochen
                        Prefs.setNearby(MainActivity.this, false);
                    }
                })
                .show();
    }

    private void stopConnectingToNearbyDevices() {
        if(nearbyConnections!=null)
            nearbyConnections.disconnect();
        getWindow().clearFlags(WindowManager.LayoutParams.FLAG_KEEP_SCREEN_ON); // Gerät darf wieder schlafen gehen

    }

    private boolean isLocationServicesEnabled() {
        try {
            int locationSetting = Settings.Secure.getInt(getContentResolver(),Settings.Secure.LOCATION_MODE);
            return locationSetting != Settings.Secure.LOCATION_MODE_OFF;
        } catch (Settings.SettingNotFoundException e) {
            Log.e(getClass().getSimpleName(),"Could not determine location setting",e);
            return false;
        }
    }

    /**
     * Wird vom BroadcastReceiver aufgerufen, wenn NearbyConnections eine Status-Änderung hat
     */
    private void onNearbyConnectionUpdate() {
        if(nearbyConnections!=null)
            getSupportActionBar().setSubtitle(nearbyConnections.status());
        else
            getSupportActionBar().setSubtitle(null);
    }

    private void onNearbyWinnerFound(String winnerName) {
        // zeige Dialog mit Name und (optional) Screenshot des Gewinners
        AlertDialog.Builder builder =
                new AlertDialog.Builder(this).
                        setMessage(getString(R.string.winnerFound,winnerName)).
                        setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int which) {
                                dialog.dismiss();
                            }
                        });
        if(nearbyConnections.lastWinnerBitmap!=null) {
            ImageView imageView = new ImageView(this);
            imageView.setImageBitmap(nearbyConnections.lastWinnerBitmap);
            builder.setView(imageView);
        }
        builder.create().show();
    }

    private void selectWordList(final BBingoDB.WordList wordList, String senderUserName) {
        // haben wir die Wordlist schon in der Datenbank?
        long id = db.findWordList(wordList);
        if(id == Prefs.getID(this))
            return; // aktuelle Wortliste, nichts zu tun
        // frage Benutzer
        AlertDialog.Builder builder =
                new AlertDialog.Builder(this).
                        setMessage(getString(R.string.selectWordListNearby,wordList.title,senderUserName)).
                        setPositiveButton(android.R.string.ok, new DialogInterface.OnClickListener() {
                            @Override
                            public void onClick(DialogInterface dialog, int which) {
                                long id = db.setWordList(wordList);
                                Prefs.setID(MainActivity.this, id);
                                loadWordList();
                            }
                        }).setNegativeButton(android.R.string.cancel, new DialogInterface.OnClickListener() {
                    @Override
                    public void onClick(DialogInterface dialog, int which) {
                    }
                });
        builder.create().show();

    }

    @Override
    public void onRequestPermissionsResult(int requestCode, @NonNull String[] permissions, @NonNull int[] grantResults) {
        if (requestCode == PERMISSIONS_REQUEST) {
            for (int grantResult : grantResults) {
                if (grantResult == PackageManager.PERMISSION_DENIED) {
                    // nicht erlaubt
                    Prefs.setNearby(this,false);
                    return;
                }
            }
            // Erlaubnis erteilt, starte Verbindung
            startConnectingToNearbyDevices();
        }
        super.onRequestPermissionsResult(requestCode, permissions, grantResults);
    }

    // Buzzwort-Buttons

    /**
     * Liefere numerische Button-ID per Reflection aus den Konstanten der R-Klasse
     * @param x Spaltennummer 0..4
     * @param y Zeilennummer 0..4
     * @return Button-ID
     */
    private int getButtonID(int x, int y) {
        try {
            // look up id by reflection
            // button id is a static int defined in R.id, e.g. R.id.Button01
            String buttonid = "Button" + y + "" + (x + 1); // e.g. "Button01"
            return R.id.class.getField(buttonid).getInt(null);
        } catch (Exception e) {
            // reflection lookup could throw exceptions e.g. if the button is
            // not found
            throw new RuntimeException("Internal error", e);
        }
    }

    private boolean isButtonSet(Button button) {
        return button.getTag() != null;
    }

    /**
     * Aendere Zustand eines Buttons von nicht gesetzt auf gesetzt
     * und umgekehrt. Dabei wird die Button-Farbe entsprechend angepasst.
     * @param button der zu aendernde Button
     */
    private void toggleButtonState(Button button) {
        if (isButtonSet(button)) {
            button.getBackground().clearColorFilter();
            button.setTag(null);
        } else {
            // Button-Hintergrund-Farbe wird über einen Filter geändert
            //noinspection deprecation
            int highlightColor = getResources().getColor(R.color.colorAccent);
            button.getBackground().setColorFilter(highlightColor, PorterDuff.Mode.SRC_ATOP);
            button.setTag("x"); // verwende die freie tag-Property zum Speichern des Zustands
        }
    }

    /**
     * Pruefe auf vollstaendige ausgewaehlte Button-Reihe
     * vertikal, horizontal und die beiden Diagonalen
     */
    private void checkForBingo() {
        boolean bingo = false;
        int x, y;
        // vertikal
        for (x = 0; x < GRID_SIZE && !bingo; x++) {
            for (y = 0; y < GRID_SIZE; y++) {
                Button button = (Button) findViewById(getButtonID(x, y));
                if (!isButtonSet(button))
                    break;
            }
            bingo = (y == GRID_SIZE);
        }
        // horizontal
        for (y = 0; y < GRID_SIZE && !bingo; y++) {
            for (x = 0; x < GRID_SIZE; x++) {
                Button button = (Button) findViewById(getButtonID(x, y));
                if (!isButtonSet(button))
                    break;
            }
            bingo = (x == GRID_SIZE);
        }
        // diagonal
        for (x = 0; x < GRID_SIZE && !bingo; x++) {
            @SuppressWarnings("SuspiciousNameCombination") Button button = (Button) findViewById(getButtonID(x, x));
            if (!isButtonSet(button))
                break;
        }
        if (x == GRID_SIZE)
            bingo = true;
        for (x = 0; x < GRID_SIZE && !bingo; x++) {
            Button button = (Button) findViewById(getButtonID(x, GRID_SIZE - 1 - x));
            if (!isButtonSet(button))
                break;
        }
        if (x == GRID_SIZE)
            bingo = true;

        if (bingo) {
            // Gewonnen, zeige ein einfaches Popup und spiele Sound
            if(!Prefs.getNoSound(this))
                playFanfare(); // asynchron
            // Die Methoden geben jeweils wieder das Builder-Objekt zurück
            // und lassen sich so leicht verketten
            AlertDialog.Builder dlg=new AlertDialog.Builder(this)
                    .setTitle(R.string.BBingoWonTitle)
                    .setMessage(R.string.BBingoWon)
                    .setNeutralButton(R.string.New, new DialogInterface.OnClickListener() {
                        @Override
                        public void onClick(DialogInterface dialog, int which) {
                            soundPool.stop(streamID); // stoppe Sound
                            shuffleWords(true);
                        }
                    })
                    .setOnCancelListener(new DialogInterface.OnCancelListener() {
                        @Override
                        public void onCancel(DialogInterface dialog) {
                            soundPool.stop(streamID); // stoppe Sound, falls er noch gespielt wird

                        }
                    });
            dlg.show();
            // informiere verbundene Geräte und sende Screnshot als Beweis
            if(nearbyConnections!=null) {
                nearbyConnections.sendWinner(createBitmap(findViewById(R.id.LinearLayoutMain)));
            }
        }
    }

    private Bitmap createBitmap(View view) {
        // View wurde bereits gelayouted
        Bitmap bitmap = Bitmap.createBitmap(view.getWidth(), view.getHeight(),Bitmap.Config.ARGB_8888);
        Canvas canvas = new Canvas(bitmap);
        canvas.drawColor(Color.WHITE); // weißer Hintergrund
        view.draw(canvas);
        return bitmap;
    }
    /**
     * Worte der Wortliste mischen und auf die Buttons verteilen
     */
    private void shuffleWords(boolean animate)
    {
        List<String> words=BBingoDB.stringToList(currentWordlist);
        // Liste darf nicht leer sein
        if(words.size()==0)
            words.add("");
        // Liste mischen und den Buttons zuweisen
        Collections.shuffle(words);
        int i=0;
        for (int x = 0; x < GRID_SIZE; x++)
            for (int y = 0; y < GRID_SIZE; y++)
            {
                Button button=(Button) findViewById(getButtonID(x, y));
                if(isButtonSet(button))
                    toggleButtonState(button); // setze Button state zurück
                if(i>=words.size())
                {
                    // overflow
                    Collections.shuffle(words);
                    i=0;
                }
                button.setText(words.get(i++));
            }
        if(animate) {
            ViewGroup mainViewGroup = (ViewGroup) findViewById(R.id.LinearLayoutMain);
            for (int j = 0; j < mainViewGroup.getChildCount(); j++) {
                ViewGroup rowViewGroup = (ViewGroup) mainViewGroup.getChildAt(j);
                rowViewGroup.startLayoutAnimation();
            }
        }

    }
}
