unit DXPlayASUtils; // 12-FEB-98 as (Arne Schpers)
// Utilities fr DirectPlay

interface
uses Windows, SysUtils, Classes, ExtCtrls, ActiveX, Dialogs,
     DPlay, DPLobby;

type
  // Gibt Objects beim Lschen von Elementen frei
  TObjectList = class(TStringList)
  public
    destructor Destroy; override;
    procedure Clear; override;
    procedure Delete(Index: Integer); override;
  end;

type  // Listet Sessions auf -> TargetList
  TSessionEnumerator = class(TComponent)
  private
    SessionDesc: TDPSessionDesc2;
    EnumTimer: TTimer;
    FOnEnumDone: TNotifyEvent;
    RetryCount: Integer;
    FTimedOut: Boolean;
    procedure OnEnumTimer(Sender: TObject);
  public
    ApplicationGUID: TGUID;
    Password: String;
    TargetList: TObjectList;
    EnumFlags: Integer;
    DPObject: IDirectPlay4;
    constructor Create(AOwner: TComponent); override;
    destructor Destroy; override;
    procedure Start;
    procedure Stop;
    property OnEnumDone: TNotifyEvent read FOnEnumDone write FOnEnumDone;
    property TimedOut: Boolean read FTimedOut;
  end;

var // mehr als einen braucht's nicht
  SessionEnumerator: TSessionEnumerator;

// 123.55.68.12 oder www.gamezone.com, optional :PortNo
function CreateTCPIPAddress(IPAddr: String): Pointer;
// Nur im lokalen Subnetz (kein Routing mglich)
function CreateIPXAddress: Pointer;
function CreateModemAddress(InitString,PhoneNr: String):Pointer;
function CreateSerialAddress(ComData: TDPComportAddress): Pointer;

type  // zum Auflisten aller DirectPlay-Dienstanbieter
  TDPConnectionInfo = record
    GuidSP: TGUID;
    DPAddress: Array[0..1] of Byte;
  end;
  PDPConnectionInfo = ^TDPConnectionInfo;

function EnumConnections(AppGUID: TGUID): TObjectList;

type  // zum Auflisten von Gruppen und Spielern
  TDPGroupAndPlayerInfo = record
    GPID: TDPID;
    GPType, GPFlags: DWORD;
  end;
  PDPGroupAndPlayerInfo = ^TDPGroupAndPlayerInfo;

function EnumGroups(DPObject: IDirectPlay4;
                    Flags: Integer): TObjectList;
function EnumPlayers(DPObject: IDirectPlay4;
                    Flags: Integer): TObjectList;
function EnumGroupsInGroup(DPObject: IDirectPlay4;
                 GroupID: TDPID; Flags: Integer): TObjectList;
function EnumGroupPlayers(DPObject: IDirectPlay4;
                 GroupID: TDPID; Flags: Integer): TObjectList;

implementation

procedure TObjectList.Clear;
var x: Integer;
begin
  for x := 0 to Count-1 do
    if Objects[x] <> nil then FreeMem(Pointer(Objects[x]));
  inherited;
end;

procedure TObjectList.Delete(Index: Integer);
begin
  if Objects[Index] <> nil then FreeMem(Pointer(Objects[Index]));
  inherited;
end;

destructor TObjectList.Destroy;
begin
  Clear; inherited;
end;

// ---- TSessionEnumerator -------------------------------
constructor TSessionEnumerator.Create(AOwner: TComponent);
begin
  inherited;
  EnumTimer := TTimer.Create(Self);
  with EnumTimer do
  begin
    OnTimer := OnEnumTimer; Enabled := False;
  end;
  TargetList := TObjectList.Create;
end;

procedure TSessionEnumerator.Start;
var ProviderCaps: TDPCaps;
begin
  if DPObject = nil then
        raise Exception.Create('Initialize DPObject first!');
  with SessionDesc do
  begin
    dwSize := SizeOf(SessionDesc);
    guidApplication := ApplicationGUID;
    lpszPassword := PChar(Password);  // kann '' sein
    if Password <> '' then EnumFlags := DPENUMSESSIONS_PASSWORDREQUIRED
     else EnumFlags := 0;
  end;
  ProviderCaps.dwSize := SizeOf(ProviderCaps);
  DPObject.GetCaps(ProviderCaps,0);
  RetryCount := 5; FTimedOut := False;
  EnumTimer.Interval := ProviderCaps.dwTimeOut div DWord(RetryCount);
  EnumTimer.Enabled := True;
  OnEnumTimer(Self);  // 1. Abfrage
end;

function EnumSessionsCallback(var lpThisSD: TDPSessionDesc2;
  var lpdwTimeOut: DWORD; dwFlags: DWORD; lpContext: Pointer) : BOOL; stdcall;
var SessionGUID: PGUID;
begin
  if dwFlags and DPESC_TIMEDOUT <> 0 then
  begin
    Result := False; Exit;
  end;
  New(SessionGUID); SessionGUID^ := lpThisSD.guidInstance;
  TSessionEnumerator(lpContext).TargetList.AddObject(
                lpThisSD.lpszSessionName,Pointer(SessionGUID));
  Result := True;
end;

procedure TSessionEnumerator.OnEnumTimer(Sender: TObject);
const EnumActive: Boolean = False;
var Res: HResult;
begin
  if EnumActive then Exit; // Rekursionsbremse
  EnumActive := True; TargetList.Clear;
  Res := DPObject.EnumSessions(SessionDesc, 0,
   @EnumSessionsCallback, Self,
   EnumFlags or DPENUMSESSIONS_AVAILABLE or DPENUMSESSIONS_ASYNC);
  EnumActive := False;
  Dec(RetryCount);
  FTimedOut := RetryCount < 0;
  EnumTimer.Enabled := not TimedOut;
  if SUCCEEDED(Res) then
  begin
    if Assigned(FOnEnumDone) then OnEnumDone(Self);
  end else if Res <> DPERR_USERCANCEL then
  begin
    EnumTimer.Enabled := False;
    ShowMessage(Errorstring(Res));
  end;
end;

procedure TSessionEnumerator.Stop;
begin
  if (DPObject <> nil) and EnumTimer.Enabled then
  begin
    DPObject.EnumSessions(SessionDesc, 0, nil,
       nil,DPENUMSESSIONS_STOPASYNC);
    EnumTimer.Enabled := False;
    DPObject := nil;  // Referenz raus
  end;
end;

destructor TSessionEnumerator.Destroy;
begin
  Stop;
  inherited;
end;

// XRef: Alle vier CreateXYZAddress-Routinen. Data mu als var
// deklariert sein, weil Delphi sonst fr das Element 0 eine
// Lokalkopie erzeugt (und der Rest dann woanders steht)
function CreateCompoundAddress(var Data: TDPCompoundAddressElement;
  NumElems: Integer): Pointer;
var dwAddrSize: DWord; DLobby: IDirectPlayLobby3; Res: HResult;
begin
  CoCreateInstance(CLSID_DirectPlayLobby, nil,
    CLSCTX_INPROC_SERVER, IID_IDirectPlayLobby3, DLobby);
  if DLobby = nil then raise
    Exception.Create('IDirectPlayLobby3 not available');
  Res := DLobby.CreateCompoundAddress(Data, NumElems,nil,dwAddrSize);
  if Res <> DPERR_BUFFERTOOSMALL then
    raise Exception.Create('CreateCompoundAddress failed');
  GetMem(Result,dwAddrSize);
  Res := DLobby.CreateCompoundAddress(Data,NumElems,
     Result,dwAddrSize);
  if FAILED(Res) then raise Exception.Create(
   'CreateCompoundAddress: '+ErrorString(Res));
end;  // Lobby wird automatisch wieder freigegeben

// '123.55.68.12' oder 'www.gamezone.com', optional mit ':xy'
// fr Portnummer
function CreateTCPIPAddress(IPAddr: String): Pointer;
var AddrElems: Array[0..2] of TDPCompoundAddressElement;
    x, NumElems: Integer; PortNo: Word;
begin
  // TCP/IP-Verbindungen brauchen einen Dienstanbieter,
  // eine (ggf. leere) IP-Adresse und (optional) eine Portnummer
  AddrElems[0].guidDataType := DPAID_ServiceProvider;
  AddrElems[0].dwDataSize := sizeof(TGUID);
  AddrElems[0].lpData := @DPSPGUID_TCPIP;
  // Hngt da eine Portnummer hintendran?
  x := Pos(':',IPAddr);
  if x <> 0 then
  begin
    NumElems := 3; PortNo := StrToInt(Copy(IPAddr,x+1,99));
    System.Delete(IPAddr,x,99);
    AddrElems[2].guidDataType := DPAID_INetPort;
    AddrElems[2].dwDataSize := SizeOf(Word);
    AddrElems[2].lpData := @PortNo;
  end
   else NumElems := 2;

  AddrElems[1].guidDataType := DPAID_INet;
  AddrElems[1].dwDataSize := Length(IPAddr)+1;
  AddrElems[1].lpData := PChar(IPAddr);

  Result := CreateCompoundAddress(AddrElems[0],NumElems);
end;

function CreateIPXAddress: Pointer;  // immer im lokalen Subnetz
var AddrElems: TDPCompoundAddressElement;
begin
  AddrElems.guidDataType := DPAID_ServiceProvider;
  AddrElems.dwDataSize := SizeOf(TGUID);
  AddrElems.lpData := @DPSPGUID_IPX;
  Result := CreateCompoundAddress(AddrElems,1);
end;

function CreateModemAddress(InitString,PhoneNr: String):Pointer;
var AddrElems: Array[0..2] of TDPCompoundAddressElement;
    NumElems: Word;
begin
  // Modemverbindungen brauchen einen Dienstanbieter,
  // einen (ggf. leeren) Init-String und eine Telefonnummer
  AddrElems[0].guidDataType := DPAID_ServiceProvider;
  AddrElems[0].dwDataSize := sizeof(TGUID);
  AddrElems[0].lpData := @DPSPGUID_MODEM;

  if InitString <> '' then
  with AddrElems[1] do
  begin
    guidDataType := DPAID_Modem;
    dwDataSize := Length(InitString)+1; lpData := PChar(InitString);
    NumElems := 3;
  end
    else NumElems := 2;
  with AddrElems[NumElems-1] do
  begin  // Telefonnummer (ASCII)
    guidDataType := DPAID_Phone;
    dwDataSize := Length(PhoneNr)+1; lpData := PChar(PhoneNr);
  end;
  Result := CreateCompoundAddress(AddrElems[0],NumElems);
end;

function CreateSerialAddress(ComData: TDPComportAddress): Pointer;
var AddrElems: Array[0..1] of TDPCompoundAddressElement;
begin
  // Serielle Verbindungen brauchen einen Dienstanbieter (auch
  // wenn das nicht im SDK steht) und eine COMPORT-Struktur mit
  // Portnummer, Geschwindigkeit, Stopbits, Paritt und Flukontrolle
  AddrElems[0].guidDataType := DPAID_ServiceProvider;
  AddrElems[0].dwDataSize := sizeof(TGUID);
  AddrElems[0].lpData := @DPSPGUID_SERIAL;

  AddrElems[1].guidDataType := DPAID_Comport;
  AddrElems[1].dwDataSize := SizeOf(ComData);
  AddrElems[1].lpData := @ComData;

  Result := CreateCompoundAddress(AddrElems[0],2);
end;

function EnumConnections(AppGUID: TGUID): TObjectList;
var DPlayObject: IDirectPlay4;
  function ConnectionsCallback(var lpguidSP: TGUID;
      lpConnection: Pointer; dwConnectionSize: DWORD;
      var lpName: TDPName; dwFlags: DWORD;
      lpContext: Pointer) : BOOL; stdcall;
  var ConnInfo: PDPConnectionInfo;
  begin
    GetMem(ConnInfo,SizeOf(TGUID)+dwConnectionSize);
    ConnInfo^.GuidSP := lpguidSP; Move(lpConnection^,
                        ConnInfo^.DPAddress,dwConnectionSize);
    TObjectList(lpContext).AddObject(
                       lpName.lpszShortName,Pointer(ConnInfo));
    Result := True;
  end;
begin
  Result := TObjectList.Create;
  CoCreateInstance(CLSID_DirectPlay, nil, CLSCTX_INPROC_SERVER,
                   IID_IDirectPlay4A, DPlayObject);
  DPlayObject.EnumConnections(AppGUID, @ConnectionsCallback,
                   Result, DPCONNECTION_DIRECTPLAY);
end;

// ---------- Abzhler (reine Fleiarbeit) ----------------
// XRef: Alle vier Enum-Funktionen
function EnumGPCallback(DPID: TDPID; dwPlayerType: DWORD;
 var lpName: TDPName; dwFlags: DWORD; lpContext: Pointer) : BOOL; stdcall;
var Info: PDPGroupAndPlayerInfo;
begin
  New(Info); Info^.GPID := DPID;
  Info^.GPType := dwPlayerType; Info^.GPFlags := dwFlags;
  TObjectList(lpContext).AddObject(lpName.lpszShortName,Pointer(Info));
  Result := True;
end;

function EnumGroups(DPObject: IDirectPlay4; Flags: Integer): TObjectList;
begin
  Result := TObjectList.Create;
  DPObject.EnumGroups(nil,@EnumGPCallback,Result,Flags);
end;

function EnumGroupsInGroup(DPObject: IDirectPlay4;
  GroupID: TDPID; Flags: Integer): TObjectList;
begin
  Result := TObjectList.Create;
  DPObject.EnumGroupsInGroup(GroupID, nil, @EnumGPCallback, Result, Flags);
end;

function EnumGroupPlayers(DPObject: IDirectPlay4;
   GroupID: TDPID; Flags: Integer): TObjectList;
begin
  Result := TObjectList.Create;
  DPObject.EnumGroupPlayers(GroupID, nil, @EnumGPCallback, Result, Flags);
end;

function EnumPlayers(DPObject: IDirectPlay4; Flags: Integer): TObjectList;
begin
  Result := TObjectList.Create;
  DPObject.EnumPlayers(nil,@EnumGPCallback,Result,Flags);
end;

end.
