unit PongDPU;  // 25-FEB-99 as (Arne Schpers)
{ Das Urmodell aller Videospiele - hier mit DirectDraw
  und DirectPlay. Vorausgesetzt werden die DirectX-Header von
  Erik Unger und DirectX 6.1 - mit DirectX 6 sind asynchrone
  Sendungen (DPSEND_ASYNC) leider eher ein Witz. Fr NT5 Beta 2
  schlicht DPlayX.DLL aus einer 6.1er-Installation kopieren.

  Optionale Kommandozeilenparameter fr die Verbindung:
   IPX, COM1, COM2
  Ohne Parameter (oder mit irgend etwas anderem): TCP/IP
}
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms,
  Dialogs, StdCtrls, Menus,
  DDraw,     // Header von Erik Unger
  DXDrawAS,  // DirectDraw-Komponenten, auf DX 6 aktualisiert
  ActiveX, DPlay, DPLobby,  // Header von Erik Unger
  DXPlayAS, DXPlayASUtils, // DirectPlay-Komponenten
  DXTimer,  // Millisekunden-Timer
  MMSystem,
  PongDPMsgs; // DirectPlay-Debugging (Anzeige, sonst nichts)

const DDrawDebugging: Boolean = False;  // TRUE fr Experimente!

type
  TPongForm = class(TForm)
    MainMenu1: TMainMenu;
    Game1: TMenuItem;
    mNewGame: TMenuItem;
    mNewBall: TMenuItem;
    N1: TMenuItem;
    mQuit: TMenuItem;
    mShowMsgs: TMenuItem;
    mSound: TMenuItem;
    mVBLData: TMenuItem;
    procedure FormCreate(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure FormResize(Sender: TObject);
    procedure FormMouseMove(Sender: TObject;
      Shift: TShiftState; X, Y: Integer);
    procedure mQuitClick(Sender: TObject);
    procedure mNewBallClick(Sender: TObject);
    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure mShowMsgsClick(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure mSoundClick(Sender: TObject);
    procedure mNewGameClick(Sender: TObject);
    procedure mVBLDataClick(Sender: TObject);
  protected  // DirectPlay
    SessionList: TObjectList;
    function ModifyConnectionInfo(const ConnInfo:
      PDPLConnection): Boolean;  // Callback bei Lobby-Verbindung
    function InitDirectPlay: Boolean;
    procedure SessionEnumerationDone(Sender: TObject);
    procedure HandleSystemMsg(Sender: TObject);  // DPlayer
    procedure HandleUserMsg(Sender: TObject);  // DPlayer
    procedure SendRacketPosition(NewY: Single);
    procedure SendBallOrg(NewBall: Boolean);
    procedure SendBallOut;
  protected  // DirectPlay-Debugging (reine Anzeige)
    procedure AddMsg(const Msg: String);
    procedure MsgFormHide(Sender: TObject);
  private  // DirectDraw
    PrimarySurf, PaintSurf: TDDSurface;
    RacketSurf, BallSurf: TDDSurface;
    SurfacesLoaded: Boolean;
    PlayTimer: TDXTimer;
    function InitDirectDraw: Boolean;
    function RestoreSurfaces: Boolean;
    procedure LoadSurfaces(var Loaded: Boolean);
  protected
    procedure OnDeactivateApp(Sender: TObject);
    procedure OnActivateApp(Sender: TObject);
    procedure WMEraseBkGnd(var Msg: TWMEraseBkGnd); message WM_ERASEBKGND;
  protected
    BallRunning: Boolean;  // True: Ball luft
    OurRacket: Integer;  // 0 oder 1
    RacketYs: Array[0..1] of Single;  // 0 bis 1-RacketHeight
    Latency: Cardinal;  // Laufzeit der Nachrichten, beim Host 0
    BallOrgTime: DWord;  // lokale Zeit (timeGetTime)
    BallOrgX, BallOrgY: Single; // Ausgangsposition des Balls
    BallDX, BallDY: Single; // Geschwindigkeit (Einheiten/msec)
    BallX, BallY: Single;  // aktuelle Position
    BallHeight: Single;  // XRef: FormResize, Trefferprfung
    function AdvanceBall: Boolean;  // XRef: OnPlayTimer
    procedure OnPlayTimer(Sender: TObject);  // Zeitraster
  protected  // Soundeffekte (oder so etwas hnliches)
    GotBounce, GotBallOut: Boolean;
    BounceStream, BallOutStream: TMemoryStream;
    procedure PlaySound(GotSound: Boolean; S: TMemoryStream);
  private
    FHosting, FSinglePlayer: Boolean;
    procedure SetSinglePlayer(Value: Boolean);
    procedure SetHosting(Value: Boolean);
  public
    Points: Array[0..1] of Integer;  // Spielstand
    FrameStartTime, FrameCount: DWord;  // DDraw-FPS
    NotInVBL, MaxScanline: Integer;  // DDraw-Messung
    property SinglePlayer: Boolean read FSinglePlayer write SetSinglePlayer;
    property Hosting: Boolean read FHosting write SetHosting;
  end;

const
  BallSize = 1/20;  // Ballradius ist 1/40 der Fensterhhe
  RacketWidth = 1/30;  // Schlgerbreite = 1/30 Fensterbreite
  RacketHeight = 1/4;  // Schlgerhhe = 1/4 Fensterhhe

var PongForm: TPongForm;

implementation
{$R *.DFM}

procedure TPongForm.OnActivateApp(Sender: TObject);
begin
  PlayTimer.Enabled := True;
end;

procedure TPongForm.OnDeactivateApp(Sender: TObject);
begin
  PlayTimer.Enabled := False; Invalidate;  // "Pause"
end;

procedure TPongForm.FormCreate(Sender: TObject);
var DriverCaps, HELCaps: TDDCaps;
  function LoadSound(var S: TMemoryStream; FName: String): Boolean;
  begin
    S := TMemoryStream.Create; Result := True;
    try
      S.LoadFromFile(ExtractFilePath(ParamStr(0))+FName);
    except
      Result := False;
    end;
  end;
begin
  Randomize;  // fr "Kantenpraller"
  // akustische Untermalung - auch zum Check der Synchronizitt
  GotBounce := LoadSound(BounceStream,'sbounce.wav');
  GotBallOut := LoadSound(BallOutStream,'d_bang.wav');
  // DirectPlay-Debugging
  PongDPMsgForm := TPongDPMsgForm.Create(Self);
  with PongDPMsgForm do
  begin  // das gbe bei Autocreate GPFs...
    OnHide := MsgFormHide;
    MsgBox.Clear;
  end;  // unterhalb der eigenen Form, mit gleicher Breite
  PongDPMsgForm.SetBounds(Left,Top+Height,Width,150);
  mShowMsgsClick(Self);  // Form sichtbar

  Hosting := False; SinglePlayer := True; // "New Ball", "New Game" aus
  if not (InitDirectPlay and InitDirectDraw) then
  begin
    PostQuitMessage(0); Exit;
  end;

  PlayTimer := TDXTimer.Create(Self);
  with PlayTimer do
  begin  // 100 Frames/sec (durch VSync begrenzt)
    Interval := 10; OnTimer := OnPlayTimer; Enabled := True;
  end;

  Application.OnDeactivate := OnDeactivateApp;
  Application.OnActivate := OnActivateApp;
  mSoundClick(Self); // Sound an
  // Kann der Treiber die aktuelle Scanline lesen?
  DriverCaps.dwSize := SizeOf(DriverCaps);
  HELCaps.dwSize := SizeOf(HELCaps);
  DirectDraw.GetCaps(DriverCaps,HelCaps);
  if DriverCaps.dwCaps and DDCAPS_READSCANLINE <> 0
    then mVBLDataClick(Self)  // "On"
    else mVBLData.Enabled := False;
end;

procedure TPongForm.FormDestroy(Sender: TObject);
begin
  sndPlaySound(nil,0);  // Stop
  BounceStream.Free; BallOutStream.Free;
end;

// GUID des Programms (Umschalt+Strg+G im Editor)
// ['{6ECD6C81-CE8E-11D2-822B-0080C84C3E5E}']
const PongGUID: TGUID =
 (D1: $6ECD6C81; D2:$CE8E; D3:$11D2;
  D4: ($82,$2B,$00,$80,$C8,$4C,$3E,$5E));

  SessionFlags =  DPSESSION_KEEPALIVE or DPSESSION_MIGRATEHOST
      or DPSESSION_DIRECTPLAYPROTOCOL;  // fr neue Sessions

// --- Verbindungsaufnahme; XRef: FormCreate -------------------
function TPongForm.ModifyConnectionInfo(const ConnInfo:
   PDPLConnection): Boolean;  // Callback bei Lobby-Verbindung
begin
  with ConnInfo^ do
    if dwFlags and DPLCONNECTION_CREATESESSION <> 0 then
    begin
      Hosting := True;
      lpSessionDesc^.dwFlags := SessionFlags;
      lpSessionDesc^.dwMaxPlayers := 2;
    end;
  Result := True;
end;

// Eine ziemlich direkte Kopie aus DelphiChat. Unterschiede:
// - Exceptions anstelle von HALT bei Fehlern
// - Auswahl zwischen IPX, COMx und TCP ber Kommandozeile
function TPongForm.InitDirectPlay: Boolean;
var Res: HResult; CompoundAddress: Pointer;
    ComData: TDPComportAddress;  // fr COM1, COM2
  procedure HardExit(Msg: String);
  begin
    DPlayer.Free; DPlayer := nil;
    raise Exception.Create(Msg);
  end;
begin
  Result := False;
  try
    DPlayer := TDPlayer.Create(Self);
    with DPlayer do
    begin
      OnSystemMsg := HandleSystemMsg;
      OnUserMsg := HandleUserMsg;
      OnModifyConnInfo := ModifyConnectionInfo;
      Res := ConnectUsingLobby;  // immer zuerst probieren!
    end;
    if SUCCEEDED(Res) then
    with DPlayer do
    begin
      AddMsg(Format('Lobby connect: %s, Player "%s"',
        [SessionDesc.lpszSessionName,PlayerName]));
    end else
     if Res <> DPERR_NOTLOBBIED then HardExit(ErrorString(Res))
     else  // DPlay-Objekt und Verbindungsdaten einzeln zusammenbauen
     with DPlayer do
     begin
       Res := CoCreateInstance(CLSID_DirectPlay, nil,
         CLSCTX_INPROC_SERVER, IID_IDirectPlay4A, DPlayObject);
       if FAILED(Res) then
         HardExit('Create DirectPlay Object: '+ ErrorString(Res));
       // Unterscheidung IPX, TCP und Serial ber Kommandozeile
       if ParamStr(1) <> '' then AddMsg('Cmdline: '+ ParamStr(1));
       if ParamStr(1) = 'IPX' then // kurzer Timeout, gut fr
         CompoundAddress := CreateIPXAddress  // Experimente
       else if Pos('COM',ParamStr(1)) = 1 then
       with ComData do
       begin
         if ParamStr(1) = 'COM1' then dwComPort := 1
          else dwComPort := 2;
         dwBaudRate := CBR_57600; dwStopBits := ONESTOPBIT;
         dwParity := NOPARITY; dwFlowControl := DPCPA_NOFLOW;
         CompoundAddress := CreateSerialAddress(ComData);
       end else
       begin // Standardvorgabe: TCP/IP, lokales Subnetz
         AddMsg('Using TCP/IP (local subnet)');
         CompoundAddress := CreateTCPIPAddress('');
       end;
       Res := DPlayObject.InitializeConnection(CompoundAddress^,0);
       if FAILED(Res)
         then HardExit('Establish Connection: '+ErrorString(Res));
       // ber diese Verbindung erreichbare Sessions auflisten
       AddMsg('Searching Sessions...');
       SessionList := TObjectList.Create;
       SessionEnumerator := TSessionEnumerator.Create(Self);
       with SessionEnumerator do
       begin
         DPObject := DPlayObject;
         ApplicationGuid := PongGUID;
         TargetList := SessionList;
         OnEnumDone := SessionEnumerationDone;
         Start;
       end;
     end; // else ist die Verbindung via Lobby zustandegekommen
     Result := True;
  except
    On E: Exception do ShowMessage(E.Message);
  end;
end;

// Hier kommen wg. DPSESSIONS_AVAILABLE automatisch nur die
// Sitzungen an, die weniger als dwMaxPlayers Teilnehmer haben
// Deshalb bis auf Statusmeldungen eine 1:1-Kopie aus DelphiChat
procedure TPongForm.SessionEnumerationDone(Sender: TObject);
var CPName: Array[0..49] of Char;
    CPSize: DWord; ConnFlags: Integer;
begin
  with SessionEnumerator do
  begin
    if (SessionList.Count = 0) and not TimedOut then Exit;
    Stop;  // DPlayObject aus dem Enum-Modus raus!
  end;
  with DPlayer do
  begin
    FillChar(SessionDesc,SizeOf(SessionDesc),0);
    with SessionDesc do
    begin
      dwSize := Sizeof(SessionDesc);
      // optional: lpszPassword setzen
      if SessionList.Count > 0 then
      begin
        guidInstance := PGUID(SessionList.Objects[0])^;
        ConnFlags := DPOPEN_JOIN;
        Hosting := False;
        AddMsg('Joining Session '+SessionList[0]);
      end else
      begin
        guidApplication := PongGUID;
        CPSize := SizeOf(CPName)-1;
        if not GetComputerName(CPName,CPSize) then CPName := 'NONAME';
        lpszSessionName := CPName;
        // fr lpszPassword: + DPSESSION_PASSWORDREQUIRED
        dwFlags := SessionFlags; dwMaxPlayers := 2;
        ConnFlags := DPOPEN_CREATE;
        AddMsg('Creating Session '+CPName);
        Hosting := True;
      end;
    end;
    CheckRes(DPlayObject.Open(SessionDesc,ConnFlags),
             'Session create/connect');
    // Player anlegen
    CPSize := SizeOf(CPName)-1;
    if not GetUserName(CPName,CPSize) then CPName := 'Unknown Player';
  end;
  AddMsg('Creating Player '+CPName);
  CheckRes(DPlayer.CreatePlayer(CPName,nil,0,0),'Create Player:');
end;

// ------------- Nachrichtendienstliches  ---------------
procedure TPongForm.HandleSystemMsg(Sender: TObject);
begin
  with DPlayer, MsgData^ do
   case dwType of
     DPSYS_CREATEPLAYERORGROUP:
     begin
       AddMsg(Format('"%s" joined', [PDPMSG_CREATEPLAYERORGROUP
          (MsgData)^.dpnName.lpszShortName]));
       SinglePlayer := False;
     end;
     DPSYS_DESTROYPLAYERORGROUP:
     begin
       AddMsg(Format('"%s" left', [PDPMSG_DESTROYPLAYERORGROUP
         (MsgData)^.dpnName.lpszShortName]));
       SinglePlayer := True;
     end;
     DPSYS_HOST:
       begin
         AddMsg('This machine is now the host');
         Hosting := True; SinglePlayer := True;
       end;
     else
       AddMsg(Format('System msg, dwType = $%x',[dwType]));
   end;
end;

// ----- Die einzelnen Nachrichtentypen ----------------
const
  PongMsg_TimeSync = 0; PongMsg_RacketPos = 1;
  PongMsg_BallOrg = 2; PongMsg_BallOut = 3;

type
  // zweimal pro Sekunde: Schlgerposition + Latenzermittlung
  TTimeSyncSubType = (HostToClient,ClientToHost,ResultToClient);
  TDPTimeSyncMsg = record
    dwType: Integer;  // PongMsg_TimeSync
    dwSubType: TTimeSyncSubType;
    Rackets: Array[0..1] of Single;  // wie RacketPosMsg
    HostTime,  // lokale Zeit des Hosts
    Latency2: Cardinal;  // Laufzeit der Nachricht (2x)
  end;
  PDPTimeSyncMsg = ^TDPTimeSyncMsg;

  // bei jeder Schlgerbewegung, ohne Empfangsbesttigung
  TDPRacketPosMsg = record
    dwType: Integer;  // PongMSg_RacketPos
    Racket: Integer;  // 0 = links, 1 = rechts
    YPos: Single;   // 0.0 .. 1.0-RacketHeight
  end;
  PDPRacketPosMsg = ^TDPRacketPosMsg;

  // Neues Spiel, neuer Ball, neuer Ball, nderung der
  // Bewegungsrichtung. Fr Ober- und Unterkante nur vom
  // Host gesendet, fr Abprall vom Schlger von beiden
  TBallOrgType = (boNewGame, boNewBall, boChangeDirection);
  TDPBallOrgMsg = record
    dwType: Integer;            // PongMsg_BallOrg
    BallOrgType: TBallOrgType;  // bo...-Konstanten
    SenderLatency: Cardinal;    // 0 beim Host
    _BallOrgX, _BallOrgY: Single;  // neuer Startpunkt
    _BallDX, _BallDY: Single;  // Vektor
  end;
  PDPBallOrgMsg = ^TDPBallOrgMsg;

  // Von dem Spieler gesendet, bei dem der Ball im Aus
  // gelandet ist
  TDPBallOutMsg = record
    dwType: Integer;    // PongMsg_BallOut
    PlayerNo: Integer;  // 0 = links, 1 = rechts (wie Schlger)
  end;
  PDPBallOutMsg = ^TDPBallOutMsg;

var LastSyncTime: DWord;  // Timer im Timer, sozusagen

procedure TPongForm.OnPlayTimer(Sender: TObject);
var MsgRec: TDPTimeSyncMsg; CurTime: DWord; CapStr: String;
begin
  if BallRunning and AdvanceBall then Invalidate;
  // Latenzberechnung und Schlgerpositionen 2mal pro Sekunde
  CurTime := timeGetTime;
  if CurTime > LastSyncTime+500 then
  begin
    LastSyncTime := CurTime;
    // Statusanzeige praktischerweise hier
    CapStr := Format('Pong [%d:%d]', [Points[0], Points[1]]);
    if BallRunning then CapStr := CapStr + Format(' %d fps',
      [(FrameCount*1000) div (timegetTime-FrameStartTime+1)]);
    if not Hosting then CapStr := CapStr +
      Format(' Lat=%d',[Latency]);
    if mVBLData.Checked then CapStr := CapStr +
      Format(' - NotVBL: %d of %d, Maxline %d',
        [NotInVBL, FrameCount, MaxScanline]);
    Caption := CapStr;

    if DPlayer.Connected and not SinglePlayer and Hosting then
    begin
      with MsgRec do
      begin
        dwType := PongMsg_TimeSync;
        dwSubType := HostToClient;
        HostTime := timeGetTime;  // lokale Zeit des Hosts
        Rackets[OurRacket] := RacketYs[OurRacket];
      end;
      // synchron, ohne Empfangsbesttigung
      DPlayer.Send(DPID_ALLPLAYERS, 0, @MsgRec, SizeOf(MsgRec));
    end;
  end;
end;

// Begrenzung, Update und Senden in einem Rutsch. ber diese
// Routine senden sowohl der Host als auch der Client
procedure TPongForm.SendRacketPosition(NewY: Single);
var MsgRec: TDPRacketPosMsg;
begin
  // Begrenzung auf den Wertebereich von 0 bis 1.0-RacketHeight
  if NewY < 0 then NewY := 0
   else if NewY > 1-RacketHeight then NewY := 1-RacketHeight;
  // Neuzeichnen und Benachrichtigung notwendig?
  if Abs(NewY-RacketYs[OurRacket]) * ClientHeight >= 1 then
  begin
    if DPlayer.Connected and not SinglePlayer then
    begin
      with MsgRec do
      begin
        dwType := PongMsg_RacketPos;
        Racket := OurRacket; YPos := NewY;
      end;
      // Asynchron, ohne Besttigung
      DPlayer.Send(DPID_ALLPLAYERS, DPSEND_ASYNC or
         DPSEND_NOSENDCOMPLETEMSG, @MsgRec, SizeOf(MsgRec));
      AddMsg(Format('S: RacketPos %d = %1.2f', [OurRacket,NewY]));
    end;
    Invalidate;  // Neuzeichnen
  end;
  RacketYs[OurRacket] := NewY;
end;

// Neues Spiel, neuer Ball, Kollision
procedure TPongForm.SendBallOrg(NewBall: Boolean);
const AddMsgs: Array[boNewGame..boChangeDirection]
  of String = ('New Game', 'New Ball', 'Change Direction');
var MsgRec: TDPBallOrgMsg;
begin
  with MsgRec do
  begin
    dwType := PongMsg_BallOrg;
    if NewBall then  // neuer Ball mit 0 Punkten: Spielstart
     if Points[0]+Points[1] = 0 then BallOrgType := boNewGame
       else BallOrgType := boNewBall
    else BallOrgType := boChangeDirection;
    SenderLatency := Latency;  // 0 fr den Host
    _BallOrgX := BallOrgX; _BallOrgY := BallOrgY;
    _BallDX := BallDX; _BallDY := BallDY;
  end;
  DPlayer.Send(DPID_ALLPLAYERS, DPSEND_GUARANTEED,
    @MsgRec,SizeOf(MsgRec));
  AddMsg('Send: New Ball Org: '+ AddMsgs[MsgRec.BallOrgType]);
end;

procedure TPongForm.SendBallOut;  // Ball im Aus
var MsgRec: TDPBallOutMsg;
begin
  MsgRec.dwType := PongMsg_BallOut;
  MsgRec.PlayerNo := OurRacket;
  DPlayer.Send(DPID_ALLPLAYERS, DPSEND_GUARANTEED,
    @MsgRec, SizeOf(MsgRec));
  AddMsg('Send: Ball Out');
end;

// ----------- Empfngerseite -----------------------
procedure TPongForm.HandleUserMsg(Sender: TObject);  // DPlayer
begin
  with DPlayer, DPlayObject do
  begin
    case MsgData^.dwType of
      PongMsg_TimeSync:
         with PDPTimeSyncMsg(MsgData)^ do
         begin
           case dwSubType of
             HostToClient:  // kommt nur beim Client an -> Host
               begin
                 Rackets[OurRacket] := RacketYs[OurRacket];
                 RacketYs[1] := Rackets[1];  // Host-Position
               end;
             ClientToHost: // kommt nur beim Host an -> Client
               begin
                 RacketYs[0] := Rackets[0];  // Client-Position
                 Latency2 := timeGetTime-HostTime;
                 HostTime := timeGetTime;
               end;
             ResultToClient:  // kommt nur beim Client an
               // Hier knnte man noch eine Mittelwertbildung
               // der letzten x Werte einbauen
               Latency := Latency2 div 2;
           end;
           if dwSubType < ResultToClient then
           begin
             Inc(dwSubType);
             with DPlayer do
               Send(MsgIDFrom,0, MsgData, MsgDataSize);
           end;
          Invalidate;
         end;
      PongMsg_RacketPos:
        with PDPRacketPosMsg(MsgData)^ do
        begin
          // die andere Seite sendet nur, wenn es wirklich
          // etwas neu zu zeichnen gibt (dY >= 1 Scanline)
          Invalidate;
          RacketYs[Racket] := YPos;
          AddMsg(Format('Rec RacketPos %d: %1.2f',[Racket,YPos]));
        end;
      PongMsg_BallOrg:
        with PDPBallOrgMsg(MsgData)^ do
        begin
          BallOrgTime := DPlayer.MsgTime;
          if Hosting then Dec(BallOrgTime,SenderLatency)
           else Dec(BallOrgTime,Latency);  // Clients
          BallOrgX := _BallOrgX; BallOrgY := _BallOrgY;
          BallDX := _BallDX; BallDY := _BallDY;
          BallRunning := True; PlayTimer.Enabled := True;
          AdvanceBall;  // Recalc!
          if BallOrgType < boNewBall then
          begin
            AddMsg('New Game'); Points[0] := 0; Points[1] := 0;
          end;
          if BallOrgType < boChangeDirection then
          begin  // Zeitmessung
            FrameCount := 0; FrameStartTime := timeGetTime;
            NotInVBL := 0; MaxScanline := -1;  // VBL-Zhlung
            AddMsg('New Ball');
          end;
          AddMsg(Format('Rec: BallX=%1.2f, Y=%1.2f',[BallX, BallY]));
          Invalidate;         // (auch fr neue Blle...)
          PlaySound(GotBounce,BounceStream);
        end;
      PongMsg_BallOut:
        with PDPBallOutMsg(MsgData)^ do
        begin
          AddMsg(Format('Ball out on Player %d',[PlayerNo]));
          BallRunning := False;
          if PlayerNo = 1 then Inc(Points[0])  // Punktestand
            else Inc(Points[1]);
          PlaySound(GotBallOut,BallOutStream);
          Invalidate;
        end;
    end;
  end;
end;

// --- DirectDraw-Ecke, fast ohne (neue) Besonderheiten ----
// XRef: FormCreate, RestoreSurfaces
function TPongForm.InitDirectDraw: Boolean;
begin
  Result := False;
  try
    // primre Oberflche, keine nderung des Videomodus,
    // Clipper fr dieses Fenster,
    DirectDrawInitialize(nil, DDSCL_NORMAL, Handle,
      DDrawDebugging, DDrawDebugging);  // UseSafeCopy, IdleCheck
    // primre Oberflche, Clipper damit verbinden
    PrimarySurf := CreatePrimarySurface(Handle);
    // weitere (in diesem Fall eine) Oberflchen anlegen
    LoadSurfaces(SurfacesLoaded);
    Result := True;
  except
    on E: DDrawException do ShowMessage(E.Message);
  end;
end;

procedure TPongForm.FormResize(Sender: TObject);
begin
  LoadSurfaces(SurfacesLoaded); Invalidate;
end;

procedure TPongForm.LoadSurfaces(var Loaded: Boolean);
  procedure MakeSurf(var Surf: TDDSurface; W,H: Integer;
    FillColor: TColor);
  var R,G,B: Byte; Color: DWord;
  begin
    Surf.Free; Surf := TDDSurface.Create(Self);
    with Surf, SurfDesc do
    begin
      // soll heien: diese Felder sind besetzt
      dwFlags := DDSD_CAPS or DDSD_HEIGHT or DDSD_WIDTH;
      dwWidth := W; dwHeight := H;
      ddsCaps.dwCaps := DDSCAPS_VIDEOMEMORY;
      SurfaceFromSurfDesc;  // kann bei 4MB-Karten schiefgehen
      // mit der gewnschten Farbe fllen
      R := GetRValue(FillColor); G := GetGValue(FillColor);
      B := GetBValue(FillColor);
      with ddpfPixelFormat do
        case dwRGBBitCount of
          8: Color := R + G shl 8 + B shl 16;  // RGB
          16:
            if dwRBitMask = $F800 then // 16 Bit: RGB 565
           Color := (R shr 3) shl 11 + (G shr 2) shl 5 + B shr 3
             else Color := (R shr 3) shl 10  // RGB 555
                + (G shr 3) shl 5 + B shr 3;
          24,32:
            Color := B + G shl 8 + R shl 16;  // BGR
          else
            Color := 0;
        end;
      // Das geht leider nicht ber die Methode Draw, weil die
      // eine Quell-Oberflche erwartet - und das mag wiederum
      // Blt bei COLORFILL nicht...
      BltFX.dwFillColor := Color; BltFX.dwSize := SizeOf(BltFX);
      SurfaceObject.Blt(nil,nil,nil,DDBLT_COLORFILL,@BltFX);
    end;
  end;

begin
  Loaded := False;
  if DirectDraw = nil then Exit; // Fehler beim Start
  try
    // Spielfeld ist der gesamte Anwendungsbereich
    MakeSurf(PaintSurf,ClientWidth,ClientHeight,clGreen);
    // Schlger, werden zweimal verwendet
    MakeSurf(RacketSurf, Trunc(ClientWidth * RacketWidth),
      Trunc(ClientHeight * RacketHeight), clMaroon);
    // Ball
    MakeSurf(BallSurf, Trunc(ClientWidth*BallSize),
      Trunc(ClientWidth*BallSize), clGreen);
    with BallSurf, Canvas do
    begin
      Brush.Color := clWhite; Pen.Color := clBlack;
      Ellipse(0,0,Width,Height);
      ReleaseDC;
    end;
    // Hhe und Breite sind wohl meist unterschiedlich
    BallHeight := BallSurf.Height / ClientHeight;
    Loaded := True;
  except
    // Loaded ist hier bereits FALSE - sonst wrde es
    // bei FormPaint krachen
    On E: Exception do ShowMessage(E.Message);
  end;
end;

// Wiederherstellungsversuch der Oberflchen. XRef: FormPaint
function TPongForm.RestoreSurfaces: Boolean;
var Res: HResult;
begin
  // Versuch, die primre Oberflche wiederherzustellen
  Res := PrimarySurf.SurfaceObject._Restore;
  if Res = DDERR_WRONGMODE then
  begin  // Videomodus gendert: komplette Reinitialisierung
    DirectDrawUninitialize;
    Result := InitDirectDraw;
  end
    else Result := SUCCEEDED(PaintSurf.SurfaceObject._Restore);
end;

procedure TPongForm.WMEraseBkGnd(var Msg: TWMEraseBkGnd);
begin
  if Application.Active and SurfacesLoaded
    then Msg.Result := 1  // wird eh bermalt
    else inherited;  // fr die Ausgabe von "Pause"
end;

procedure TPongForm.FormPaint(Sender: TObject);
var InVerticalBlank: BOOL; CurScanLine: Integer;
begin
  if DirectDraw = nil then Exit;
  // lebt die primre Oberflche noch?
  if FAILED(PrimarySurf.SurfaceObject.IsLost) then
    if not RestoreSurfaces then Exit;

  if not (Application.Active and SurfacesLoaded) then
    with Canvas do
    begin
      Brush.Style := bsClear;
      Font.Name := 'Arial'; Font.Size := 35;
      SetTextAlign(Canvas.Handle,TA_CENTER or VTA_CENTER);
      TextOut(ClientWidth div 2, ClientHeight div 2,'Pause');
    end else
  begin
    with PaintSurf do
    begin
      BltFlags := DDBLT_ASYNC or DDBLT_WAIT;
      // Hintergrund fllen
      SurfaceObject.Blt(nil,nil,nil,DDBLT_COLORFILL or BltFlags,@BltFX);
      // Ball einsetzen, kann ber die Fenstergrenzen hinausgehen
      if BallRunning then DrawClipped(Trunc(BallX*ClientWidth),
        Trunc(BallY*ClientHeight), BallSurf);
      if not SinglePlayer  // linker Schlger
        then Draw(0,Trunc(RacketYs[0]*ClientHeight), RacketSurf);
      Draw(ClientWidth-Integer(RacketSurf.Width),
           Trunc(RacketYs[1]*ClientHeight), RacketSurf);
    end;
    with PrimarySurf do
    begin  // Backbuffer -> Bildschirm
      ClipperWnd := Handle;   // Koordinaten-Umsetzung, Clipper
      BltFlags := DDBLT_ASYNC;  // fr VBlank vllig wurscht?
      // Warten auf den vertikalen Strahlrcklauf des Bildschirms
      DirectDraw.WaitForVerticalBlank(DDWAITVB_BLOCKBEGIN, 0);
      Draw(0,0,PaintSurf);
      // Abfrage *nach* der Zeichenoperation
      DirectDraw.GetVerticalBlankStatus(InVerticalBlank);
      if not InVerticalBlank then
      begin
        DirectDraw.GetScanLine(DWord(CurScanLine));
        Inc(NotInVBL);
        if CurScanline > MaxScanline
           then MaxScanline := CurScanline;
      end;
    end;
  end;
  Inc(FrameCount);
end;

function TPongForm.AdvanceBall: Boolean;
var OldBallX, OldBallY: Single;
    CurTime, TimePassed: DWord; NewOrg, SendOrg: Boolean;

  function CheckRacket(YRacket: Single): Boolean;

    procedure Edge(Top: Boolean);
    begin  // Kantenpraller, schwer unphysikalisch
      BallDX := -BallDX * (1 + Random(10) / 100);
      BallDY := BallDY * (1 + Random(5) / 100);
      if (BallDY > 0) and Top then BallDY := -BallDY;
      if (BallDY < 0) and not Top then BallDY := -BallDY;
      if BallDY = 0 then BallDY := (Random(5)-4) / 10000;
    end;
  begin
    Result := False;
    if (BallY+BallHeight >= YRacket) and
       (BallY <= YRacket+RacketHeight) then
    begin
      Result := True; // Ball im Bereich des Schlgers
      if (BallY+BallHeight / 2 < YRacket+RacketHeight*0.1)
       then Edge(True)
       else if (BallY+BallHeight / 2 > YRacket+RacketHeight*0.9)
         then Edge(False)
         else BallDX := -BallDX;  // einfache Reflektion
    end; // else oben bzw. unten vorbei
  end;
begin
  CurTime := timeGetTime; TimePassed := CurTime-BallOrgTime;
  OldBallX := BallX; OldBallY := BallY;
  NewOrg := False;  // Bewegungsrichtung unverndert
  SendOrg := False; // mu auch nicht gesendet werden
  // Ballbewegung fortschreiben ("koppeln")
  BallY := BallOrgY + TimePassed*BallDY;
  BallX := BallOrgX + TimePassed*BallDX;
  // Abprall an der Ober- und Unterkante. Wird von beiden
  // Spielern in Eigenregie berechnet
  if BallY >= 1-BallHeight then
  begin
    BallDY := -BallDY; BallY := 1-BallHeight; NewOrg := True;
  end else if BallY <= 0 then
  begin
    BallDY := - BallDY; BallY := 0; NewOrg := True;
  end;
  // Trefferprfung fr beide Schlger
  if BallX < RacketWidth then
  begin
    if SinglePlayer then
    begin
      if BallX <= 0 then
      begin
        BallX := 0; BallDX := -BallDX; NewOrg := True;
      end;
    end else if CheckRacket(RacketYs[0]) then
    begin
      BallX := RacketWidth; NewOrg := True;
      // nur fr den eigenen Schlger senden
      SendOrg := OurRacket = 0;
    end;
  end else if (BallX > 1-RacketWidth-BallSize)
     and CheckRacket(RacketYs[1]) then
  begin
    NewOrg := True; BallX := 1-RacketWidth-BallSize;
    SendOrg := OurRacket = 1;  // nur fr den eigenen Schlger!
  end;

  // nderung der Bewegungsrichtung?
  if NewOrg then
  begin
    BallOrgX := BallX; BallOrgY := BallY; BallOrgTime := CurTime;
    if not SinglePlayer and SendOrg then SendBallOrg(False);
    PlaySound(GotBounce,BounceStream);
  end;

  // Ball in "unserem" Aus?
  if ((OurRacket = 0) and (BallX < 0))
   or ((OurRacket = 1) and (BallX > 1 + BallSize)) then
  begin
    // Punkt fr den Gegner
    if OurRacket = 0 then Inc(Points[1])
     else Inc(Points[0]);
    BallRunning := False;
    SendBallOut;
    PlaySound(GotBallOut,BallOutStream);
  end;

  // Neuzeichnen notwendig?
  Result := (Abs(BallX-OldBallX)*ClientWidth >= 1) or
     (Abs(BallY-OldBallY)*ClientHeight >= 1) or not BallRunning;
end;


// --------- Verdrahtung ---------------------
procedure TPongForm.SetSinglePlayer(Value: Boolean);
begin
  FSinglePlayer := Value;
  mNewGame.Enabled := Hosting;  // bei FormCreate erst mal False
  mNewBall.Enabled := Hosting;
  Invalidate;
end;

procedure TPongForm.SetHosting(Value: Boolean);
begin
  FHosting := Value; SinglePlayer := Hosting;
  if Hosting then OurRacket := 1 // rechter Schlger
   else OurRacket := 0;
end;

procedure TPongForm.mQuitClick(Sender: TObject);
begin
  Close;
end;

procedure TPongForm.mNewBallClick(Sender: TObject);
begin
  if (Random(2) = 0) then
  begin // 1 Sekunde, von links nach rechts
    BallDX := 0.001; BallOrgX := 0.25;
  end else
  begin  // von rechts nach links
    BallOrgX := 0.75; BallDX := -0.001;
  end;
  BallDY := (Random(5)-4) / 10000;
  BallX := BallOrgX; BallY := BallOrgY;
  BallRunning := True; PlayTimer.Enabled := True;
  BallOrgTime := timeGetTime;  // das kann nur der Host
  SendBallOrg(True);  // = neuer Ball und/oder neues Spiel
  FrameStartTime := timeGetTime; FrameCount := 0;
  NotInVBL := 0; MaxScanline := -1;  // VBL-Zhlung
end;

procedure TPongForm.mNewGameClick(Sender: TObject);
begin
  Points[0] := 0; Points[1] := 0; mNewBallClick(Self);
end;

procedure TPongForm.mSoundClick(Sender: TObject);
begin
  with mSound do
  begin
    Checked := not Checked;
    if Checked then Caption := '&Sound: On'
      else Caption := '&Sound: Off';
  end;
end;

// --- Steuerung der Schlger ---------------------------------
procedure TPongForm.FormKeyPress(Sender: TObject; var Key: Char);
begin
  case Key of
   'E','e': SendRacketPosition(RacketYs[OurRacket] - 0.1);
   'X','x': SendRacketPosition(RacketYs[OurRacket] + 0.1);
  end;
end;

procedure TPongForm.FormMouseMove(Sender: TObject;
  Shift: TShiftState; X, Y: Integer);
begin
  Dec(Y, RacketSurf.Height div 2);  // Maus in der Mitte
  SendRacketPosition(Y / ClientHeight);
end;

// ----------- Anzeige von DirectPlay-Nachrichten -----------
procedure TPongForm.AddMsg(const Msg: String);
begin
  if mShowMsgs.Checked then
  with PongDPMsgForm.MsgBox do
  begin
    Items.Add(Msg); ItemIndex := Items.Count-1;
  end;
end;

const MenuTitles: Array[False..True] of String = ('Show &Msgs','Hide &Msgs');
procedure TPongForm.mShowMsgsClick(Sender: TObject);
begin
  with mShowMsgs do
  begin
    Checked := not Checked;
    Caption := MenuTitles[Checked];
    if Checked then PongDPMsgForm.Show else PongDPMsgForm.Hide;
  end;
end;

procedure TPongForm.MsgFormHide(Sender: TObject);
begin  // gegen mShowMsgsClick(Self) hat die VCL was
  if not (csDestroying in ComponentState) then
  with mShowMsgs do
  begin
    Checked := PongDPMsgForm.Visible;
    Caption := MenuTitles[Checked];
  end;
end;

procedure TPongForm.PlaySound(GotSound: Boolean; S: TMemoryStream);
begin
  if GotSound and mSound.Checked then
  begin
    sndPlaySound(nil,0);
    sndPlaySound(PChar(S.Memory),SND_ASYNC or SND_MEMORY);
  end;
end;

procedure TPongForm.mVBLDataClick(Sender: TObject);
begin
  with mVBLData do
  begin
    Checked := not Checked;
    if Checked then Caption := '&VBlank data: On'
      else Caption := '&VBlank data: Off';
  end;
end;

end.
