/* omsearch.c
   Quicksort und Optimal-Mismatch-Algorithmus in ANSI-C
   (c) c't 3/95 & Guido Gronek                                 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include "omsearch.h"

typedef size_t shifttab_t[UCHAR_MAX+1];    /* Shifttabellentyp */

static shifttab_t Td1;                    /* eine Shifttabelle */
static PAT pattern[MAXPAT];           /* ein geordnetes Muster */
static size_t Plen;                            /* Musterlaenge */

static int Cfreq[256] =                /* Zeichenhaeufigkeiten */
{ 0,0,0,0,0,0,0,0,0,0,1970,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,3,0,0,0,0,0,14964,75,0,0,1455,0,0,0,1,1,0,0,1577,
  100,680,727,126,761,499,296,233,191,165,144,136,131,
  235,221,0,0,0,90,0,181,175,54,271,204,142,433,280,
  79,217,108,144,212,85,32,103,2,68,373,151,204,143,
  289,0,0,76,89,47,89,0,0,27,3845,1363,2037,4353,12599,
  771,1892,3814,5730,92,547,1996,1669,7890,1499,314,
  2,5284,4350,4184,3187,453,1166,2,22,660,0,0,1,0,0,
  0,423,0,0,238,0,0,0,0,0,0,0,0,0,13,0,0,0,0,0,190,0,
  0,0,0,3,10,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  302,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
  0,0,0,0,0,0,0
};

/* Fuer besonders pingelige Compiler beim qsort()              */
typedef int (*cmpfunc_t)(const void *, const void *);

/* Vergleichfunktion, basierend auf Zeichenhaeufigkeit         */
int om_cmp(PAT *p1, PAT *p2)
{ int df = Cfreq[p1->p_c] - Cfreq[p2->p_c];
  return df ? df : (p2->p_index - p1->p_index);
}

/* Preprocessing-Funktion von OM
   Generiert die Shifttabelle und das geordnete Muster.
   In:  das Muster
   Out: Adresse der Shift-Tabelle oder 0 bei Fehler            */
PAT *init_omsearch(const char *pstr)
{ int c, nw;                 /* akt. Zeichen, Wildcard-Zaehler */
  size_t m;                                         /* Zaehler */
  const char *p;                                 /* Laufzeiger */
  size_t * shift,               /* Zeiger auf die Shifttabelle */
           deltamax=0;                /* max. Shift, falls > 0 */
  PAT *pat;                    /* Zeiger auf geordnetes Muster */

  for(p=pstr; *p; ++p) ;
  if((m=p-pstr) > MAXPAT) return 0;
  Plen = m;                               /* Musterleange Plen */

  /* Initialisiere Shifttabelle mit Plen+1 */
  for(++m, shift=Td1, c=UCHAR_MAX+1; --c >=0;) *shift++ = m;

  /* Muster scannen, dabei Shifttabelle anpassen
     und geordnetes Muster aufbauen                            */
  for(nw=0, shift=Td1, p=pstr, pat=pattern; --m, c=*p; ++p)
  { if ( *p != '?' )             /* Zeichen ist kein Wildcard, */
    { pat->p_index = p-pstr-Plen;     /* wird also uebernommen */
      pat->p_c=c;
      ++pat;
      shift[c]=m;
    }
    else                               /* Zeichen ist Wildcard */
    { deltamax=m;                   /* und begrenzt die Shifts */
      ++nw;
    }
  }
  pat->p_c = 0;
  if ( deltamax )                          /* Shifts begrenzen */
    for(shift=Td1, c=UCHAR_MAX+1; --c >= 0; ++shift)
      if(*shift > deltamax) *shift=deltamax;
  /* geordnetes Muster erzeugen */
  qsort(pattern, Plen-nw, sizeof(PAT), (cmpfunc_t)om_cmp);
  return pattern;
}

/* Quicksearch-Suche nach einem Muster in einem Text
   In:  Pointer auf den Text und das Muster sowie die Laenge
        des Textes, falls bekannt, sonst 0
   Out: Adresse der Fundstelle im Text oder 0                  */
char *qsearch(const char *text, const char *pstr, size_t tlen)
{ const char *p, *t, *tx;
  const char *txe;                          /* Ende des Textes */
  char c;
  size_t m, *td1=Td1;

  if(pstr==0 || text==0) return 0;
  m=Plen;
  if(tlen > 0)                  /* Textlaenge wurde uebergeben */
    txe=text+tlen;
  else
  { for(txe=text; *txe++;);        /* ... oder wird berechnet. */
    --txe;
  }
  tx=text;
  while(tx+m <= txe)
  { p = pstr; t = tx;
    do
    { if((c = *p++) == 0)         /* Muster komplett gescannt, */
        return (char *)tx;                      /* also Match! */
    } while(c == *t++);
    tx += td1[*(tx+m)];    /* Shift zur naechsten Textposition */
  }
  return 0;
}

/* Optimal-Mismatch-Suche nach einem Muster
   In:  Pointer auf den Text, das geordnete Muster sowie die
        Textlaenge, falls bekannt, sondt 0
   Out: Adresse der Fundstelle im Text oder 0                  */
char *omsearch(const char *text, const PAT *pat, size_t tlen)
{ const char *tx;                             /* Laufvariablen */
  char c;
  int i;
  const PAT *p;
  const char *txe;                                 /* Textende */
  size_t *td1 = Td1;                          /* Shift-Tabelle */

  if(text==0) return 0;
  if((c = pat->p_c) ==0 ) return (char *)text;
  i=pat->p_index; ++pat;
  if (tlen > 0)                 /* Textlaenge wurde uebergeben */
    txe = text + tlen;
  else                             /* ... oder wird berechnet. */
  { for(txe = text; *txe++;);
    --txe;
  }
  tx=text+Plen;
  while(tx <= txe)
  { while(c != *(tx+i))                    /* Suche 1. Zeichen */
    { tx += td1[*tx];
      if(tx >txe) return 0;
    }
    for(p = pat; p->p_c; ++p)         /* Suche weitere Zeichen */
      if(p->p_c != *(tx + p->p_index))
        goto mismatch;                 /* Mismatch aufgetreten */
    return (char *)(tx-Plen);                       /* Match ! */
mismatch:
    tx += td1[*tx]; }      /* Shift zur naechsten Textposition */
  return 0;
}
