unit DDrawDemoU;  // 06-OCT-98 as (Arne Schpers)
// Alpha-Blending zwischen zwei Bitmaps, Vergleich zwischen
// Get/SetDIBits und DirectDraw-Oberflchen. Vorausgesetzt
// wird die Unit DDraw von Erik Unger in der von mir
// um IFDEF DIRECTX5 erweiterten Version.

interface
{$DEFINE DIRECTX5}   // sonst ist DirectX 6 gefordert!
uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls,
  JPEG,  // <- nur zum Umbau der JPGs in Bitmaps
  MMSystem,  // <- Zeitmessung
  DDraw;

type
  TForm1 = class(TForm)
    bPic1ToPic2: TButton;    // berblendung Bild 1 nach 2
    bPic2ToPic1: TButton;    // dito, in umgekehrte Richtung
    ScrollBar1: TScrollBar;  // Position = Alpha
    rSelect: TRadioGroup;    // Pixel vs. DIB vs. DDraw
    Label1: TLabel;          // "Frametime (misec)"
    lFrameTime: TLabel;      // Zeit in msec pro Bild
    procedure FormCreate(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FormPaint(Sender: TObject);
    procedure ScrollBar1Change(Sender: TObject);
    procedure bPic2ToPic1Click(Sender: TObject);
    procedure bPic1ToPic2Click(Sender: TObject);
    procedure rSelectClick(Sender: TObject);
  private
    Alpha: Integer;         // berblend-Faktor (0..128)
    PositionStep: Integer;  // (IFDEF FASTER): 2, sonst 1
    procedure GetSetPixel;
    procedure GetSetDIBits;
    procedure GetSetDDrawBits;
  private
    Pic1, Pic2: TBitmap;  // berblend-Bilder
    WorkPic: TBitmap;
  private  // DirectDraw
    DDrawObject: IDirectDraw2;  // das zentrale Objekt
    DDPrimarySurf: IDirectDrawSurface;   // primre Oberflche
    DDrawClipper: IDirectDrawClipper;  // fr DDPrimarySurf
    // DDraw-Oberflchen, analog zu den Bitmaps
    Pic1Surface, Pic2Surface, WorkPicSurface: IDirectDrawSurface;
    // DDraw-Oberflcheneigenschaften, analog zu den Bitmaps
    Pic1Desc, Pic2Desc, WorkPicDesc: TDDSurfaceDesc;
  private
    function InitDirectDraw: Boolean;
    procedure ExitDirectDraw;  // Aufrumen
    function LoadSurfaces: Boolean;  // Bitmaps -> Oberflchen
    function RestoreSurfaces: Boolean;  // Reinitialisierung
  end;

const PaintYPoint = 100;  { Oberkante beim Zeichnen der Bilder }
var Form1: TForm1;

implementation
{$R *.DFM}

// Ldt ein JPG und liefert ein Bitmap zurck
function JPGToBMP(FName: String): TBitmap;
var JPGDec: TJPEGImage;
begin
  Result := TBitmap.Create; JPGDec := TJPEGImage.Create;
  JPGDec.LoadFromFile(FName); Result.Assign(JPGDec);
{$IFDEF FASTER} // wenn's schneller gehen soll... (Hack)
  Result.Canvas.Draw(0,-50,Result);
  Result.Height := Result.Height - 200;
  Form1.PositionStep := 2;
{$ENDIF}
  JPGDec.Free;
end;

procedure TForm1.FormCreate(Sender: TObject);
begin
  PositionStep := 1;
  Pic1 := JPGToBMP(ExtractFilePath(ParamStr(0))+'Milena1.jpg');
  Pic2 := JPGToBMP(ExtractFilePath(ParamStr(0))+'m&m.jpg');
  // Ziel-Bitmap fr die berblendung per SetPixel und DIBits
  WorkPic := TBitmap.Create; WorkPic.Assign(Pic1);
  // Form-Hhe = Controls plus Platz fr das Bild
  ClientHeight := PaintYPoint+Pic1.Height;
  with ScrollBar1 do
  begin
    Position := 0; Max := 128; LargeChange := 26;  // ca. 1/4
  end;
  // Grundeinstellung: Blending per GetSetDIBits
  rSelect.ItemIndex := 1; rSelectClick(Self);
  if not InitDirectDraw  // schade, aber...
    then rSelect.Items.Delete(2);
end;

procedure TForm1.FormDestroy(Sender: TObject);
begin
  Pic1.Free; Pic2.Free; WorkPic.Free;
  ExitDirectDraw;
end;

procedure TForm1.FormPaint(Sender: TObject);
var TargetRect, SrcRect: TRect; BltFX: TDDBltFX; Res: HResult;
begin
  case rSelect.ItemIndex of
    0, // GetSetPixel und GetSetDIBits: WorkPic (TBitmap)
    1: Canvas.Draw(0,100, WorkPic);
    2:
    begin
      // lebt die primre Oberflche noch?
      if FAILED(DDPrimarySurf.IsLost) then
        if not RestoreSurfaces then Exit;
      // DDraw arbeitet mit Bildschirm-Koordinaten
      SrcRect := Rect(0,0,Pic1.Width,Pic1.Height);
      TargetRect := SrcRect;  // kein Stretching, bitte...
      OffsetRect(TargetRect,0,PaintYPoint);
      MapWindowPoints(Handle,0,TargetRect,2);
      // Spezialeffekte beim Blitting: keine, aber Erik hat
      // diesen Parameter nicht als Zeiger definiert
      FillChar(BltFX, SizeOf(BltFX),0);
      BltFX.dwSize := SizeOf(BltFX);
      // WorkPic -> Desktop, ggf. warten, bis Blitter frei
      DDPrimarySurf.Blt(@TargetRect, WorkPicSurface,
        @SrcRect, DDBLT_WAIT, BltFX);
    end;
  end;  { case }
end;

// -------- DirectDraw-Init und -Exit ------------------
function TForm1.InitDirectDraw: Boolean;
var
  Res: HResult;
  SurfDesc: TDDSurfaceDesc;  // Oberflchenbeschreibung
  PixelFormat: TDDPixelFormat;  // ebendies
  DDraw1Object: IDirectDraw;
begin
  Result := False;
  // Anlegen des DirectDraw-Objekts fr das primre Gert
  // (1. Parameter = nil). Hier kommt erst einmal die
  // Originalversion der DirectDraw-Schnittstelle heraus
  Res := DirectDrawCreate(nil, DDraw1Object, nil);
  if FAILED(Res) then
  begin
    ShowMessage('DirectDrawCreate: '+ErrorString(Res)); Exit;
  end;
  // Abfrage dieses Objekts nach IDirectDraw2. Nur zur Demon-
  // stration, weil das Programm mit der Urversion auskommt
  Res := DDraw1Object.QueryInterface(IID_IDirectDraw2,DDrawObject);
  if FAILED(Res) then
  begin
    ShowMessage('DirectDraw.QueryInterface: '+ErrorString(Res));
    Exit;
  end;
  DDraw1Object := nil;  // wird nicht mehr gebraucht
  // Festlegen der Kooperationsebene (normal, d.h. keine
  // nderung des Videomodus) mit dem Handle der Form
  Res := DDrawObject.SetCooperativeLevel(Handle, DDSCL_NORMAL);
  if FAILED(Res) then
  begin
    ShowMessage('SetCooperativeLevel: '+ErrorString(Res)); Exit;
  end;
  // Ausfllen einer Beschreibung fr die primre Oberflche
  FillChar(SurfDesc, SizeOf(SurfDesc), 0);
  with SurfDesc do
  begin
    dwSize := SizeOf(SurfDesc);  // DX-blich: Grenangabe
    dwFlags := DDSD_CAPS;  // heit: Feld dwCaps ist besetzt
    ddsCaps.dwCaps := DDSCAPS_PRIMARYSURFACE;
  end;
  // Anlegen der primren Oberflche
  Res := DDrawObject.CreateSurface(SurfDesc, DDPrimarySurf, nil);
  if FAILED(Res) then
  begin
    ShowMessage('Create primary Surface: '+ErrorString(Res));
    Exit;
  end;
  // Clipper anlegen und mit der primren Oberflche verbinden
  Res := DDrawObject.CreateClipper(0, DDrawClipper, nil);
  if FAILED(Res) then
  begin  // kein Exit - es geht auch ohne!
    ShowMessage('CreateClipper: '+ ErrorString(Res));
  end else
  begin  // Clipper mit dem Fenster der Form verbinden
    Res := DDrawClipper.SetHWnd(0, Handle);
    if FAILED(Res) then
    begin
      ShowMessage('DDrawClipper.SetHWnd: '+ErrorString(Res));
    end else
    begin
      // Clipper mit der primren Oberflche verbinden. Zeichen-
      // aktionen auf dieser Oberflche sollen gegebenenfalls
      // begrenzt werden, weil andere Fenster sonst leiden
      Res := DDPrimarySurf.SetClipper(DDrawClipper);
      if FAILED(Res) then
      begin
        ShowMessage('SetClipper: '+ErrorString(Res));
      end else
      begin  // dabei kommt eine berflssige Referenz heraus...
        DDrawClipper._Release;
      end;
    end;
  end;
  // Pixelformat der primren Oberflche (= Videomodus) ermitteln
  FillChar(PixelFormat, SizeOf(PixelFormat), 0);
  PixelFormat.dwSize := SizeOf(PixelFormat);
  Res := DDPrimarySurf.GetPixelFormat(PixelFormat);
  if FAILED(Res) then
  begin
    ShowMessage('GetPixelFormat: ' + ErrorString(Res));
    Exit;  // das ist wiederum terminaler Natur
  end;
  with PixelFormat do
  begin
    // Palettenmodus?
    if dwFlags and DDPF_PALETTEINDEXED8 <> 0 then
    begin
      ShowMessage('Mit Paletten kann dieses Programm '+
        'leider nicht umgehen');
      Exit;
    end;
    if dwFlags and DDPF_RGB = 0 then
    begin  // Mehr als unwahrscheinlich, aber...
      ShowMessage('Desktop nicht im RGB-Modus? (Wow...)');
      Exit;
    end;
  end;
  // Soweit alles OK; Arbeits-Oberflchen anlegen
  Result := LoadSurfaces;
end;

procedure TForm1.ExitDirectDraw;
begin
  // Reihenfolge scheint obligatorisch zu sein: Erst Bitmaps,
  // dann Clipper, primre Oberflche und zum Schlu das
  // DDraw-Objekt. Ansonsten gibt's beim Abrumen der Form
  // einen GPF (an dem vermutlich der Clipper schuld ist)
  Pic1Surface := nil; Pic2Surface := nil; WorkPicSurface := nil;
  DDrawClipper := nil; DDPrimarySurf := nil;
  DDrawObject := nil;
end;

// Laden der Oberflchen. XRef: DSERR_SURFACELOST bei Paints
function TForm1.LoadSurfaces: Boolean;

  // Bitmap -> DirectDraw-Oberflche, braucht DDrawObject
  function SurfaceFromBMP(BMP: TBitmap;  // Quelle
    var Surface: IDirectDrawSurface;  // Objekt, Beschreibung
    var SurfDesc: TDDSurfaceDesc; VMem: Boolean): Boolean;
  var SurfDC: HDC; Res: HResult;
  begin
    Result := False;
    // Oberflchenbeschreibung initialisieren
    FillChar(SurfDesc, SizeOf(SurfDesc), 0);
    with SurfDesc do
    begin
      dwSize := SizeOf(SurfDesc);  // DX-bliche Grenangabe
      // soll heien: diese Felder sind besetzt
      dwFlags := DDSD_CAPS or DDSD_HEIGHT or DDSD_WIDTH;
      // Haupt- oder Bildspeicher?
      if VMem then ddsCaps.dwCaps := DDSCAPS_OFFSCREENPLAIN
        else ddsCaps.dwCaps := DDSCAPS_SYSTEMMEMORY;
      // Breite und Hhe in Pixeln
      dwWidth := BMP.Width; dwHeight := BMP.Height;
    end;
    Res := DDrawObject.CreateSurface(SurfDesc, Surface, nil);
    if FAILED(Res) then
    begin
      ShowMessage('BMPToSurface: '+ErrorString(Res)); Exit;
    end;
    // SurfDesc mit den kompletten Informationen besetzen
    Res := Surface.GetSurfaceDesc(SurfDesc);
    if FAILED(Res) then
    begin
      ShowMessage('BMPToSurface: '+ErrorString(Res)); Exit;
    end;
    // Gertekontext abholen; Bitmap -> Oberflche
    Res := Surface.GetDC(SurfDC);
    if SUCCEEDED(Res) then
    begin
      BitBlt(SurfDC,0,0,BMP.Width,BMP.Height,
                    BMP.Canvas.Handle,0,0,SRCCOPY);
      Res := Surface.ReleaseDC(SurfDC);
    end;
    if FAILED(Res)
      then ShowMessage('BMPToSurface: '+ErrorString(Res))
      else Result := True;
  end;

begin
  // Anlegen von drei Oberflchen im Bildspeicher - analog
  // zu den drei Bitmaps. Pic1 und Pic2: Hauptspeicher,
  // WorkPic im Bildspeicher
  Result := SurfaceFromBMP(Pic1,Pic1Surface,Pic1Desc, False) and
    SurfaceFromBMP(Pic2,Pic2Surface,Pic2Desc, False) and
    SurfaceFromBMP(Pic1,WorkPicSurface, WorkPicDesc, True);
end;

// Wiederherstellungsversuch der Oberflchen. XRef: FormPaint
function TForm1.RestoreSurfaces: Boolean;
var Res: HResult;
begin
  Caption := 'DDrawDemo2: Restore!';
  // Versuch, die primre Oberflche wiederherzustellen
  Res := DDPrimarySurf._Restore;
  if Res = DDERR_WRONGMODE then
  begin  // Videomodus gendert: komplette Reinitialisierung
    ExitDirectDraw; Result := InitDirectDraw;
  end else if FAILED(Res) then
  begin  // Hack: weg von DDraw, gibt sonst Endlosschleifen
    rSelect.ItemIndex := 0; rSelectClick(Self);
    ShowMessage('Restore: '+ErrorString(Res));
    Result := False;
  end else  // OK, Re-Initialisierung der Oberflchen
    Result := LoadSurfaces;
end;

// ---- Steuerlogik der Form (minimalistisch) -------------
procedure TForm1.ScrollBar1Change(Sender: TObject);
begin
  Alpha := ScrollBar1.Position;  { 0..128 }
  case rSelect.ItemIndex of
   0: GetSetPixel;
   1: GetSetDIBits;
   2: GetSetDDrawBits;
  end;
end;

procedure TForm1.bPic2ToPic1Click(Sender: TObject);
begin  { von Pic1 nach Pic2 berblenden }
  with ScrollBar1 do
    while Position > Min do
    begin
      Position := Position - PositionStep;
      Application.ProcessMessages;
    end;
end;

procedure TForm1.bPic1ToPic2Click(Sender: TObject);
begin  { von Pic2 nach Pic1 berblenden }
  with ScrollBar1 do
    while Position < Max do
    begin
      Position := Position + PositionStep;
      Application.ProcessMessages;
    end;
end;

procedure TForm1.rSelectClick(Sender: TObject);
begin
  { fortlaufende berblendungen mit GetSetPixel - danke nein }
  bPic1ToPic2.Enabled := rSelect.ItemIndex <> 0;
  bPic2ToPic1.Enabled := bPic1ToPic2.Enabled;
  Invalidate;
end;

// ---------- Alpha-Blending -------------------------
var AlphaTable24: Array[-255..255] of Integer;

procedure BuildAlphaTable24(Alpha: Integer);
var x: Integer;
begin
  for x := -255 to 255 do
    AlphaTable24[x] := (Alpha*x) div 128;
end;

// Mischen von RGB-Wert 1 mit RGB-Wert 2
function MixRGB(Val1,Val2: Integer): Integer;
begin
{ Die Standardformel fr einen auf 0..1 normierten Alpha-Wert:
  Result := Alpha * Val2 + (1-Alpha) * Val1;
  Fr Alpha von 0..128 dann:
  Result := (Alpha * Val2 + (128-Alpha) * Val1) div 128;
  Umgestellt:
  Result := Alpha* (Val2-Val1) div 128 + Val1;
  Der Wertebereich von Val1 und Val2 ist 0..255, mgliche Ergeb-
  nisse von Val2-Val1 liegen also im Bereich von -255..+255. ber
  ein entsprechend vorausberechnetes Array wird aus den Multi-
  plikationen und Divisionen (bzw. Shifts fr Werte wie 128)
  eine einfache - vom Prozessor skalierte - Indizierung.
}
  Result := AlphaTable24[Val2-Val1] + Val1;
end;

// Die einfachste Mglichkeit - braucht fast zehn Sekunden
procedure TForm1.GetSetPixel;
var x,y: Integer; RGB1, RGB2, RGBMix: TColor;
    StartTime: Integer;
begin
  StartTime := timeGetTime;
  BuildAlphaTable24(Alpha);
  for y := 0 to Pic1.Height-1 do
  begin
    for x := 0 to Pic1.Width-1 do
    begin
      RGB1 := GetPixel(Pic1.Canvas.Handle,x,y);
      RGB2 := GetPixel(Pic2.Canvas.Handle,x,y);
      RGBMix := RGB(
        MixRGB(GetRValue(RGB1), GetRValue(RGB2)),
        MixRGB(GetGValue(RGB1), GetGValue(RGB2)),
        MixRGB(GetBValue(RGB1), GetBValue(RGB2)));
      SetPixel(WorkPic.Canvas.Handle,x,y,RGBMix);
    end;
    // Damit hier keiner Abstrze vermutet...
    if y mod 10 = 0 then Canvas.Draw(0,PaintYPoint,WorkPic);
  end;
  lFrameTime.Caption := IntToStr(timeGetTime-StartTime);
  Refresh;  
end;

// RGB-Blending; alle drei Bitmaps mit gleicher Gre
procedure AlphaBlendDIB24(BMWidth, BMHeight: Integer;
    Bits1, Bits2, MixBits: PByte);
var x,y,z: Integer;
begin
  for y := 1 to BMHeight do
    for x := 1 to BMWidth do
      for z := 1 to 3 do
      begin // Direktberechnung spart gegenber Aufrufen
            // um die 20 msec - trotz Delphis Optimierungen
        MixBits^ := AlphaTable24[Bits2^-Bits1^] + Bits1^;
        Inc(MixBits); Inc(Bits1); Inc(Bits2);
      end;
end;

// Die DirectX-freie Methode: DIBits abholen, umrechnen und
// wieder in WorkPic einsetzen
procedure TForm1.GetSetDIBits;
var BMInfo1, BMInfo2, BMInfoMix: PBitmapInfo;
    BMInfoSize: Integer;
    BMBits1, BMBits2, BMMixBits: Pointer;
    BMBitSize: Integer;  // alle Bitmaps haben dieselbe Gre
    StartTime: Integer;

  // Ermittelt das Format eines Bitmaps und liefert die Bits
  function _GetDIBits(BMHandle: HBitmap;
                      var BMInfo: PBitmapInfo): Pointer;
  begin
    GetMem(BMInfo,BMInfoSize); FillChar(BMInfo^,BMInfoSize,0);
    BMInfo^.bmiHeader.biSize := Sizeof(TBitmapInfoHeader);
    if GetDIBits(Canvas.Handle, BMHandle, 0, 0, nil,
            BMInfo^, DIB_RGB_COLORS) = 0
      then raise Exception.Create('Fehler bei GetBitmapInfo');
    with BMInfo^.bmiHeader do
    begin
      BMBitSize := (biWidth*biHeight*biBitCount+31) div 8;
      GetMem(Result,BMBitSize);
      if GetDIBits(Canvas.Handle, BMHandle, 0, biHeight,
        Result, BMInfo^, DIB_RGB_COLORS) = 0
        then raise Exception.Create('Fehler bei GetDIBits');
    end;
  end;

begin
  StartTime := timeGetTime;
  BMInfoSize := SizeOf(TBitmapInfo)+256*SizeOf(TRGBQUAD);
  BMBits1 := _GetDIBits(Pic1.Handle, BMInfo1);
  BMBits2 := _GetDIBits(Pic2.Handle, BMInfo2);
  BMMixBits := _GetDIBits(WorkPic.Handle, BMInfoMix);
  case BMInfo1^.bmiHeader.biBitCount of
    8: begin  // Paletten - nein danke
         rSelect.Items.Delete(1);
       end;   
    24:
      begin
        BuildAlphaTable24(Alpha);
        AlphaBlendDIB24(Pic1.Width, Pic1.Height,
                        BMBits1, BMBits2, BMMixBits);
      end;
    else { 15, 16, 32: }
      raise Exception.Create('Mit DDBs geht''s auch nicht ' +
         'schneller, deswegen fehlt die Implementation.');
  end;
  if SetDIBits(Canvas.Handle, WorkPic.Handle, 0, WorkPic.Height,
    BMMixBits, BMInfoMix^, DIB_RGB_COLORS) = 0
      then raise Exception.Create('Fehler bei SetDIBits');
  Canvas.Draw(0,PaintYPoint, WorkPic);
  // Aufrumen
  FreeMem(BMInfo1, BMInfoSize); FreeMem(BMInfo2, BMInfoSize);
  FreeMem(BMInfoMix, BMInfoSize);
  FreeMem(BMBits1, BMBitSize); FreeMem(BMBits2, BMBitSize);
  FreeMem(BMMixBits, BMInfoSize);
  lFrameTime.Caption := IntToStr(timeGetTime-StartTime);
end;

// Alpha-Blending mit DirectDraw fr 15 und 16 bpp
procedure AlphaBlendDDraw1516(Bits1,Bits2,MixBits: PWord;
  SDesc1, SDesc2, TargetDesc: TDDSurfaceDesc);
var x,y: Integer; LineBytes: Integer;
  // AlphaTable24 mit entsprechend verschobenen Bitpositionen
  BTable, GTable, RTable: Array[-255..255] of Integer;
  R1, G1, B1, RMask, BMask, GMask: Integer;
  GShift: Integer;   // 32 fr RGB555, 64 fr RGB565
begin
  with SDesc1 do
  begin
    LineBytes := dwWidth*2;
    RMask := ddpfPixelFormat.dwRBitMask;
    BMask := ddpfPixelFormat.dwBBitMask;
    GMask := ddpfPixelFormat.dwGBitMask;
  end;
  // auf die Angabe "16 Bit" ist kein Verla!
  if RMask = $F800 then GShift := 64
    else GShift := 32;  // RGB555: $7C00
  for x := -255 to 255 do
  begin  // hochgeschobene Alpha-Tabellenwerte
    BTable[x] := AlphaTable24[x];  // redundant
    GTable[x] := AlphaTable24[x]* GShift;
    RTable[x] := AlphaTable24[x]* 32 * GShift;  // "RShift"
  end;
  for y := 1 to SDesc1.dwHeight do
  begin
    // Zwei getrennte Schleifen, damit der Compiler die divs
    // als Shift-Befehle mit Konstanten codieren kann.
    // Variablen gehen hier aber grausam auf die Performance
    if GShift = 32 then
      for x := 1 to SDesc1.dwWidth do
      begin
        B1 := Bits1^ and BMask; G1 := Bits1^ and GMask;
        R1 := Bits1^ and RMask;
        MixBits^ :=
          (BTable[(Bits2^ and BMask) - B1] + B1)
          or (GTable[((Bits2^ and GMask) - G1) div 32] + G1)
          or (RTable[((Bits2^ and RMask) - R1) div (32*32)] + R1);
        Inc(MixBits); Inc(Bits1); Inc(Bits2);
       end
    else
      for x := 1 to SDesc1.dwWidth do
      begin
        B1 := Bits1^ and BMask; G1 := Bits1^ and GMask;
        R1 := Bits1^ and RMask;
        MixBits^ :=
          (BTable[(Bits2^ and BMask) - B1] + B1)
          or (GTable[((Bits2^ and GMask) - G1) div 64] + G1)
          or (RTable[((Bits2^ and RMask) - R1) div (32*64)] + R1);
        Inc(MixBits); Inc(Bits1); Inc(Bits2);
       end;
    // zum nchsten Zeilenanfang; Pointer um LineBytes erhht
    Inc(Bits1, (SDesc1.lPitch-LineBytes) div 2);
    Inc(Bits2, (SDesc2.lPitch-LineBytes) div 2);
    Inc(MixBits, (TargetDesc.lPitch-LineBytes) div 2);
  end;  // for y
end;

// Alpha-Blending mit DirectDraw fr 24 und 32 bpp
procedure AlphaBlendDDraw2432(Bits1, Bits2, MixBits: PByte;
   SDesc1, SDesc2, TargetDesc: TDDSurfaceDesc);
var x,y,z,
    BPP, LineBytes: Integer;  // 3 oder 4 = 24 vs. 32 Bit
begin
  with SDesc1 do
  begin
    BPP := ddpfPixelFormat.dwRGBBitCount div 8;
    LineBytes := dwWidth * BPP;
  end;
  // alle 3 Oberflchen haben dieselbe Gre. Wehe, wenn nicht!
  for y := 1 to SDesc1.dwHeight do
  begin
    for x := 1 to SDesc1.dwWidth do
    begin
      for z := 1 to 3 do
      begin // Direktberechnung spart gegenber Aufrufen
            // um die 20 msec - trotz Delphis Optimierungen
        MixBits^ := AlphaTable24[Bits2^-Bits1^] + Bits1^;
        Inc(MixBits); Inc(Bits1); Inc(Bits2);
      end;
      if BPP = 4 then
      begin  // 32 vs. 24 Bit
        Inc(MixBits); Inc(Bits1); Inc(Bits2);
      end;
    end;  // for x
    // zum nchsten Zeilenanfang; Pointer um LineBytes erhht
    Inc(Bits1, SDesc1.lPitch-LineBytes);
    Inc(Bits2, SDesc2.lPitch-LineBytes);
    Inc(MixBits, TargetDesc.lPitch-LineBytes);
  end;  // for y
end;

// Wenn True, dann arbeiten SafeLockDDBits und SafeUnlockDDBits
// mit kopierten Daten, die Oberflchen (und die GDI) werden nur
// kurzfristig gesperrt. Bei False wird mit den Originaldaten
// gearbeitet - Breakpoints im Debugger, MessageBoxes, Exceptions
// und so weiter bringen beim Einschirmbetrieb das System
// zum vlligen Stillstand. Vor *jedem* Experiment auf jeden Fall
// auf True setzen!
const UseSafeCopy: Boolean = False;

// Liefert einen direkten Zeiger auf die Bits der Oberflche
// oder eine Kopie der Bits (UseSafeCopy = True). Im ersten
// Fall bleibt die Oberflche gesperrt - und damit die GDI(!)
function SafeLockDDBits(Surface: IDirectDrawSurface;
   SurfDesc: TDDSurfaceDesc): Pointer;
var Res: HResult; BitmapSize: Integer;
begin
  Res := Surface.Lock(nil, SurfDesc, DDLOCK_WAIT,0);
  if SUCCEEDED(Res) then
  begin
    if UseSafeCopy then  // Kopie anlegen, danach entsperren
    with SurfDesc do
    begin
      BitmapSize := dwHeight * dwWidth *
                    ddpfPixelFormat.dwRGBBitCount div 8;
      GetMem(Result,BitmapSize);
      Move(lpSurface^,Result^,BitmapSize);
      Surface.Unlock(nil);  // bei Fehlern ist hier eh Sabbat...
    end
      else Result := SurfDesc.lpSurface;  // Originaldaten
  end
   else raise Exception.Create('GetSafeDDBits: '+ErrorString(Res));
end;

// Entsperren der Oberflche. UseSafeCopy = True: Kopie
// wieder freigeben, bei WriteBack = True vorher zurckschreiben
procedure SafeUnlockDDBits(Surface: IDirectDrawSurface;
  SurfDesc: TDDSurfaceDesc; BMBits: Pointer; WriteBack: Boolean);
var Res: HResult; BitmapSize: Integer;
begin
  if UseSafeCopy then
  begin
    if WriteBack then
    begin  // Daten zurckschreiben
      Res := Surface.Lock(nil, SurfDesc, DDLOCK_WAIT,0);
      if SUCCEEDED(Res) then
      with SurfDesc do
      begin
        BitmapSize := dwHeight * dwWidth *
                       ddpfPixelFormat.dwRGBBitCount div 8;
        Move(BMBits^,lpSurface^,BitmapSize);
        Res := Surface.Unlock(nil);
      end;
    end
     else Res := DD_OK;
    FreeMem(BMBits);  // Kopie freigeben
    if FAILED(Res) then raise
      Exception.Create('SafeUnlockDDBits: '+ErrorString(Res));
  end
    else Surface.Unlock(nil);  // Originaldaten entsperren
end;

// Die DirectDraw-Variante
procedure TForm1.GetSetDDrawBits;
var BMBits1, BMBits2, BMMixBits: Pointer;
    StartTime: Integer;
begin
  StartTime := timeGetTime;
  // Bits (oder eine Kopie davon) abholen
  BMBits1 := SafeLockDDBits(Pic1Surface, Pic1Desc);
  BMBits2 := SafeLockDDBits(Pic2Surface, Pic2Desc);
  BMMixBits := SafeLockDDBits(WorkPicSurface, WorkPicDesc);
  BuildAlphaTable24(Alpha);  // in jedem Fall
    case WorkPicDesc.ddpfPixelFormat.dwRGBBitCount of
      15,16:
       begin
         AlphaBlendDDraw1516(BMBits1, BMBits2, BMMixBits,
           Pic1Desc, Pic2Desc, WorkPicDesc);
       end;
      24, 32:
         AlphaBlendDDraw2432(BMBits1, BMBits2, BMMixBits,
           Pic1Desc, Pic2Desc, WorkPicDesc);
      else
        raise Exception.Create('GetSetDDrawBits: <> 15..32 bpp');
    end;
  // UseSafeCopy = True: Bits wieder einsetzen, Kopien freigeben
  // UseSafeCopy = False: Unlock, sonst nichts
  SafeUnlockDDBits(Pic1Surface, Pic1Desc, BMBits1, False);
  SafeUnLockDDBits(Pic2Surface, Pic2Desc, BMBits2, False);
  SafeUnlockDDBits(WorkPicSurface, WorkPicDesc, BMMixBits, True);
  FormPaint(Self);
  lFrameTime.Caption := IntToStr(timeGetTime-StartTime);
end;

end.
