#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#define MIN(x,y) ((x)<(y)?(x):(y))

static void createPeriodicFilters(int dataLength, int filtLen, float *h, float *g,float **ht, float **gt);
void __far __pascal _export dpwt(int level, float *inData, float *outData, int dataLength, float *h, float *g, int filtLen, int flag);
void __far __pascal _export idpwt(int level, float *inData, float *outData, int dataLength, float *h, float *g, int filtLen);
int  __far __pascal _export rightShift(int number, int shift);
int  __far __pascal _export leftShift(int number, int shift);

/*******************************************************************************/
/*                                                                             */
/*    function name : dpwt                                                     */
/*                                                                             */
/*    parameters    : int level - number of decomposition levels               */
/*                                0 decomposes in log2(dataLength) levels      */
/*                    float *inData  - a pointer to the float to be decomposed */
/*                    float *outData - a pointer to the decomposed data        */
/*                                    memory must already be allocated         */
/*                    int dataLength - the length of the two float arrays      */
/*                                     it must be a power of 2 - not checked   */
/*                    float *h     - a pointer to the low pass filter          */
/*                    float *g     - a pointer to the high pass filter         */
/*                    int filtLen  - the length of the filters h and g         */
/*                    int flag     - if set inData will be changed to carry    */
/*                                   the approximation informations            */
/*                                                                             */
/*    return value  : none                                                     */
/*                                                                             */
/*    description   : calculates the DPWT of inData                            */
/*                    The DPWT decomposes a signal in approximations of the    */
/*                    signal in different orthogonal L_2 spaces and some       */
/*                    additional detail signals, which carry the information   */
/*                    lost by the approximations.                              */
/*                    the inData is identyfied with the approximation at       */
/*                    level 0. The approximations are calculated by convolving */
/*                    the last approximation with the filter h, the corres-    */
/*                    ponding details are calculated by convolving the last    */
/*                    approximation with the filter g.                         */
/*                    During the convolution the data is also downsampled by 2.*/
/*                    The data is assumed to be periodic with period dataLength*/
/*                    The complete decompostion will be in approx and detail.  */
/*                    For the reconstructionast                                */
/*                    approximation is needed. This data is being copied to    */
/*                    outData.                                                 */
/*                                                                             */
/*    side effects  : changes the contents of outData                          */
/*                    if flag is set inData will also be changed               */
/*                                                                             */
/*    comments      : when the signal length of the approximation gets small   */
/*                    (i.e. smaller than the filter length) the convolution    */
/*                    needs some modification accomplished by the "periodic"   */
/*                    filters, such that the reconstuction is correct.         */
/*                    Thats the special thing about the DPWT to take care      */
/*                    of this effect. See the paper bey Neil Getz for more     */
/*                    information.                                             */
/*                                                                             */
/*******************************************************************************/

void __far __pascal _export dpwt(int level, float *inData, float *outData, int dataLength, float *h, float *g, int filtLen, int flag)
{
    int i, j, n, k, currentSignalLength, dataIndex, cnt=0;
    float **approx, **detail;
    float *ht, *gt;

    if (level == 0)
        while(dataLength >> (++level+1))   /* gives log2(dataLength) in level */
            ;

    /**** get the 'necessary' memory pointers ****/
    
    approx    = (float**) calloc((level+1),sizeof(float*));
    detail    = (float**) calloc((level+1),sizeof(float*));
    approx[0] = inData;

    /**** do the periodic wavelet transform ****/
    
    for (j=0; j < level; j++)
    {
        currentSignalLength = (dataLength >> j);
        detail[j+1] = (float*) calloc(currentSignalLength/2, sizeof(float));
        approx[j+1] = (float*) calloc(currentSignalLength/2, sizeof(float));
        createPeriodicFilters(currentSignalLength, filtLen, h, g, &ht, &gt);

        for (i=0; i < currentSignalLength/2; i++)
        {
            for (n = 2*i, k=0;
                 n < 2*i + MIN(filtLen, currentSignalLength);
                 n++, k++)
            {
                dataIndex = n % currentSignalLength;
                approx[j+1][i] += ht[k] * approx[j][dataIndex];
                detail[j+1][i] += gt[k] * approx[j][dataIndex];
            }
        }
    }

    /**** now we put the result in outData ... ****/

    for (j=1; j <= level; j++)
        for (i=0; i < (dataLength >> j); i++)
            outData[cnt++] = detail[j][i];

    for (i=0; i < (dataLength >> level); i++)
        outData[cnt++] = approx[level][i];

    if (flag)
    { 
        cnt = 0;
        for (j=1; j <= level; j++)
            for (i=0; i < (dataLength >> j); i++)
                inData[cnt++] = approx[j][i];
    }
    /**** free memory ****/
    
    for (j=0; j < level; j++)
        free(approx[j+1]), free(detail[j+1]);
    free(approx), free(detail);
    
    return;
}

/*******************************************************************************/
/*                                                                             */
/*    function name : idpwt                                                    */
/*                                                                             */
/*    parameters    : int level - number of decomposition levels               */
/*                                0 decomposes in log2(dataLength) levels      */
/*                    float *inData  - a pointer to the decomposed float to    */
/*                    float *outData - a pointer to the reconstructed float    */
/*                                    memory must already be allocated         */
/*                    int dataLength - the length of the two float arrays      */
/*                                     it must be a power of 2 - not checked   */
/*                    float *h     - a pointer to the low pass filter          */
/*                    float *g     - a pointer to the high pass filter         */
/*                    int filtLen - the length of the filters h and g          */
/*                                                                             */
/*    return value  : none                                                     */
/*                                                                             */
/*    description   : calculates the IDPWT of inData in outData.               */
/*                    The IDPWT reconstructs a signal from its DPWT, which     */
/*                    carries the detail informations and the last approx-     */
/*                    imation. Starting with the approximation at the last     */
/*                    level and the last detail coefficients, the previous     */
/*                    approximations at level j-1 are calculated by convolving */
/*                    the approximation at level j with h, convolving the      */
/*                    details at level j with g and adding them.               */
/*                                                                             */
/*    side effects  : changes the contents o                                   */
/*                                                                             */
/*    comments      : in the beginning of this functions the decomposed float  */
/*                    first must be put to the right places.                   */
/*                                                                             */
/*******************************************************************************/

void __far __pascal _export idpwt(int level, float *inData, float *outData, int dataLength,float *h, float *g, int filtLen)
{
    int i, j, n, filtIndex, dataIndex, currentSignalLength, cnt=0;
    float **approx, **detail;
    float *ht=NULL, *gt=NULL;
             
    if (level == 0)
        while(dataLength >> (++level+1))   /* gives log2(dataLength) in level */
            ;   
    /**** get the 'necessary' memory pointers ****/

    approx = (float**) calloc((level+1),sizeof(float*));
    detail = (float**) calloc((level+1),sizeof(float*));

    /**** put the decomposition to the right places and get the memory for it **/
    
    for (j=1; j <= level; j++)
    {
        approx[j] = (float*) calloc((dataLength >> j), sizeof(float));      
        detail[j] = (float*) calloc((dataLength >> j), sizeof(float));
        for (i=0; i<(dataLength >> j); i++)
            detail[j][i] = inData[cnt++];
    }
    for (i=0; i<(dataLength >> level); i++)
        approx[level][i] = inData[cnt++];
    approx[0] = outData; /* in approx[0] will be the reconstructed float */
    
    /**** do the inverse periodic wavelet transform ****/

    for (j=level; j > 0; j--)
    {       
        currentSignalLength = (dataLength >> j);
        createPeriodicFilters(currentSignalLength*2, filtLen, h, g, &ht, &gt);

        for (i=0; i < currentSignalLength*2; i++)
        {
            for (n = i/2;
                 n > i/2 - MIN(filtLen/2, currentSignalLength);
                 n--)
            {
                dataIndex = (n + 4*currentSignalLength) % currentSignalLength;
                filtIndex = (i-2*n) % (currentSignalLength*2);
                approx[j-1][i] += ht[filtIndex] * approx[j][dataIndex] 
                                + gt[filtIndex] * detail[j][dataIndex];
            }
            
        }
    }

    /**** free memory ****/
    
    for (j=0; j < level; j++)
        free(approx[j+1]), free(detail[j+1]);
    free(approx), free(detail);

    return;
}

/*******************************************************************************/
/*                                                                             */
/*    function name : createPeriodicFilters   (locally declared)               */
/*                                                                             */
/*    parameters    : int dataLength - the length of the float being convolved  */
/*                    int filtLen - the length of the filters h and g          */
/*                    float *h     - a pointer to the low pass filter           */
/*                    float *g     - a pointer to the high pass filter          */
/*                    float **ht   - a pointer to the address of the periodic   */
/*                                  low pass filter to be calculated           */
/*                    float **gt   - a pointer to the address of the periodic   */
/*                                  high pass filter to be calculated          */
/*                                                                             */
/*    return value  : none                                                     */
/*                                                                             */
/*    description   : calculates the necessary periodic filters for dpwt()     */
/*                    and idpwt().                                             */
/*                    If the dataLength is >= the periodic filters are just    */
/*                    the original filters. ing   */
/*                    allocated for the periodic filters and they are          */
/*                    calculated. For more information on the calculations     */
/*                    see the memorandum by Neil Getz.                         */
/*                                                                             */
/*    side effects  : changes the contents of *ht and *gt                      */
/*                    finally leaves the last *ht and *gt allocated - they     */
/*                    don't get freed                                          */
/*                                                                             */
/*    comments      : this calculation may be done once for all lengths        */
/*                    and put globally for some speed up.                      */
/*                                                                             */
/*******************************************************************************/

static void createPeriodicFilters(int dataLength, int filtLen, 
                                  float *h, float *g, float **ht, float **gt)
{
    int i, k;
    
    if ( filtLen <= dataLength )
    {
        *ht = h;
        *gt = g;
    }
    else
    {
        if ((*ht != h) && (*ht != NULL))
        {
            free(*ht);
            free(*gt);
        }
        *ht = (float*) calloc(dataLength, sizeof(float));
        *gt = (float*) calloc(dataLength, sizeof(float));
        for (i=0; i < dataLength; i++)
        {
            for (k=0; k <= (filtLen - 1 - i)/dataLength; k++)
            {
                *(*ht+i) += h[i+k*dataLength];
                *(*gt+i) += g[i+k*dataLength];
            }
        }
    }
    return;
}


/*******************************************************************************/
/*                                                                             */
/*  convenience functions for Visual Basic which doesn't support shifts        */
/*  (as far as I know)                                                         */
/*                                                                             */
/*******************************************************************************/

int  __far __pascal _export rightShift(int number, int shift)
{
    return number >> shift;
} 

int  __far __pascal _export leftShift(int number, int shift)
{
    return number << shift;
}

