#include <stdlib.h>
#include <stdio.h>
#include <conio.h>
#include <dos.h>
#include "tcpip.h"

// define:
// mis = 1/2^10 = 1/1024 second    = approx 1 ms
// uis = 1/2^20 = 1/1048576 second = approx 1 us
//  Ts = 1/2^14 = 1/16384 second   = 64 uis = approx 61 us

#define BUFFER_LEN 1024
#define PORT 8765

// high bit of buffers is a high/low flag; low 15 bits are in Ts
volatile unsigned int buffer[2][BUFFER_LEN];
volatile unsigned int *current_buffer = buffer[0];
volatile unsigned int *other_buffer = buffer[1];
volatile unsigned int current_buffer_len=0;

volatile unsigned int overflow_count=0;	// in mis

// Timer overflows at this value
unsigned int timer_max;

void ipc_sleep(int ms) {
	union  REGS  inregs, outregs;
	struct SREGS segregs;
	inregs.x.ax = 0x0900;
  	inregs.x.bx = ms;
  	int86x(0xAC, &inregs, &outregs, &segregs);
}

void push(int high, unsigned int over, unsigned int count)
{
	unsigned int *t;

	// If buffer is full, ignore it
   if(current_buffer_len >= BUFFER_LEN)
   	return;

	// okay to drop volatile because interrupts are disabled
   t = (unsigned int *)&(current_buffer[current_buffer_len]);
   *t = ((unsigned int)high) << 15;
	*t += (over << 4) & 0x7FF0U;
	*t += (unsigned int)((((unsigned long)count)<<4) / timer_max) & 0x0F;
	current_buffer_len++;
}

void overflow_handler(int high) {
	overflow_count++;	// in mis

	// Longest we can represent in our buffer is 2s.
   // If it's more than that, send it off.
	if(overflow_count & 0xF800) { // >= 2048 mis
		push(high,0x7FFU,timer_max-1);
		overflow_count=0;
	}
}

void edge_handler(int timer)
{
	union REGS inregs, outregs;
	struct SREGS segregs;
	unsigned int count;

	// Read timer count
	inregs.h.ah = 0x88;
	inregs.h.al = (unsigned char)timer;
	int86x(0xA1, &inregs, &outregs, &segregs);
	count = outregs.x.ax;

	// Reset timer count
	inregs.h.ah = 0x89;
	inregs.h.al = (unsigned char)timer;
	inregs.x.dx = 0;			// reset to 0
	int86x(0xA1, &inregs, &outregs, &segregs);

   // timer 0 counts when signal is high
   // count is in units of 1/timer_max mis.
	// overflow_count is in units of mis.
   push(!timer,overflow_count,count);
  	overflow_count=0;
}

void interrupt int2_handler(void)
{
	outportb(0x600,1);
	edge_handler(1);			// rising edge -- was using timer1
}

void interrupt int4_handler(void)
{
	outportb(0x600,2);
	edge_handler(0);			// falling edge -- was using timer0
}

void interrupt tmr0_handler(void)
{
	outportb(0x600,4);
	overflow_handler(1);		// timer0 counts when signal is high
}

void interrupt tmr1_handler(void)
{
	outportb(0x600,8);
	overflow_handler(0);		// timer1 counts when signal is low
}

void main(int argc, char *argv[])
{
	union  REGS  inregs, outregs;
	struct SREGS segregs;

	int int_num[] =
   	{ 2, 4, 8, 9 };
	void interrupt (*new_func[sizeof(int_num)/sizeof(*int_num)])() =
		{ int2_handler, int4_handler, tmr0_handler, tmr1_handler };
	void interrupt (*old_func[sizeof(int_num)/sizeof(*int_num)])();

	int i;
   int socket, err;
   unsigned int *tmp;

   struct sockaddr_in dest;

   // Set STDIO focus to application
   inregs.h.ah = 0x11;
   inregs.h.al = 0x02;
   int86(0xA0, &inregs, &outregs);

   // Check command line arguments
	if(argc!=2) {
   	printf("usage: %s [dotted-ip-address]\n",*argv);
      printf("IR data will be sent to the given host on UDP port %d\n",PORT);
		goto exit_restorefocus;
   }

	printf("LIRCBox v1.0 initializing...\r\n");

   // Create socket and related data structures
	if((socket=opensocket(AF_INET,&err))==API_ERROR) {
    	printf("error: can't create socket: error %d\n",err);
		goto exit_restorefocus;
   }

   if(Set_Blocking_Mode(socket,0,&err)==API_ERROR) {
   	printf("error: can't set blocking mode: error %d\n",err);
		goto exit_closesocket;
   }

	dest.sin_family = AF_INET;
	if(inet_addr(argv[1],&dest.sin_addr.s_addr)==API_ERROR) {
   	printf("error: can't parse IP address: %s\n",argv[1]);
		goto exit_closesocket;
   }

  	dest.sin_port = htons(PORT);

   printf("Sending IR data to %s port %d\n",argv[1],PORT);

  	// Disable ALE so we can use the output LEDs for debugging
	inregs.h.ah = 0x83;
	inregs.x.dx = 0x40;
	int86(0xA2, &inregs, &outregs);
   outportb(0x600,0xff);

   // Set the max timer such that it overflows once per mis
   inregs.h.ah = 0x8A;
   inregs.h.al = 0x03;
   int86(0xA1, &inregs, &outregs);
   // dx:ax holds the ticks per second
	timer_max = (outregs.x.dx << 6) | (outregs.x.ax >> 10);

   // Set interrupt handlers
   for(i=0;i<(sizeof(int_num)/sizeof(*int_num));i++) {
   	inregs.h.ah = 0x84;
      inregs.x.dx = int_num[i];
      inregs.x.cx = 1;
      segregs.es  = FP_SEG(new_func[i]);
      inregs.x.bx = FP_OFF(new_func[i]);
      int86x(0xA1, &inregs, &outregs, &segregs);
      old_func[i] = MK_FP(segregs.es,outregs.x.bx);
   }

	// Setup timers
	for(i=0;i<2;i++) {
		// Initialize
		inregs.h.ah = 0x85;
		inregs.h.al = (unsigned char)i;
		inregs.x.dx = 0x03;
		inregs.x.cx = timer_max;
		int86x(0xA1, &inregs, &outregs, &segregs);

		// Start
      inregs.h.ah = 0x86;
      inregs.h.al = (unsigned char)i;
      int86x(0xA1, &inregs, &outregs, &segregs);
	}

	// Enable PWD
	inregs.h.ah = 0x87;
	int86x(0xA2, &inregs, &outregs, &segregs);

	// Receive data until we get an ESC
   printf("Running.  Press ESC to quit.\r\n");

	for(;!(kbhit() && getch()==27);) {
		i=0;
		disable();
      if(current_buffer_len>0) {
			// ok to drop volatile because interrupts are disabled
   		tmp=(unsigned int *)current_buffer;
         current_buffer=other_buffer;
         other_buffer=tmp;
         i=current_buffer_len;
         current_buffer_len=0;
      }
      enable();
		if(i>0) { // there's a packet to send
      	outportb(0x600,16);
			if(sendto(socket,(char *)other_buffer,i*2,
         	0,(struct sockaddr *)&dest,&err)==API_ERROR) {
            printf("send error: %d\n",err);
         }
      }

  		// the buffer holds elements with size at least one Ts.
      // since they come 16384 per second, we expect to fill
      // a 1024 element buffer in 1/16 = 62.5ms at absolute fastest.
      // We'll sleep 100ms between checks; there's no way we'll actually
      // get data that fast (if we do, it will get silently dropped)
		ipc_sleep(100);
   }

   printf("Restoring interrupt handlers...\r\n");

	// Stop timers
   for(i=0;i<2;i++) {
   	inregs.h.ah = 0x87;
      inregs.h.al = (unsigned char)i;
		int86x(0xA1, &inregs, &outregs, &segregs);
	}

   // Restore interrupt handlers
   for(i=0;i<(sizeof(int_num)/sizeof(*int_num));i++) {
   	inregs.h.ah = 0x84;
      inregs.x.dx = int_num[i];
      inregs.x.cx = 1;
      segregs.es  = FP_SEG(old_func[i]);
      inregs.x.bx = FP_OFF(old_func[i]);
      int86x(0xA1, &inregs, &outregs, &segregs);
   }

	// deinit PWD (set to PIO)
	inregs.h.ah = 0x82;
	inregs.h.al = 0x01;
	inregs.x.dx = 0x01<<6;
	int86x(0xA2, &inregs, &outregs, &segregs);

   printf("Done.\r\n");

exit_closesocket:
   // Shut down UDP/IP
   closesocket(socket,&err);

exit_restorefocus:
   // Set STDIO focus back to console
   inregs.h.ah = 0x11;
   inregs.h.al = 0x01;
   int86(0xA0, &inregs, &outregs);
}

