/* LowLevel EEPROM initialization routines version 1.0

	Copyright (C) 1999 Stefan Verse, Andreas Kemper

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2, or (at your option)
	any later version.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include "parms.h"
#include "masisr.h"
#include "atmelini.h"
#include "dos.h"

extern MASLPTPARAMS lptP;

/* Zeichen-Definitionen (acht Definitions-Bytes pro Zeile, erste Zeile = */
/* erstes Zeichen, das im Zeichensatz den Wert Null hat.) */
const byte ZeichenTab[8][8] =
{{0x1f, 0x1f, 0x1f, 0x0f, 0x07, 0x07, 0x03, 0x01},
 {0x00, 0x00, 0x00, 0x10, 0x18, 0x1c, 0x1c, 0x1e},
 {0x00, 0x00, 0x00, 0x01, 0x03, 0x07, 0x07, 0x0f},
 {0x1f, 0x1f, 0x1f, 0x1e, 0x1c, 0x1c, 0x18, 0x10},
 {0x1f, 0x1f, 0x1f, 0x1f, 0x1f, 0x00, 0x00, 0x00},
 {0x18, 0x1c, 0x1e, 0x1e, 0x1f, 0x1f, 0x1f, 0x1f},
 {0x1f, 0x1f, 0x1f, 0x1f, 0x1e, 0x1e, 0x1c, 0x18},
 {0x00, 0x00, 0x00, 0x1f, 0x1f, 0x1f, 0x1f, 0x1f}};

/* Hier die vier Zeilen, aus denen dann das Logo zusammengebaut wird. Dabei */
/* werden neben den Zeichen, die zuvor definiert wurden auch das */
/* Leerzeichen (hex 14) und das ganz ausgefuellte Zeichen (hex ff) aus dem */
/* normalen Zeichensatz verwendet. */
const byte LogoTab[4][16]=
{{0xff, 0x00, 0x01, 0x02, 0x03, 0xff, 0x14, 0xff,
	0x04, 0x04, 0x05, 0x14, 0x04, 0x04, 0x04, 0x05},
 {0xff, 0x14, 0x00, 0x03, 0x14, 0xff, 0x14, 0xff,
	0x07, 0x07, 0x06, 0x14, 0x14, 0x07, 0x07, 0xff},
 {0xff, 0x14, 0x14, 0x14, 0x14, 0xff, 0x14, 0xff,
	0x14, 0x14, 0x1c, 0x14, 0x14, 0x14, 0x14, 0xff},
 {0xff, 0x14, 0x14, 0x14, 0x14, 0xff, 0x14, 0xff,
	0x14, 0x14, 0x14, 0x14, 0x07, 0x07, 0x07, 0x06}};

/* Schreibsequenz fuer ein Byte auf den Controller */
void AtmelWrite(byte reg, byte dat)
{
	int tot;
	/* Spezielle Registeradressen, z.B. fuer EEPROM Programmierung */
	const byte strg[]={0x0a, 0x02, 0x08, 0x00, 0x0c, 0x04};

	/* Steuersignale zur Registerauswahl anlegen */
	outp(lptP.lptCtrl,strg[reg]);
	/* Datenbyte anlegen */
	outp(lptP.lptAdr,dat);
	/* Strobeflag fuer gueltige Daten setzen */
	outp(lptP.lptCtrl,strg[reg]+STROBE);

	/* Setzen des BUSY-Flags abwarten */
	tot=0;
	delay(1);
	while(!(inp(lptP.lptStat)&BUSY)&&(tot++<1000));

	/* Strobeflag zuruecknehmen */
	outp(lptP.lptCtrl,strg[reg]);

	/* Loeschen des BUSY-Flags abwarten */
	tot=0;
	delay(1);
	while((inp(lptP.lptStat)&BUSY)&&(tot++<1000));
}

/* Lesesequenz fuer ein Byte aus dem Controller */
byte AtmelRead(byte reg)
{
	int tot;
	byte i,x,p;
	/* Spezielle Registeradressen, z.B. fuer EEPROM Programmierung */
	const byte strg[]={0x2e, 0x26, 0x2c, 0x24};

	/* Strobeflag togglen */
	outp(lptP.lptCtrl,strg[reg]);
	outp(lptP.lptCtrl,strg[reg]+STROBE);

	/* Warten auf Busyflag=high und Auslesen des ersten Bits ber Error-Ltg */
	tot=0;
	delay(1);
	while(!((p=inp(lptP.lptStat))&BUSY)&&(tot++<100));

	/* Gelesenes Bit auswerten */
	if(p&0x08)
		x=1;
	else
		x=0;

	/* Auslesen der restlichen sieben Bits */
	for(i=1;i<8;i++)
	{
		/* Strobeflag setzen, Initflag zuruecknehmen */
		outp(lptP.lptCtrl,strg[reg]+STROBE-INIT);

		/* Warten auf Busyflag=low innerhalb des Timeout */
		tot=0;
		delay(1);
		while((inp(lptP.lptStat)&BUSY)&&(tot++<1000));

		/* Initflag erneut setzen */
		outp(lptP.lptCtrl,strg[reg]+STROBE);

		/* Warten auf Busyflag=high innerhalb des Timeout */
		tot=0;
		delay(1);
		while(!((p=inp(lptP.lptStat))&BUSY)&&(tot++<1000));

		/* Gelesenes Bit auswerten */
		if (p&0x08)
			x = x +(1<<i);
	}

	/* Steuerregister auf urspruenglichen Wert zuruecksetzen */
	outp(lptP.lptCtrl,strg[reg]);

	/* Vor Beendigung der Routine noch auf Ruecknahme des Busyflags warten */
	tot=0;
	delay(1);
	while(((p=inp(lptP.lptStat))&BUSY)&&(tot++<1000));

	return x;
}

/* Ein Byte in EEPROM an vorgegebene Adresse schreiben */
void Seteeprom(int Adr, byte Data)
{
	if (Adr<512)
	{
		AtmelWrite(4, (Adr&0xff));
		AtmelWrite(1, 0x23);
		AtmelWrite(4, (Adr>>8));
		AtmelWrite(1, 0x24);
		AtmelWrite(4, Data);
		AtmelWrite(1, 0x25);
	}
}

/* Ein Byte aus vorgegebener EEPROM Adresse lesen */
byte Geteeprom(int Adr)
{
	AtmelWrite(1, 0x15);
	if (Adr<512)
	{
		AtmelWrite(4, (Adr&0xff));
		AtmelWrite(1, 0x23);
		AtmelWrite(4, (Adr>>8));
		AtmelWrite(1, 0x24);
		return AtmelRead(1);
  }
  return 0;
}

/* Initialisierungssequenz zum Beschreiben des EEPROM */
void AtmelInit(byte LCD_Kontrast)
{
	int i;

	for(i=0; i<64; i++)
		Seteeprom(i, ZeichenTab[i/8][i%8]);	/* Sonderzeichen in EEPROM schreiben */
	for(i=0; i<64; i++)
		Seteeprom(i+64,LogoTab[i/16][i%16]);	/* Logo in EERPOM schreiben */

	Seteeprom(509, LCD_Kontrast);	 	/* Kontrastwert schreiben */
	Seteeprom(510,255);   				/* Beleuchtung einschalten */
	Seteeprom(511,1);						/* EEPROM Schreibvorgang abschliessen */
	AtmelWrite(1,0x04);					/* Atmel Reset durchfuehren */
}

