require 'serialport'

class Brainlink
  def initialize(port, options = {})
    max_retries = options[:max_retries] || 3
    retry_interval = options[:retry_interval] || 0.2
    begin
      @brainlink = SerialPort.new( 
        port,            # port name,
        115200,          # baud rate
        8,               # data bits
        1,               # stop bits
        SerialPort::NONE # parity
      )
    rescue => ex
      if max_retries > 0
        max_retries -= 1
        sleep retry_interval
        retry
      else
        raise "Could not connect to Brainlink: #{ex.message}"
      end
    end
    @brainlink.read_timeout = 300
    @brainlink.read
  end

  def connect
    send_command '*'
  end

  def discovery_mode
    send_command 'Q'
  end

  def light_sensor_value
    (send_command 'L')[0]
  end

  def thermistor_sensor_value
    (send_command 'T')[0]
  end

  def start_tone(frequency)
    cca = 62500 / frequency - 1
    send_command ['B'.ord, high_byte(cca), low_byte(cca)].pack('C3')
  end

  def stop_tone
    send_command 'b'
  end

  def voltage
    (send_command 'V')[0] * 2650 / 128
  end

  def set_led(r, g, b)
    send_command ['O'.ord, r, g, b].pack('C4')
  end

  def get_raw_acceleration_values
    send_command 'A'
  end

  def get_x_acceleration
    get_raw_acceleration_values[0]
  end

  def get_y_acceleration
    get_raw_acceleration_values[1]
  end

  def get_z_acceleration
    get_raw_acceleration_values[2]
  end

  def has_been_shaken?
    (get_raw_acceleration_values[3] & 0x80) == 0x80
  end

  def has_been_tapped?
    (get_raw_acceleration_values[3] & 0x20) == 0x20
  end

  def get_analog_input(port)
    (send_command 'X')[port] || 0
  end

  def get_digital_input(port)
    result = send_command ['<'.ord, port + '0'.ord].pack('C2')
    return result[0] == '1'.ord
  end

  def set_digital_output(port, value)
    send_command ['>'.ord, port + '0'.ord, (value ? '1' : '0').ord].pack('C3')
  end

  def turn_off_ir
    send_command '!'
  end

  def set_aux_serial_config(baud, scale)
    send_command ['C'.ord, high_byte(baud), low_byte(baud), scale].pack('C3c')
  end

  def aux_serial_receive
    send_command 'r'
  end

  def aux_serial_transmit(data)
    command = data.clone
    command.unshift data.length
    command.unshift 't'.ord
    send_command command.pack('C*')
  end

  def set_dac(port, value)
    send_command ['d'.ord, port + '0'.ord, value * 255 / 3300].pack('C3')
  end

  def send_stored_ir_signal(signal, repeat_time)
    send_command ['G'.ord, signal + '0'.ord, high_byte(repeat_time), low_byte(repeat_time)].pack('C4')
  end

  def send_raw_ir_signal(data, repeat_time)
    command = data.clone
    command.unshift 's'.ord
    command.push high_byte(repeat_time)
    command.push low_byte(repeat_time)
    send_command command.pack('C*')
  end

  def send_ir_command(data, repeat_time)
    command = data.clone
    command.unshift 'i'.ord
    command.push high_byte(repeat_time)
    command.push low_byte(repeat_time)
    send_command command.pack('C*')
  end

  def capture_ir_signal
    send_command 'R', 5000    
  end

  def store_last_ir_signal(location)
    send_command ['S'.ord, location + '0'.ord].pack('C2')
  end

  def print_stored_ir_signal(location)
    send_command ['g'.ord, location + '0'.ord].pack('C2')
  end

  def init_ir_system(frequency, pulses, encoding, bit_length, one_length, zero_length, updown_length)
    data = ['I'.ord]
    f = 32000000 / frequency
    data.push high_byte(f); data.push low_byte(f)
    data.push pulses.length
    pulses.each do |b|
      data.push high_byte(b); data.push low_byte(b)
    end
    data.push encoding
    data.push bit_length
    data.push high_byte(one_length); data.push low_byte(one_length)
    data.push high_byte(zero_length); data.push low_byte(zero_length)
    data.push high_byte(updown_length); data.push low_byte(updown_length)
    send_command data.pack('C*')    
  end

  def dump
    result = {
      'light-sensor' => light_sensor_value,
      'thermistor' => thermistor_sensor_value,
      'voltage' => voltage,
      'stored-ir-signals' => [],
      'aux-serial' => aux_serial_receive,
      'acceleration' => get_raw_acceleration_values,
      'analog-input' => [],
      'digital-input' => []
    }
    0.upto(4) do |i|
      result['stored-ir-signals'].push print_stored_ir_signal i
    end
    0.upto(5) do |i|
      result['analog-input'].push get_analog_input i
    end
    0.upto(9) do |i|
      result['digital-input'].push get_digital_input i
    end
    result
  end

  def Brainlink.to_g(value)
    g = to_6bit_value(value.nil? ? 0 : value) * 1.5 / 32.0
    if g < -1.5
      return -1.5
    elsif g > 1.453125
      return 1.453125
    else
      return g
    end
  end

  private

  def send_command(command, timeout = 300)
    @brainlink.read_timeout = timeout
    @brainlink.write command
    sleep 0.001
    response = @brainlink.read
    echo = response[0, command.length]
    echo.force_encoding('ASCII-8BIT')
    command.force_encoding('ASCII-8BIT')
    result = response[command.length .. -1]
    if command != '*' and echo.bytes.to_a != command.bytes.to_a
      $stderr.puts "Oops! #{echo} != #{command}"
    end
    !result.nil? ? result.bytes.to_a : []
  end

  def high_byte(value)
    (value & 0xff00) >> 8
  end

  def low_byte(value)
    value & 0x00ff
  end

  def Brainlink.to_6bit_value(value)
    clean_val = [[value, 0x3f].min, 0x00].max
    clean_val <= 0x1f ? clean_val : clean_val - 64
  end
end

