//-----------------------------------------------------------------------------
//  ComM.c
/// @file ComM.c
///
/// @brief Implementation of all functions for the bluetooth communication
///
/// All functions for the communication via bluetooth are implemented in this module.
///
/// @copyright 2012-2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-22 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------
#include "Basetyp.h"
#include "null.h"

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "ComM.h"
#include "OS.h"
#include "MessageDefs.h"
#include "Tools.h"
#include "DControlAPI.h"

#include "PE_LDD.h"
#include "PE_Error.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------
/// Maximum number of data bytes in the receive buffer
#define MAX_NUM_OF_REC_DATA_BYTES 128
/// Maximum number of data bytes in the transmit buffer
#define MAX_NUM_OF_TRM_DATA_BYTES 256
/// Maximum number of parameters in a command
#define MAX_NUM_OF_PARAMS (MAX_NUM_OF_TRM_DATA_BYTES - SIZE_OF_DC_API_HEADER - 1)

/// Defintion of the data type for the received data
typedef enum tagDATA_TYPE
{
   DATA_TYPE_UNKNOWN = 0,     ///< Unknown data type
   DATA_TYPE_MSG_HEADER,      ///< Data type message header
   DATA_TYPE_MSG_DATA         ///< Data type message data
} DATA_TYPE;

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------
/// Flag: Data is transmitted via UART interface
extern bool m_UART_DataTransmitted;

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------
/// Handle to the UART
LDD_TDeviceData * m_UART_Handle;
/// Receive data buffer
UBYTE m_ReceiveDataBuffer[MAX_NUM_OF_REC_DATA_BYTES];
/// Transmit data buffer
UBYTE m_TransmitDataBuffer[MAX_NUM_OF_TRM_DATA_BYTES];
/// Actual data type (header/data)
DATA_TYPE m_DataType;

/// Commandheader
CMD_HEADER m_CmdHeader;
/// Flag file tarnsfer active
UBYTE m_FileTransferActive;
/// Total number of data block (commands) in the actual file transfer
UWORD m_TotalNumOfBlocks;
/// Number of data in file in bytes
UDWORD m_NumOfDataInFile;
/// Index actual processed data block
UWORD m_iCurFileBlock;

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------
// Handler for the message with the id MSG_INIT_SYSTEM
void  ComM_MsgInitSystem();
// Handler for the message with the id MSG_UART_DB_RECEIVED
void  ComM_MsgUartDBReceived();
// Handler for the message with the id MSG_DEVICE_INFO
void  ComM_MsgDeviceInfo (MSG_DATA_DEVICE_INFO * pDeviceInfo);
// Handler for the message with the id MSG_DEVICE_CONFIG
void  ComM_MsgDeviceConfig (MSG_DATA_DEVICE_CONFIG * pDeviceConfig);
// Handler for the message with the id MSG_ACTUAL_MVALUES
void  ComM_MsgActualMValues (MSG_DATA_ACTUAL_MVALUES * pActualMValues);
// Handler for the message with the id MSG_MEASURE_MODE
void  ComM_MsgMeasureMode (MSG_DATA_MEASURE_MODE * pMeasureMode);
// Handler for the message with the id MSG_START_FFS_FILE_TRANSFER
void  ComM_MsgStartFFSFileTransfer (MSG_DATA_START_MEASURE_FILE_TRANSFER * pStartMeasureFileTransfer);
// Handler for the message with the id MSG_START_FFS_FILE_TRANSFER
void  ComM_MsgFFSFileBlock (MSG_DATA_MEASURE_FILE_BLOCK * pMeasureFileBlock);
// Handler for the message with the id MSG_FFS_FILE_EOF
void  ComM_MsgFFSFileEof();
// Handler for the message with the id MSG_ENABLE_BLUETOOTH
void  ComM_MsgEnableBluetooth (MSG_DATA_ENABLE_BLUETOOTH *  pEnableBluetooth );

// Request the next data block from the uart
void  ComM_InitReceiveNextDataBlock( UBYTE SizeOfNextDataBlock );
// Send the command CMD_ID_PING_ACK via bluetooth
void  ComM_SendPingAck();
// Send the command CMD_ID_ACK via bluetooth
void  ComM_SendAck();
// Send the command CMD_ID_NAK via bluetooth
void  ComM_SendNak();
// Send the command CMD_ID_CONNECT via bluetooth
void  ComM_SendConnect();

// Process the commands received over bluetooth
UBYTE ComM_ProzessCmd(CMD_HEADER * pCmdHeader);
// Handler for the bluetooth command CMD_ID_PING
void  ComM_ProzessCmd_Ping ();
// Handler for the bluetooth command CMD_ID_REQ_CONNECT
void  ComM_ProcessCmd_ReqConnect();
// Handler for the bluetooth command CMD_ID_REQ_DEVICE_CONFIG
void  ComM_ProcessCmd_ReqDeviceConfig();
// Handler for the bluetooth command CMD_ID_REQ_DEVICE_INFO
void  ComM_ProcessCmd_ReqDeviceInfo();
// Handler for the bluetooth command CMD_ID_REQ_ACTUAL_MVALUES
void  ComM_ProcessCmd_ReqActualMValues();
// Handler for the bluetooth command CMD_ID_RESET_MVALUES
void  ComM_ProcessCmd_ResetMValues();
// Handler for the bluetooth command CMD_ID_REQ_MEASURE_MODE
void  ComM_ProcessCmd_ReqMeasureMode();
// Handler for the bluetooth command CMD_ID_CLEAR_ALL_DATA_FILES
void  ComM_ProcessCmd_ClearAllDataFiles();
// Handler for the bluetooth command CMD_ID_ACK
void  ComM_ProcessCmd_Ack();

// Process the commands with data received over bluetooth
void  ComM_ProcessCmdData( PUBYTE pMsgRawData, CMD_HEADER * pCmdHeader );
//  Handler for the bluetooth command CMD_ID_DEVICE_CONFIG
void  ComM_ProcessCmdData_DeviceConfig(CMD_HEADER * pCmdHeader, PUBYTE pMsgRawData);
// Handler for the bluetooth command CMD_ID_MEASURE_MODE
void  ComM_ProcessCmdData_MeasureMode (CMD_HEADER * pCmdHeader, PUBYTE pMsgRawData);
// Handler for the bluetooth command CMD_ID_REQ_FFS_FILE
void  ComM_ProcessCmdData_ReqFFSFile (CMD_HEADER * pCmdHeader, PUBYTE pMsgRawData);

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function ComM_TaskFunction
///
/// @brief Taskfunction of the ComM-Task
///
/// @param pvParameters (@c [in] void *) - Task parameter (unused)
///
/// This is the taskfunction of the communication task. All message from other
/// task in the application will be received in this function.
/// Attention: Please do not process the message directly in this function!
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_TaskFunction (void * pvParameters)
{
   MESSAGE Message;
   
   while(1)
   {
      if (OS_ReceiveMessage (TASK_ID_COM_M, &Message, WAIT_MAX) == ERR_OS_OK)
      {
         switch (Message.MessageID)
         {
            case MSG_INIT_SYSTEM:                  ComM_MsgInitSystem();                                                                              break;
            case MSG_UART_DB_RECEIVED:             ComM_MsgUartDBReceived();                                                                          break;
            case MSG_DEVICE_INFO:                  ComM_MsgDeviceInfo((MSG_DATA_DEVICE_INFO *) Message.pMessageData);                                 break;
            case MSG_DEVICE_CONFIG:                ComM_MsgDeviceConfig((MSG_DATA_DEVICE_CONFIG *) Message.pMessageData);                             break;
            case MSG_ACTUAL_MVALUES:               ComM_MsgActualMValues((MSG_DATA_ACTUAL_MVALUES *) Message.pMessageData);                           break;
            case MSG_MEASURE_MODE:                 ComM_MsgMeasureMode((MSG_DATA_MEASURE_MODE *) Message.pMessageData);                               break;
            case MSG_START_FFS_FILE_TRANSFER:      ComM_MsgStartFFSFileTransfer ((MSG_DATA_START_MEASURE_FILE_TRANSFER *) Message.pMessageData);      break;
            case MSG_FFS_FILE_BLOCK:               ComM_MsgFFSFileBlock ((MSG_DATA_MEASURE_FILE_BLOCK *) Message.pMessageData);                       break;
            case MSG_FFS_FILE_EOF:                 ComM_MsgFFSFileEof();                                                                              break;
            case MSG_ENABLE_BLUETOOTH:             ComM_MsgEnableBluetooth ((MSG_DATA_ENABLE_BLUETOOTH * ) Message.pMessageData );                    break;
            default:
               break;
         }

         // free the memory
         if( Message.pMessageData != NULL )
         {
            FreeMemory( Message.pMessageData );
         }
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgInitSystem
///
/// @brief Handler for the message with the id MSG_INIT_SYSTEM
///
/// This function is the handler for the message with the id MSG_INIT_SYSTEM.
/// The method set all communication parameter for the communication via
/// bluetooth.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgInitSystem()
{
   // init the uart
   m_UART_Handle = AS1_Init(NULL);  

#ifdef _DEBUG
   // enable bluetooth
   GPIO1_SetPortBits (NULL, GPIO1_B_POWER_CTRL_MASK);
   
   // wait until bluetooth is started
   OS_Sleep(5000);
   
   // set bluetooth module in data mode
   m_UART_DataTransmitted = FALSE;
   AS1_SendBlock (m_UART_Handle, "ATD0\r", strlen("ATD0\r"));
   while(m_UART_DataTransmitted == FALSE);   
   
   OS_Sleep(1000);
   
  #ifdef SET_BAUDRATE_38400 
   m_UART_DataTransmitted = FALSE;
   AS1_SendBlock (m_UART_Handle, "ATL3\r", strlen("ATL3\r"));
   while(m_UART_DataTransmitted == FALSE);  
  #endif
  #ifdef SET_BAUDRATE_57600  
   m_UART_DataTransmitted = FALSE;
   AS1_SendBlock (m_UART_Handle, "ATL4\r", strlen("ATL4\r"));
   while(m_UART_DataTransmitted == FALSE);  
  #endif
  #ifdef SET_BAUDRATE_115200  
   m_UART_DataTransmitted = FALSE;
   AS1_SendBlock (m_UART_Handle, "ATL5\r", strlen("ATL5\r"));
   while(m_UART_DataTransmitted == FALSE);  
  #endif
  #ifdef SET_BAUDRATE_230400 
   m_UART_DataTransmitted = FALSE;
   AS1_SendBlock (m_UART_Handle, "ATL6\r", strlen("ATL6\r"));
   while(m_UART_DataTransmitted == FALSE);  
 #endif
#endif      

   // init the data type
   m_DataType = DATA_TYPE_MSG_HEADER;

   // Request the next data block (command header)
   ComM_InitReceiveNextDataBlock (SIZE_OF_DC_API_HEADER);
}

//-----------------------------------------------------------------------------
//  Function ComM_ProzessCmd
///
/// @brief Process the commands received over bluetooth
///
/// @param pCmdHeader (@c [in] CMD_HEADER *) - command header
///
/// @return UBYTE State: 0 - Command is processed completely, 1 - Command has
///               data and can only processed if all data bytes are received
///
/// This function process the commands received over bluetooth.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE ComM_ProzessCmd (CMD_HEADER * pCmdHeader)
{
   UBYTE RetVal = 0;

   if( pCmdHeader != NULL )
   {
      switch( pCmdHeader->CommandID )
      {
         case CMD_ID_REQ_DEVICE_INFO:
         {
            ComM_ProcessCmd_ReqDeviceInfo();
         }
         break;
         
         case CMD_ID_REQ_CONNECT:
         {
            ComM_ProcessCmd_ReqConnect();
         }
         break;

         case CMD_ID_REQ_DEVICE_CONFIG:
         {
            ComM_ProcessCmd_ReqDeviceConfig();
         }
         break;
         
         case CMD_ID_DEVICE_CONFIG:
         {
            // the command has data, we must get the data
            RetVal = 1;          
         }
         break;

         case CMD_ID_MEASURE_MODE:
         {
            // the command has data, we must get the data
            RetVal = 1;          
         }
         break;

         case CMD_ID_REQ_FFS_FILE:
         {
            // the command has data, we must get the data
            RetVal = 1;          
         }
         break;
         
         case CMD_ID_REQ_ACTUAL_MVALUES:
         {
            ComM_ProcessCmd_ReqActualMValues();
         }
         break;
         
         case CMD_ID_PING:
         {
            ComM_ProzessCmd_Ping();
         }
         break;
         
         case CMD_ID_RESET_MVALUES:
         {
            ComM_ProcessCmd_ResetMValues();
         }
         break;

         case CMD_ID_REQ_MEASURE_MODE:
         {
            ComM_ProcessCmd_ReqMeasureMode();
         }
         break;
         
         case CMD_ID_CLEAR_ALL_DATA_FILES:
         {
            ComM_ProcessCmd_ClearAllDataFiles();
         }
         break;

         case CMD_ID_ACK:
         {
            ComM_ProcessCmd_Ack();
         }
         break;
         
         default:
            break;
      }
   }

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmd_ReqDeviceInfo
///
/// @brief Handler for the bluetooth command CMD_ID_REQ_DEVICE_INFO
///
/// This function is the handler for the bluetooth command CMD_ID_REQ_DEVICE_INFO.
///
/// @status  Draft
/// @date    2013-05-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_ReqDeviceInfo (void)
{
   MESSAGE Msg;

   Msg.MessageID = MSG_REQ_DEVICE_INFO;
   Msg.pMessageData = NULL;

   if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
   {
      // Fehlerbehandlung
   }
   
   return;
}

//-----------------------------------------------------------------------------
//  Function ComM_ProzessCmd_Ping 
///
/// @brief Handler for the bluetooth command CMD_ID_PING
///
/// This function is the handler for the bluetooth command CMD_ID_PING.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProzessCmd_Ping ()
{     
   return ComM_SendPingAck();
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmd_ReqConnect
///
/// @brief Handler for the bluetooth command CMD_ID_REQ_CONNECT
///
/// This function is the handler for the bluetooth command CMD_ID_REQ_CONNECT.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_ReqConnect()
{
   ComM_SendAck();

   while (AS1_GetTxCompleteStatus(m_UART_Handle) == 0);

   ComM_SendConnect();

   return;
}

//-----------------------------------------------------------------------------
//  Function  ComM_ProcessCmd_ReqDeviceConfig
///
/// @brief Handler for the bluetooth command CMD_ID_REQ_DEVICE_CONFIG
///
/// This function is the handler for the bluetooth command CMD_ID_REQ_DEVICE_CONFIG.
///
/// @status  Draft
/// @date    2013-06-09
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-09 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_ReqDeviceConfig (void)
{
   MESSAGE Msg;

   Msg.MessageID = MSG_REQ_DEVICE_CONFIG;
   Msg.pMessageData = NULL;

   if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
   {
      // Fehlerbehandlung
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmd_ReqActualMValues
///
/// @brief Handler for the bluetooth command CMD_ID_REQ_ACTUAL_MVALUES
///
/// This function is the handler for the bluetooth command CMD_ID_REQ_ACTUAL_MVALUES.
///
/// @status  Draft
/// @date    2013-06-09
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-09 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_ReqActualMValues()
{
   MESSAGE Msg;

   Msg.MessageID = MSG_REQ_ACTUAL_MVALUES;
   Msg.pMessageData = NULL;

   if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
   {
      // Fehlerbehandlung
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmd_ResetMValues
///
/// @brief Handler for the bluetooth command CMD_ID_RESET_MVALUES
///
/// This function is the handler for the bluetooth command CMD_ID_RESET_MVALUES.
///
/// @status  Draft
/// @date    2013-06-14
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-14 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_ResetMValues()
{
   MESSAGE Msg;

   Msg.MessageID = MSG_RESET_MVALUES;
   Msg.pMessageData = NULL;

   if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
   {
      // Fehlerbehandlung
   }

   // send the acknowledge for the command
   ComM_SendAck();
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmd_ReqMeasureMode
///
/// @brief Handler for the bluetooth command CMD_ID_REQ_MEASURE_MODE
///
/// This function is the handler for the bluetooth command CMD_ID_REQ_MEASURE_MODE.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_ReqMeasureMode()
{
   MESSAGE Msg;

   Msg.MessageID = MSG_REQ_MEASURE_MODE;
   Msg.pMessageData = NULL;

   if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
   {
      // Fehlerbehandlung
   }

   // send the acknowledge for the command
   ComM_SendAck();
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmd_Ack
///
/// @brief Handler for the bluetooth command CMD_ID_ACK
///
/// This function is the handler for the bluetooth command CMD_ID_ACK.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_Ack()
{
   MESSAGE Msg;

   if (m_FileTransferActive == 1)
   {
      Msg.MessageID    = MSG_GET_NEXT_FFS_FILE_BLOCK;
      Msg.pMessageData = NULL;

      if (OS_SendMessage (TASK_ID_PROCESS_M, &Msg, WAIT_MAX) != ERR_OS_OK)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function  ComM_ProcessCmd_ClearAllDataFiles
///
/// @brief Handler for the bluetooth command CMD_ID_CLEAR_ALL_DATA_FILES
///
/// This function is the handler for the bluetooth command CMD_ID_CLEAR_ALL_DATA_FILES.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmd_ClearAllDataFiles()
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_CLEAR_ALL_DATA_FILES;
   Msg.pMessageData = NULL;

   if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
   {
      // Fehlerbehandlung
   }

   // send the acknowledge for the command
   ComM_SendAck();
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmdData 
///
/// @brief Process the commands with data received over bluetooth
///
/// @param pMsgRawData (@c [in] PUBYTE) - command data
///
/// @param pCmdHeader (@c [in] CMD_HEADER *) - command header
///
/// This function process the commands with data received over bluetooth.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmdData (PUBYTE pMsgRawData, CMD_HEADER * pCmdHeader)
{
   if( pCmdHeader != NULL )
   {
      switch (pCmdHeader->CommandID)
      {
         case CMD_ID_DEVICE_CONFIG:
         {
            ComM_ProcessCmdData_DeviceConfig (pCmdHeader, pMsgRawData);
         }
         break;

         case CMD_ID_MEASURE_MODE:
         {
            ComM_ProcessCmdData_MeasureMode (pCmdHeader, pMsgRawData);
         }
         break;

         case CMD_ID_REQ_FFS_FILE:
         {
            ComM_ProcessCmdData_ReqFFSFile (pCmdHeader, pMsgRawData);
         }
         break;

         default:
            break;
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmdData_ReqFFSFile 
///
/// @brief Handler for the bluetooth command CMD_ID_REQ_FFS_FILE
///
/// @param pCmdHeader (@c [in] CMD_HEADER *) - Data for the command
///
/// @param pMsgRawData (@c [in] PUBYTE) - Header of the command
///
/// This function is the handler for the bluetooth command CMD_ID_REQ_FFS_FILE.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmdData_ReqFFSFile (CMD_HEADER * pCmdHeader, PUBYTE pMsgRawData)
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_REQ_FFS_FILE;
   Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof( MSG_DATA_REQ_MEASURE_FILE ));

   if (Msg.pMessageData != NULL)
   {
      if (BaroAltiAPI_GetMsgReqMeasureFileData(pMsgRawData, (MSG_DATA_REQ_MEASURE_FILE *) Msg.pMessageData) == DC_API_OK)
      {
         ((MSG_DATA_REQ_MEASURE_FILE *) Msg.pMessageData)->MaxBlockSize = MAX_NUM_OF_PARAMS;

         if (OS_SendMessage (TASK_ID_PROCESS_M, &Msg, WAIT_MAX) != ERR_OS_OK)
         {
            // Fehlerbehandlung
         }
      }
      else
      {
         FreeMemory(Msg.pMessageData);
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmdData_MeasureMode 
///
/// @brief Handler for the bluetooth command CMD_ID_MEASURE_MODE
///
/// @param pCmdHeader (@c [in] CMD_HEADER *) - Data for the command
///
/// @param pMsgRawData (@c [in] PUBYTE) - Header of the command
///
/// This function is the handler for the bluetooth command CMD_ID_MEASURE_MODE.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmdData_MeasureMode (CMD_HEADER * pCmdHeader, PUBYTE pMsgRawData)
{
   MESSAGE Msg;

   Msg.MessageID = MSG_MEASURE_MODE;
   Msg.pMessageData = (UBYTE *) AllocateMemory( sizeof( MSG_DATA_MEASURE_MODE ) );

   if( Msg.pMessageData != NULL )
   {
      if (BaroAltiAPI_GetMsgMeasureModeData(pMsgRawData, (MSG_DATA_MEASURE_MODE *) Msg.pMessageData) == DC_API_OK)
      {
         if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
         {
            // Fehlerbehandlung
         }
      }
      else
      {
         FreeMemory(Msg.pMessageData);
      }
   }

   // send the acknowledge for the command
   ComM_SendAck();
}

//-----------------------------------------------------------------------------
//  Function ComM_ProcessCmdData_DeviceConfig 
///
/// @brief Handler for the bluetooth command CMD_ID_DEVICE_CONFIG
///
/// @param pCmdHeader (@c [in] CMD_HEADER *) - Data for the command
///
/// @param pMsgRawData (@c [in] PUBYTE) - Header of the command
///
/// This function is the handler for the bluetooth command CMD_ID_DEVICE_CONFIG.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_ProcessCmdData_DeviceConfig (CMD_HEADER * pCmdHeader, PUBYTE pMsgRawData)
{
   MESSAGE Msg;

   Msg.MessageID = MSG_DEVICE_CONFIG;
   Msg.pMessageData = (UBYTE *) AllocateMemory( sizeof( MSG_DATA_DEVICE_CONFIG ) );

   if( Msg.pMessageData != NULL )
   {
      if (BaroAltiAPI_GetMsgDeviceConfigData(pMsgRawData, (MSG_DATA_DEVICE_CONFIG *) Msg.pMessageData) == DC_API_OK)
      {
         if( OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX ) != ERR_OS_OK )
         {
            // Fehlerbehandlung
         }
      }
      else
      {
         FreeMemory(Msg.pMessageData);
      }
   }

   // send the acknowledge for the command
   ComM_SendAck();
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgDeviceInfo
///
/// @brief Handler for the message with the id MSG_DEVICE_INFO
///
/// @param pDeviceInfo (@c [in] MSG_DATA_DEVICE_INFO *) - Data for the command
///
/// This function is the handler for the message with the id MSG_DEVICE_INFO.
///
/// @status  Draft
/// @date    2013-05-30
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-30 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgDeviceInfo (MSG_DATA_DEVICE_INFO * pDeviceInfo)
{
   UBYTE SizeOfMsg;
   
   ComM_SendAck();

   while (AS1_GetTxCompleteStatus(m_UART_Handle) == 0);

   SizeOfMsg = BaroAltiAPI_SetMsgDeviceInfo (m_TransmitDataBuffer, pDeviceInfo);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock (m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }
}

//-----------------------------------------------------------------------------
//  Function  ComM_MsgActualMValues 
///
/// @brief Handler for the message with the id MSG_ACTUAL_MVALUES
///
/// @param pActualMValues (@c [in] MSG_DATA_ACTUAL_MVALUES *) - Data for the command
///
/// This function is the handler for the message with the id MSG_ACTUAL_MVALUES.
///
/// @status  Draft
/// @date    2013-06-09
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-09 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgActualMValues (MSG_DATA_ACTUAL_MVALUES * pActualMValues)
{
   UBYTE SizeOfMsg;

   ComM_SendAck();

   while (AS1_GetTxCompleteStatus(m_UART_Handle) == 0);

   SizeOfMsg = BaroAltiAPI_SetMsgActualMValues (m_TransmitDataBuffer, pActualMValues);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock (m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgMeasureMode
///
/// @brief Handler for the message with the id MSG_MEASURE_MODE
///
/// @param pMeasureMode (@c [in] MSG_DATA_MEASURE_MODE *) - Data for the command
///
/// This function is the handler for the message with the id MSG_MEASURE_MODE.
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgMeasureMode (MSG_DATA_MEASURE_MODE * pMeasureMode)
{
   UBYTE SizeOfMsg;

   ComM_SendAck();

   while (AS1_GetTxCompleteStatus(m_UART_Handle) == 0);

   SizeOfMsg = BaroAltiAPI_SetMsgMeasureMode (m_TransmitDataBuffer, pMeasureMode);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock (m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgDeviceConfig 
///
/// @brief Handler for the message with the id MSG_DEVICE_CONFIG
///
/// @param pDeviceConfig (@c [in] MSG_DATA_DEVICE_CONFIG *) - Data for the command
///
/// This function is the handler for the message with the id MSG_DEVICE_CONFIG.
///
/// @status  Draft
/// @date    2013-06-09
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-09 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgDeviceConfig (MSG_DATA_DEVICE_CONFIG * pDeviceConfig)
{
   UBYTE SizeOfMsg;

   if (pDeviceConfig == NULL)
      return;
   
   ComM_SendAck();

   while (AS1_GetTxCompleteStatus(m_UART_Handle) == 0);

   SizeOfMsg = BaroAltiAPI_SetMsgDeviceConfig (m_TransmitDataBuffer, pDeviceConfig);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock (m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgUartDBReceived
///
/// @brief Handler for the message with the id MSG_UART_DB_RECEIVED
///
/// This function is the handler for the message with the id MSG_UART_DB_RECEIVED.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgUartDBReceived()
{
   switch (m_DataType)
   {
      case DATA_TYPE_MSG_HEADER:
      {
         if( BaroALtiAPI_GetCmdHeader( m_ReceiveDataBuffer, &m_CmdHeader ) == DC_API_OK )
         {
            if( ComM_ProzessCmd (&m_CmdHeader) == 0)
            {
               m_DataType = DATA_TYPE_MSG_HEADER;
               ComM_InitReceiveNextDataBlock( SIZE_OF_DC_API_HEADER );
            }
            else
            {
               m_DataType = DATA_TYPE_MSG_DATA;
               ComM_InitReceiveNextDataBlock( m_CmdHeader.SizeOfPacket );
            }
         }
         else
         {
            // Header korrupt, Daten verwerfen
            m_DataType = DATA_TYPE_MSG_HEADER;
            ComM_InitReceiveNextDataBlock( SIZE_OF_DC_API_HEADER );
         }
      }
      break;

      case DATA_TYPE_MSG_DATA:
      {
         m_DataType = DATA_TYPE_MSG_HEADER;

         ComM_ProcessCmdData( m_ReceiveDataBuffer, &m_CmdHeader );

         ComM_InitReceiveNextDataBlock( SIZE_OF_DC_API_HEADER );
      }
      break;

      default:
         break;
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgStartFFSFileTransfer 
///
/// @brief Handler for the message with the id MSG_START_FFS_FILE_TRANSFER
///
/// @param pStartMeasureFileTransfer (@c [in] MSG_DATA_START_MEASURE_FILE_TRANSFER *) - Data for the command
///
/// This is the handler for the message with the id MSG_START_FFS_FILE_TRANSFER.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgStartFFSFileTransfer (MSG_DATA_START_MEASURE_FILE_TRANSFER * pStartMeasureFileTransfer)
{
   UBYTE SizeOfMsg;

   m_FileTransferActive = 1;
   m_iCurFileBlock      = 1;

   m_TotalNumOfBlocks = pStartMeasureFileTransfer->TotalNumOfDataBlocks;
   m_NumOfDataInFile  = pStartMeasureFileTransfer->NumOfDataInFile;

   ComM_SendAck();

   while (AS1_GetTxCompleteStatus(m_UART_Handle) == 0);

   SizeOfMsg = BaroAltiAPI_SetFileHeader (m_TransmitDataBuffer, m_NumOfDataInFile);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock (m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgFFSFileBlock 
///
/// @brief Handler for the message with the id MSG_START_FFS_FILE_TRANSFER
///
/// @param pMeasureFileBlock (@c [in] MSG_DATA_MEASURE_FILE_BLOCK *) - Data for the command
///
/// This is the handler for the message with the id MSG_START_FFS_FILE_TRANSFER.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgFFSFileBlock (MSG_DATA_MEASURE_FILE_BLOCK * pMeasureFileBlock)
{
   UWORD SizeOfMsg;

   SizeOfMsg = BaroAltiAPI_SetFileBlock (m_TransmitDataBuffer, pMeasureFileBlock->MemoryHandle, pMeasureFileBlock->SizeOfMeasureFileBlock, m_iCurFileBlock, m_TotalNumOfBlocks );

   if (SizeOfMsg > 0)
   {      
      AS1_SendBlock (m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);

      while (AS1_GetTxCompleteStatus(m_UART_Handle) == 0);

      m_iCurFileBlock++;
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgFFSFileEof
///
/// @brief Handler for the message with the id MSG_FFS_FILE_EOF
///
/// This is the handler for the message with the id MSG_FFS_FILE_EOF.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgFFSFileEof()
{
   UBYTE SizeOfMsg;

   m_FileTransferActive = 0;

   SizeOfMsg = BaroAltiAPI_SetFFSFileComplete (m_TransmitDataBuffer );

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock (m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_MsgEnableBluetooth 
///
/// @brief Handler for the message with the id MSG_ENABLE_BLUETOOTH
///
/// @param pEnableBluetooth (@c [in,out] MSG_DATA_ENABLE_BLUETOOTH *) - Data for the command
///
/// This is the handler for the message with the id MSG_ENABLE_BLUETOOTH.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_MsgEnableBluetooth (MSG_DATA_ENABLE_BLUETOOTH *  pEnableBluetooth )
{
   if (pEnableBluetooth == NULL)
      return;

   if (pEnableBluetooth->EnableBluetooth == 1)
   {
      GPIO1_SetPortBits(NULL, GPIO1_B_POWER_CTRL_MASK);
   }
   else
   {
      GPIO1_ClearPortBits(NULL, GPIO1_B_POWER_CTRL_MASK);
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_SendPingAck
///
/// @brief Send the command CMD_ID_PING_ACK via bluetooth
///
/// This method send the command CMD_ID_PING_ACK via bluetooth.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_SendPingAck()
{
   UBYTE SizeOfMsg;

   SizeOfMsg = BaroAltiAPI_SetMsgPingAck(m_TransmitDataBuffer);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock( m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }

   return;
}

//-----------------------------------------------------------------------------
//  Function ComM_SendAck
///
/// @brief Send the command CMD_ID_ACK via bluetooth
///
/// This function send the command CMD_ID_ACK via bluetooth.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_SendAck()
{
   UBYTE SizeOfMsg;

   SizeOfMsg = BaroAltiAPI_SetMsgAck(m_TransmitDataBuffer);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock( m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }

   return;
}

//-----------------------------------------------------------------------------
//  Function ComM_SendNak
///
/// @brief Send the command CMD_ID_NAK via bluetooth
///
/// This function sSend the command CMD_ID_ACK via bluetooth.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_SendNak()
{
   UBYTE SizeOfMsg;

   SizeOfMsg = BaroAltiAPI_SetMsgNak(m_TransmitDataBuffer);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock( m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }

   return;
}

//-----------------------------------------------------------------------------
//  Function ComM_SendConnect
///
/// @brief Send the command CMD_ID_CONNECT via bluetooth
///
/// This function send the command CMD_ID_CONNECT via bluetooth.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_SendConnect()
{
   UBYTE SizeOfMsg;

   SizeOfMsg = BaroAltiAPI_SetMsgConnect(m_TransmitDataBuffer, 0x01, 0x01, MAX_NUM_OF_PARAMS);

   if (SizeOfMsg > 0)
   {
      AS1_SendBlock( m_UART_Handle, m_TransmitDataBuffer, SizeOfMsg);
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_InitReceiveNextDataBlock
///
/// @brief Request the next data block from the uart
///
/// @param SizeOfNextDataBlock (@c [in] UBYTE) - Size of the next data block
///
/// This function request the next data block from the uart.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_InitReceiveNextDataBlock (UBYTE SizeOfNextDataBlock)
{
   if( SizeOfNextDataBlock <= MAX_NUM_OF_REC_DATA_BYTES )
   {
      if( AS1_ReceiveBlock( m_UART_Handle, m_ReceiveDataBuffer, SizeOfNextDataBlock ) != ERR_OK)
      {
         // Fehlerbehandlung
      }
   }
}

//-----------------------------------------------------------------------------
//  Function ComM_DataBlockReceived
///
/// @brief Event from the ISR whenever a complete data block is received
///
/// This function is called from the ISR whenever a complete data block is
/// received. The function send a message MSG_UART_DB_RECEIVED to the
/// communication manager.
///
/// @status  Draft
/// @date    2013-05-27
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-05-27 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void ComM_DataBlockReceived()
{
   MESSAGE Msg;

   Msg.MessageID    = MSG_UART_DB_RECEIVED;
   Msg.pMessageData = NULL;

   if( OS_SendMessageNB( TASK_ID_COM_M, &Msg ) != ERR_OS_OK )
   {
      // Fehlerbehandlung
   }
}
   
