//-----------------------------------------------------------------------------
//  FlashMemM.c
/// @file
///
/// @brief Implementation of all function for handle the flash memory
///
/// All functions for the handling the flash memory are implemented in this module.
///
/// @copyright 2012-2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-23 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------
#include "null.h"

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "PE_LDD.h"
#include "FlashMemM.h"
#include "OS.h"
#include "MessageDefs.h"
#include "Tools.h"
//#include "SM1.h"
#include "SST25PFx.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------
/// Enum fr Filezugriff
typedef enum tagFILE_ACCESS
{
   NO_ACCESS = 0,             ///< Zugriffsrecht unbestimmt
   READ_ACCESS,               ///< Lesezugriff
   WRITE_ACCESS,              ///< Schreibzugriff
   READ_WRITE_ACCESS          ///< Lese- und Schreibzugriff
} FILE_ACCESS;

/// Beschreibungsstruktur fr aktives File
typedef struct tagACTIVE_FILE
{
   UBYTE       FileNumber;    ///< Dateinummer
   FILE_ACCESS FileAccess;    ///< Zugriffsrecht auf die Datei
   UDWORD      ReadAddress;   ///< Lesezeiger
   UDWORD      WriteAddress;  ///< Schreibzeiger
} ACTIVE_FILE;

/// Directoryeintrag fr eine Datei
typedef struct tagDIRECTORY_ENTRY
{
   UDWORD FileStartAddress;   ///< Startaddress of a file in the flash memory
   UDWORD FileEndAddress;     ///< Endaddress of a file in the flash memory
} DIRECTORY_ENTRY;

/// Startadresse fr die Konfigurationsdaten
#define STARTADDRESS_CONFIG_DATA    0x000000

/// Startadresse fr das Directory der Dateien
#define STARTADDRESS_FFS_DIRECTORY  0x001000

/// Startaddress fr die Dateien
#define STARTADDRESS_FFS_FILES      0x002000

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------
/// Active file structure
ACTIVE_FILE m_ActiveFile;
/// Next free address for writing file data
UDWORD      m_FlashNextFreeFileAddress;
/// Number of files in the simple file system
UWORD       m_NumOfFilesInFFS;
/// Flag file system is corrupt
UBYTE       m_FileSystemIsCorrupt;
/// Startaddress for a file transfer
UDWORD      m_FileTransferStartAddress;
/// Endaddress for a file transfer
UDWORD      m_FileTransferEndAddress;
/// Actual address during a file transfer
UDWORD      m_FileTransferActAddress;
/// Maximal block size for a file transfer
UWORD       m_FileTransferMaxBlockSize;
/// Pointer to the data for a file transfer
PUBYTE      m_pFileDataForTransfer;

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------
// Handler for the message with the id MSG_REQ_CONFIG_DATA
void FlashMemM_MsgReqConfigData();
// Handler for the message with the id MSG_CLOSE_ACTIVE_FILE
void FlashMemM_MsgCloseActiveFile();
// Handler for the message with the id MSG_INSERT_DATASET_IN_FILE
void FlashMemM_MsgInsertDatasetInFile (MSG_DATA_INSERT_DATASET * pInsertDataset);
// Handler for the message with the id MSG_CREATE_NEW_MEASURE_FILE
void FlashMemM_MsgCreateNewMeasureFile();
// Handler for the message with the id MSG_INIT_SYSTEM
void FlashMemM_MsgInitSystem();
// Handler for the message with the id MSG_REQ_FALSHMEM_INFO
void FlashMemM_MsgReqFlashMemInfo();
// Handler for the message with the id MSG_WRITE_DEVICE_CONFIG_DATA
void FlashMemM_MsgWriteDeviceConfigData(MSG_DATA_DEVICE_CONFIG * pConfigData);
// Handler for the message with the id MSG_READ_DEVICE_CONFIG_DATA
void FlashMemM_MsgReqReadDeviceConfigData();
// Checks the file system for errors
void FlashMemM_CheckFileSystem();
// Handler for the message with the id MSG_FORMAT_FILE_SYSTEM
void FlashMemM_MsgFormatFileSystem();
// Repair simple errors in file system directory entrys
UBYTE FlashMemM_RepairFileSystem (UDWORD DirectoryEntryAddress, DIRECTORY_ENTRY * pDirectoryEntry);
// Handler for the message with the id MSG_REQ_FFS_FILE
void FlashMemM_MsgReqFFSFile (MSG_DATA_REQ_FFS_FILE * pReqFFSFile);
// Starts the transfer of the next file data block
void FlashMemM_StartTransferNextBlock ();
// Starts a file transfer
void FlashMemM_StartFileTransfer();
// Handler for the message with the id MSG_GET_NEXT_FFS_FILE_BLOCK
void FlashMemM_MsgGetNextFFSFileBlock();

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function FlashMemM_TaskFunction
///
/// @brief Taskfunction of the FlashMemM task
///
/// @param pvParameters (@c [in] void *) - unused
///
/// This is the taskfunction of the FlashMemM task.
///
/// @status  Draft
/// @date    2013-04-23
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-23 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_TaskFunction( void * pvParameters )
{
   MESSAGE Message;

   while(1)
   {
      if( OS_ReceiveMessage( TASK_ID_FLASHMEM_M, &Message, WAIT_MAX ) == ERR_OS_OK )
      {
         switch( Message.MessageID )
         {
            case MSG_INIT_SYSTEM:                  FlashMemM_MsgInitSystem();                                                             break;
            case MSG_REQ_CONFIG_DATA:              FlashMemM_MsgReqConfigData();                                                          break;
            case MSG_CREATE_NEW_MEASURE_FILE:      FlashMemM_MsgCreateNewMeasureFile();                                                   break;
            case MSG_CLOSE_ACTIVE_FILE:            FlashMemM_MsgCloseActiveFile();                                                        break;
            case MSG_INSERT_DATASET_IN_FILE:       FlashMemM_MsgInsertDatasetInFile ((MSG_DATA_INSERT_DATASET *) Message.pMessageData);   break;
            case MSG_REQ_FALSHMEM_INFO:            FlashMemM_MsgReqFlashMemInfo();                                                        break;
            case MSG_REQ_READ_DEVICE_CONFIG_DATA:  FlashMemM_MsgReqReadDeviceConfigData();                                                break;
            case MSG_WRITE_DEVICE_CONFIG_DATA:     FlashMemM_MsgWriteDeviceConfigData ((MSG_DATA_DEVICE_CONFIG *) Message.pMessageData);  break;
            case MSG_FORMAT_FILE_SYSTEM:           FlashMemM_MsgFormatFileSystem();                                                       break;
            case MSG_REQ_FFS_FILE:                 FlashMemM_MsgReqFFSFile ((MSG_DATA_REQ_FFS_FILE *) Message.pMessageData);              break;
            case MSG_GET_NEXT_FFS_FILE_BLOCK:      FlashMemM_MsgGetNextFFSFileBlock();                                                    break;
            default:
               break;
         }

         // Speicher wenn notwendig freigeben
         if( Message.pMessageData != NULL )
         {
            FreeMemory( Message.pMessageData );
         }
      }
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgInitSystem
///
/// @brief Handler for the message with the id MSG_INIT_SYSTEM
///
/// This is the handler for the message with the id MSG_INIT_SYSTEM.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgInitSystem()
{
   MESSAGE Msg;

   LDD_TDeviceData * SPI_Handle;

   m_ActiveFile.FileNumber    = 0;
   m_ActiveFile.FileAccess    = NO_ACCESS;
   m_ActiveFile.ReadAddress   = 0;
   m_ActiveFile.WriteAddress  = 0;

   //-------------------------------------------------------------------------
   // SPI Initialisierung
   //-------------------------------------------------------------------------
   SPI_Handle = SM1_Init(NULL);

   //-------------------------------------------------------------------------
   // Flashtreiber Initialisierung
   //-------------------------------------------------------------------------
   SST25PFX_Init(SPI_Handle);

   // Blcke zum schreiben freigeben
   SST25PFX_DisableBlockProtection();
   
   FlashMemM_CheckFileSystem();

   // Send number of files in ffs to the process manager
   Msg.MessageID = MSG_NUM_OF_FILES_IN_FFS;
   Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_NUM_OF_FILES_IN_FFS));

   if( Msg.pMessageData )
   {
      ((MSG_DATA_NUM_OF_FILES_IN_FFS *)Msg.pMessageData)->NumOfFilesInFFS = m_NumOfFilesInFFS;

      (void)OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX );
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgReqFlashMemInfo
///
/// @brief Handler for the message with the id MSG_REQ_FALSHMEM_INFO
///
/// This is the handler for the message with the id MSG_REQ_FALSHMEM_INFO.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgReqFlashMemInfo()
{
   UBYTE ManID;
   UBYTE DeviceID;
   MESSAGE Msg;

   SST25PFX_ReadID( &ManID, &DeviceID );

   Msg.MessageID = MSG_FALSHMEM_INFO;
   Msg.pMessageData = (PUBYTE) AllocateMemory(sizeof(MSG_DATA_FALSHMEM_INFO));

   if( Msg.pMessageData )
   {
      // Daten in die Nachricht formatieren
      switch(ManID)
      {
         case 0xBF:  ((MSG_DATA_FALSHMEM_INFO *)Msg.pMessageData)->FlashDeviceType = FLASH_MANFCT_MICROCHIP;  break;
         default:    ((MSG_DATA_FALSHMEM_INFO *)Msg.pMessageData)->FlashDeviceType = FLASH_MANFCT_NO_INFO;    break;
      }

      switch(DeviceID)
      {
         case 0x8C:  ((MSG_DATA_FALSHMEM_INFO *)Msg.pMessageData)->FlashSize = FLASH_SIZE_2MBIT;     break;
         case 0x8D:  ((MSG_DATA_FALSHMEM_INFO *)Msg.pMessageData)->FlashSize = FLASH_SIZE_4MBIT;     break;
         case 0x8E:  ((MSG_DATA_FALSHMEM_INFO *)Msg.pMessageData)->FlashSize = FLASH_SIZE_8MBIT;     break;
         default:    ((MSG_DATA_FALSHMEM_INFO *)Msg.pMessageData)->FlashSize = FLASH_SIZE_NO_INFO;   break;
      }

      // Nachricht an den Prozessmanager versenden
      (void)OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX );
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgCreateNewMeasureFile 
///
/// @brief Handler for the message with the id MSG_CREATE_NEW_MEASURE_FILE
///
/// This is the handler for the message with the id MSG_CREATE_NEW_MEASURE_FILE.
///
/// @status  Draft
/// @date    2013-06-23
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-23 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgCreateNewMeasureFile ()
{
   DIRECTORY_ENTRY * pDirectoryEntry;
   
   // check if no file is already open
   if (m_ActiveFile.FileNumber != 0 && m_FileSystemIsCorrupt == 0)
   {
      // another file is already open, please close the other file first
      return;
   }
   
   // allocate memory for the directory entry
   pDirectoryEntry = (DIRECTORY_ENTRY *) AllocateMemory(sizeof(DIRECTORY_ENTRY));

   if (pDirectoryEntry)
   {
      // fill the directory entry
      pDirectoryEntry->FileStartAddress = m_FlashNextFreeFileAddress;
      pDirectoryEntry->FileEndAddress = 0xFFFFFFFF;
      
      // write the directory entry
      if (SST25PFX_WriteBlock (STARTADDRESS_FFS_DIRECTORY + (m_NumOfFilesInFFS * sizeof (DIRECTORY_ENTRY)), 
                               (PUBYTE) pDirectoryEntry, 
                               sizeof(DIRECTORY_ENTRY)) == sizeof(DIRECTORY_ENTRY))
      {      
         // init the active file member
         m_ActiveFile.FileNumber = m_NumOfFilesInFFS + 1;
         m_ActiveFile.FileAccess = READ_WRITE_ACCESS;
         m_ActiveFile.ReadAddress = m_FlashNextFreeFileAddress;
         m_ActiveFile.WriteAddress = m_FlashNextFreeFileAddress;
      }
      else
      {
         // todo errorhandling
      }
      
      // free the allocated memory
      FreeMemory(pDirectoryEntry);
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgCloseActiveFile
///
/// @brief Handler for the message with the id MSG_CLOSE_ACTIVE_FILE
///
/// This is the handler for the message with the id MSG_CLOSE_ACTIVE_FILE.
///
/// @status  Draft
/// @date    2013-06-23
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-23 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgCloseActiveFile()
{
   DIRECTORY_ENTRY * pDirectoryEntry;
   MESSAGE           Msg;

   // check if a file is already open
   if (m_ActiveFile.FileNumber == 0)
   {
      // no file is open -> there's nothing to do
      return;
   }
   
   // allocate memory for the directory entry
   pDirectoryEntry = (DIRECTORY_ENTRY *) AllocateMemory(sizeof(DIRECTORY_ENTRY));

   if (pDirectoryEntry)
   {
      // read the directory entry from the opened file
      if( SST25PFX_ReadBlock (STARTADDRESS_FFS_DIRECTORY + (m_NumOfFilesInFFS * sizeof (DIRECTORY_ENTRY)),
                              (PUBYTE)pDirectoryEntry,
                              sizeof (DIRECTORY_ENTRY)) == sizeof(DIRECTORY_ENTRY))
      {      
         // fill the end file address of the directory entry

         // is any data in the file
         if (m_ActiveFile.WriteAddress == m_FlashNextFreeFileAddress)
         {
            // no
            pDirectoryEntry->FileEndAddress = m_ActiveFile.WriteAddress;
         }
         else
         {
            // yes
            pDirectoryEntry->FileEndAddress = m_ActiveFile.WriteAddress - 1;
         }

         // write the directory entry
         if (SST25PFX_WriteBlock (STARTADDRESS_FFS_DIRECTORY + (m_NumOfFilesInFFS * sizeof (DIRECTORY_ENTRY)), 
                                  (PUBYTE) pDirectoryEntry, 
                                  sizeof(DIRECTORY_ENTRY)) == sizeof(DIRECTORY_ENTRY))
         {            
            // increment the file pointer
            m_NumOfFilesInFFS++;
            
            // set the next free adress for writing file data
            m_FlashNextFreeFileAddress = m_ActiveFile.WriteAddress;
            
            // reset the member for the active file so we can detect 'no file is opened'
            m_ActiveFile.FileNumber    = 0;
            m_ActiveFile.FileAccess    = NO_ACCESS;
            m_ActiveFile.ReadAddress   = 0;
            m_ActiveFile.WriteAddress  = 0;
         }
      }
      
      // free the allocated memory
      FreeMemory(pDirectoryEntry);

      // Send number of files in ffs to the process manager
      Msg.MessageID = MSG_NUM_OF_FILES_IN_FFS;
      Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_NUM_OF_FILES_IN_FFS));

      if( Msg.pMessageData )
      {
         ((MSG_DATA_NUM_OF_FILES_IN_FFS *)Msg.pMessageData)->NumOfFilesInFFS = m_NumOfFilesInFFS;

         (void)OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX );
      }
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgInsertDatasetInFile 
///
/// @brief Handler for the message with the id MSG_INSERT_DATASET_IN_FILE
///
/// @param pInsertDataset (@c [in] MSG_DATA_INSERT_DATASET *) - Command data
///
/// This is the handler for the message with the id MSG_INSERT_DATASET_IN_FILE.
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgInsertDatasetInFile (MSG_DATA_INSERT_DATASET * pInsertDataset)
{
   MESSAGE Msg;

   if (pInsertDataset == NULL || m_ActiveFile.FileNumber == 0)
	   return;
		
   if (SST25PFX_WriteBlock (m_ActiveFile.WriteAddress, 
                            (PUBYTE) pInsertDataset, 
                            sizeof(MSG_DATA_INSERT_DATASET)) == sizeof(MSG_DATA_INSERT_DATASET))
   {
    	m_ActiveFile.WriteAddress += sizeof(MSG_DATA_INSERT_DATASET);
   }
   else
   {
      // set the system to normal mode (flash memory is full)
      Msg.MessageID    = MSG_MEASURE_MODE;
      Msg.pMessageData = (UBYTE *) AllocateMemory (sizeof( MSG_DATA_MEASURE_MODE ));

      if( Msg.pMessageData != NULL )
      {
         ((MSG_DATA_MEASURE_MODE *) Msg.pMessageData)->MeasureMode = MM_NORMAL;

         if (OS_SendMessage (TASK_ID_PROCESS_M, &Msg, WAIT_MAX) != ERR_OS_OK)
         {
            // Fehlerbehandlung
         }
      }
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgReqConfigData
///
/// @brief Handler for the message with the id MSG_REQ_CONFIG_DATA
///
/// This is the handler for the message with the id MSG_REQ_CONFIG_DATA.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgReqConfigData()
{
   MESSAGE Msg;

   Msg.MessageID = MSG_CONFIG_DATA;
   Msg.pMessageData = (UBYTE *) AllocateMemory(sizeof(MSG_DATA_DEVICE_CONFIG));

   if( Msg.pMessageData )
   {
      if( SST25PFX_ReadBlock(STARTADDRESS_CONFIG_DATA, Msg.pMessageData, sizeof(MSG_DATA_DEVICE_CONFIG)) == sizeof(MSG_DATA_DEVICE_CONFIG))
      {
         // Nachricht an den Prozessmanager versenden
         (void)OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX );
      }
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgWriteDeviceConfigData
///
/// @brief Handler for the message with the id MSG_WRITE_DEVICE_CONFIG_DATA
///
/// @param pConfigData (@c [in] MSG_DATA_CONFIG_DATA *) - Command data
///
/// This is the handler for the message with the id MSG_WRITE_DEVICE_CONFIG_DATA.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgWriteDeviceConfigData (MSG_DATA_DEVICE_CONFIG * pConfigData)
{
   if( pConfigData == NULL )
      return;

   SST25PFX_SectorErase( STARTADDRESS_CONFIG_DATA );

   if( SST25PFX_WriteBlock( STARTADDRESS_CONFIG_DATA, (PUBYTE) pConfigData, sizeof(MSG_DATA_DEVICE_CONFIG)) != sizeof(MSG_DATA_DEVICE_CONFIG) )
   {
      // todo: Fehlerbehandlung
   }

   /// @todo EM 2013-06-20 Maybe call back to the processm that the writing
   /// of the device config was successfully
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgFormatFileSystem
///
/// @brief Handler for the message with the id MSG_FORMAT_FILE_SYSTEM
///
/// This is the handler for the message with the id MSG_FORMAT_FILE_SYSTEM.
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgFormatFileSystem()
{
   MESSAGE Msg;
   int iSector;
   int iBlock;

   SST25PFX_SectorErase (STARTADDRESS_FFS_DIRECTORY);
   
   for (iSector = 1; iSector < 8; iSector++)
   {
      SST25PFX_SectorErase (iSector * 0x1000);
   }
   
   SST25PFX_32KBlockErase (0x8000);
   
   for (iBlock = 1; iBlock < 16; iBlock++)
   {
      SST25PFX_64KBlockErase (iBlock * 0x10000);      
   }
   
   m_FileSystemIsCorrupt = 0;
   
   m_FlashNextFreeFileAddress = STARTADDRESS_FFS_FILES;

   m_NumOfFilesInFFS = 0;
   
   m_ActiveFile.FileAccess   = NO_ACCESS;
   m_ActiveFile.FileNumber   = 0;
   m_ActiveFile.ReadAddress  = 0;
   m_ActiveFile.WriteAddress = 0;
   
   // Send number of files in ffs to the process manager
   Msg.MessageID = MSG_NUM_OF_FILES_IN_FFS;
   Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_NUM_OF_FILES_IN_FFS));

   if( Msg.pMessageData )
   {
      ((MSG_DATA_NUM_OF_FILES_IN_FFS *)Msg.pMessageData)->NumOfFilesInFFS = m_NumOfFilesInFFS;

      (void)OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX );
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgReqReadDeviceConfigData
///
/// @brief Handler for the message with the id MSG_READ_DEVICE_CONFIG_DATA
///
/// This is the handler for the message with the id MSG_READ_DEVICE_CONFIG_DATA.
///
/// @status  Draft
/// @date    2013-06-23
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-23 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgReqReadDeviceConfigData()
{
   MESSAGE Msg;

   Msg.MessageID = MSG_READ_DEVICE_CONFIG_DATA;
   Msg.pMessageData = (UBYTE *) AllocateMemory(sizeof(MSG_DATA_DEVICE_CONFIG));

   if( Msg.pMessageData )
   {
      if( SST25PFX_ReadBlock(STARTADDRESS_CONFIG_DATA, Msg.pMessageData, sizeof(MSG_DATA_DEVICE_CONFIG)) == sizeof(MSG_DATA_DEVICE_CONFIG))
      {
         // Nachricht an den Prozessmanager versenden
         (void)OS_SendMessage( TASK_ID_PROCESS_M, &Msg, WAIT_MAX );
      }
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_CheckFileSystem
///
/// @brief Checks the file system for errors
///
/// This function is checking the file system for erors. Simple errors can
/// repair directly.
///
/// @see FlashMemM_RepairFileSystem()
///
/// @status  Draft
/// @date    2013-06-23
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-23 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_CheckFileSystem()
{
   UDWORD            CurAddress;
   DIRECTORY_ENTRY * pDirectoryEntry;

   m_FileSystemIsCorrupt = 0;

   m_NumOfFilesInFFS = 0;

   pDirectoryEntry = (DIRECTORY_ENTRY *) AllocateMemory(sizeof(DIRECTORY_ENTRY));

   if (pDirectoryEntry)
   {
      CurAddress = STARTADDRESS_FFS_DIRECTORY;

      do 
      {
         if (SST25PFX_ReadBlock(CurAddress, (PUBYTE)pDirectoryEntry, sizeof(DIRECTORY_ENTRY)) == sizeof(DIRECTORY_ENTRY))
         {
            if(   pDirectoryEntry->FileStartAddress == 0xFFFFFFFF
               && pDirectoryEntry->FileEndAddress   == 0xFFFFFFFF)
            {
               // Abbruch, wir haben alle Eintrge durchsucht
               if (m_NumOfFilesInFFS == 0)
               {
                  // Filesystem ist leer
                  m_FlashNextFreeFileAddress = STARTADDRESS_FFS_FILES;
               }

               break;
            }
            else if(   pDirectoryEntry->FileStartAddress != 0xFFFFFFFF
                    && pDirectoryEntry->FileEndAddress   == 0xFFFFFFFF)
            {
               // Datei wurde nicht korrekt geschlossen
               if (FlashMemM_RepairFileSystem (CurAddress, pDirectoryEntry) != 0)
               {
                  m_FileSystemIsCorrupt = 1;
               }
               break;
            }
            else if (pDirectoryEntry->FileEndAddress < pDirectoryEntry->FileStartAddress)
            {
               // Startadresse ist groesser wie die Endadresse
               m_FileSystemIsCorrupt = 1;
               break;
            }
            else if (   pDirectoryEntry->FileStartAddress < STARTADDRESS_FFS_FILES
                     || pDirectoryEntry->FileEndAddress < STARTADDRESS_FFS_FILES)
            {
               // Start- oder Endadresse liegen in einem ungltigen Bereich               
               m_FileSystemIsCorrupt = 1;
               break;               
            }
            
            m_FlashNextFreeFileAddress = pDirectoryEntry->FileEndAddress + 1;
            m_NumOfFilesInFFS++;
         }
         else
         {
            m_FileSystemIsCorrupt = 1;
            break;
         }

         CurAddress += sizeof(DIRECTORY_ENTRY);
      } while (1);

      // free the allocated memory
      FreeMemory (pDirectoryEntry);
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_RepairFileSystem 
///
/// @brief Repair simple errors in file system directory entrys
///
/// @param DirectoryEntryAddress (@c [in] UDWORD) - Address for the directory entry
///
/// @param pDirectoryEntry (@c [in,out] DIRECTORY_ENTRY *) - Data for directory entry
///
/// @return UBYTE Return Code
///
/// This function repairs simple errors in file system directory entrys.
///
/// @see FlashMemM_CheckFileSystem()
///
/// @status  Draft
/// @date    2013-06-26
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-26 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE FlashMemM_RepairFileSystem (UDWORD DirectoryEntryAddress, DIRECTORY_ENTRY * pDirectoryEntry)
{
   UBYTE RetVal = 0;
   MSG_DATA_INSERT_DATASET * pDataset;
   UDWORD CurAddress;
   
   if (pDirectoryEntry != NULL)
   {
      pDataset = (MSG_DATA_INSERT_DATASET *) AllocateMemory(sizeof(MSG_DATA_INSERT_DATASET));

      if (pDataset != NULL)
      {
         CurAddress = pDirectoryEntry->FileStartAddress;

         do 
         {
            if (SST25PFX_ReadBlock (CurAddress, (PUBYTE)pDataset, sizeof(MSG_DATA_INSERT_DATASET)) != sizeof(MSG_DATA_INSERT_DATASET))
            {
               FreeMemory (pDataset);
               return 1;
            }
            
            CurAddress += sizeof(MSG_DATA_INSERT_DATASET);
         } while (pDataset->TimestampHi != 0xFFFFFFFF);

         CurAddress -= sizeof(MSG_DATA_INSERT_DATASET);

         if (pDirectoryEntry->FileStartAddress == CurAddress)
         {
            pDirectoryEntry->FileEndAddress = pDirectoryEntry->FileStartAddress;
         }
         else
         {
            pDirectoryEntry->FileEndAddress = CurAddress - 1;
         }
         
         // write the directory entry
         if (SST25PFX_WriteBlock (DirectoryEntryAddress, 
                                  (PUBYTE) pDirectoryEntry, 
                                  sizeof(DIRECTORY_ENTRY)) != sizeof(DIRECTORY_ENTRY))
         {
            RetVal = 1;
         }

         FreeMemory (pDataset);
      }
   }
   else
   {
      RetVal = 1;
   }

   return RetVal;
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgReqFFSFile 
///
/// @brief Handler for the message with the id MSG_REQ_FFS_FILE
///
/// @param pReqFFSFile (@c [in] MSG_DATA_REQ_FFS_FILE *) - File parameter
///
/// This is the handler for the message with the id MSG_REQ_FFS_FILE.
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgReqFFSFile (MSG_DATA_REQ_FFS_FILE * pReqFFSFile)
{
   UDWORD            CurAddress;
   UWORD             iCurDirEntry;
   DIRECTORY_ENTRY * pDirectoryEntry;

   if (pReqFFSFile == NULL || pReqFFSFile->FileNumber > m_NumOfFilesInFFS)
      return;
	
   pDirectoryEntry = (DIRECTORY_ENTRY *) AllocateMemory(sizeof(DIRECTORY_ENTRY));
	
   if (pDirectoryEntry)
   {
      CurAddress = STARTADDRESS_FFS_DIRECTORY;
      
      iCurDirEntry = 1;
      
      do 
      {
	        if (SST25PFX_ReadBlock(CurAddress, (PUBYTE)pDirectoryEntry, sizeof(DIRECTORY_ENTRY)) == sizeof(DIRECTORY_ENTRY))
	        {
	           if (iCurDirEntry == pReqFFSFile->FileNumber)
	           {
                 m_FileTransferMaxBlockSize = pReqFFSFile->MaxBlockSize;

	              m_pFileDataForTransfer =  (PUBYTE) AllocateMemory(m_FileTransferMaxBlockSize * sizeof (UBYTE));
                 
                 if (m_pFileDataForTransfer != NULL)
                 {
	                 m_FileTransferStartAddress = pDirectoryEntry->FileStartAddress;
	                 m_FileTransferActAddress   = pDirectoryEntry->FileStartAddress;
	                 m_FileTransferEndAddress   = pDirectoryEntry->FileEndAddress+1;

                    FlashMemM_StartFileTransfer();
                 }	    
                 else
                 {
                    // error
                 }
	              
	              break;
	           }
	           
	           iCurDirEntry++;
	         }
	         else
	         {
	            m_FileSystemIsCorrupt = 1;
	            break;
	         }

	         CurAddress += sizeof(DIRECTORY_ENTRY);
	      } while (1);

	      // free the allocated memory
	      FreeMemory (pDirectoryEntry);
	   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_StartFileTransfer
///
/// @brief Starts a file transfer
///
/// This function starts a file transfer.
///
/// @see FlashMemM_MsgReqFFSFile()
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_StartFileTransfer (void)
{
   MESSAGE Msg;
   UWORD NumOfBlocks;

   Msg.MessageID    = MSG_START_FFS_FILE_TRANSFER;
   Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_START_FFS_FILE_TRANSFER));

   if (Msg.pMessageData != NULL)
   {
      if (m_FileTransferStartAddress != m_FileTransferEndAddress)
      {
         NumOfBlocks = ((m_FileTransferEndAddress - m_FileTransferStartAddress) / m_FileTransferMaxBlockSize) + 1;
      }
      else
      {
         NumOfBlocks = 0;
      }

      ((MSG_DATA_START_FFS_FILE_TRANSFER *) Msg.pMessageData)->TotalNumOfDataBlocks = NumOfBlocks;
      ((MSG_DATA_START_FFS_FILE_TRANSFER *) Msg.pMessageData)->NumOfDataInFile = m_FileTransferEndAddress - m_FileTransferStartAddress;

      (void)OS_SendMessage (TASK_ID_PROCESS_M, &Msg, WAIT_MAX);
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_StartTransferNextBlock 
///
/// @brief Starts the transfer of the next file data block
///
/// This functions starts the transfer of the next file data block.
///
/// @see FlashMemM_StartFileTransfer()
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_StartTransferNextBlock ()
{
   UWORD   ActualBlockSize;
   MESSAGE Msg;
   
   if ((m_FileTransferEndAddress - m_FileTransferActAddress) >= m_FileTransferMaxBlockSize)
   {
      ActualBlockSize = m_FileTransferMaxBlockSize;
   }
   else
   {
      ActualBlockSize = m_FileTransferEndAddress - m_FileTransferActAddress;
   }

   if (ActualBlockSize == 0)
   {
      // free the memory
      FreeMemory (m_pFileDataForTransfer);
      
      // end of file
      Msg.MessageID    = MSG_FFS_FILE_EOF;
      Msg.pMessageData = NULL;

      (void)OS_SendMessage (TASK_ID_PROCESS_M, &Msg, WAIT_MAX);

      return;
   }

   if (SST25PFX_ReadBlock (m_FileTransferActAddress, m_pFileDataForTransfer, ActualBlockSize) == ActualBlockSize)
   {
      Msg.MessageID    = MSG_FFS_FILE_BLOCK;
      Msg.pMessageData = (PUBYTE) AllocateMemory(sizeof(MSG_DATA_FFS_FILE_BLOCK));

      if (Msg.pMessageData != NULL)
      {
         ((MSG_DATA_FFS_FILE_BLOCK *) Msg.pMessageData)->MemoryHandle       = m_pFileDataForTransfer;
         ((MSG_DATA_FFS_FILE_BLOCK *) Msg.pMessageData)->SizeOfFFSFileBlock = ActualBlockSize;

         (void)OS_SendMessage (TASK_ID_PROCESS_M, &Msg, WAIT_MAX);
      }
      
      m_FileTransferActAddress += ActualBlockSize;
   }
   else
   {
      // error
   }
}

//-----------------------------------------------------------------------------
//  Function FlashMemM_MsgGetNextFFSFileBlock
///
/// @brief Handler for the message with the id MSG_GET_NEXT_FFS_FILE_BLOCK
///
/// This is the handler for the message with the id MSG_GET_NEXT_FFS_FILE_BLOCK.
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FlashMemM_MsgGetNextFFSFileBlock()
{
   FlashMemM_StartTransferNextBlock();
}
