//-----------------------------------------------------------------------------
//  Tools.c
/// @file Tools.c
///
/// @brief Implementations of the toolfunctions for the application
///
/// All toolfunctions of the project are implementet in this module.
///
/// @copyright 2012-2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-14 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------
#include "stdlib.h"

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "Basetyp.h"
#include "Tools.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------
/// Allocate the memory
extern void *pvPortMalloc( size_t xWantedSize );
/// Free the memory
extern void vPortFree (void *pv);

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------
// Get the CRC8 value
UBYTE GetCRC8 (UBYTE CRCValue, UBYTE Data);

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function AllocateMemory
///
/// @brief Allocate memory from the head/os
///
/// @param MemorySize (@c [in] UWORD) - Size of memory for allocation in bytes
///
/// @return void* pointer to the allocated memory block
///
/// This function allocate memory from the head/os.
///
/// @see FreeMemory()
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void * AllocateMemory (UWORD MemorySize)
{
   return pvPortMalloc (MemorySize);
}

//-----------------------------------------------------------------------------
//  Function FreeMemory
///
/// @brief Free an allocated memory block
///
/// @param pMemory (@c [in] void *) - Handle/pointer to the memory block
///
/// This function free an allocated memory block.
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void FreeMemory (void * pMemory)
{
   vPortFree (pMemory);
}

//-----------------------------------------------------------------------------
//  Function CalculateCRC8 
///
/// @brief Calculate a CRC8 checksum on a data stream
///
/// @param pSource (@c [in] PUBYTE) - datastream
///
/// @param Offset (@c [in] int) - offset in byte
///
/// @param NumOfData (@c [in] int) - number of data in byte
///
/// @param CRCValue (@c [in] UBYTE) - CRC start value
///
/// @return UBYTE CRC8 value
///
/// This function calculate a CRC8 checksum on a data stream.
///
/// @see GetCRC8()
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE CalculateCRC8 (PUBYTE pSource, int Offset, int NumOfData, UBYTE CRCValue)
{
   int iCurItem;

   for (iCurItem = Offset; iCurItem < (Offset + NumOfData); iCurItem++)
   {
      // calculate the CRC value for the actual item
      CRCValue = GetCRC8 (CRCValue, pSource[iCurItem]);
   }

   CRCValue = GetCRC8(CRCValue, (UBYTE) 0x00);

   return CRCValue;
}

//-----------------------------------------------------------------------------
//  Function GetCRC8 
///
/// @brief Calculate a CRC8 value at a single byte
///
/// @param CRCValue (@c [in] UBYTE) - CRC start value
///
/// @param Data (@c [in] UBYTE) - data byte for calculation
///
/// @return UBYTE New CRC8 value
///
/// This function calculate a CRC8 value at a single byte.
///
/// @see CalculateCRC8()
///
/// @status  Draft
/// @date    2013-07-05
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-05 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE GetCRC8 (UBYTE CRCValue, UBYTE Data)
{
   int iCurBit;

   for (iCurBit = 0; iCurBit < 8; iCurBit++)
   {
      if ((CRCValue & 0x80) == 0x80)
      {
         CRCValue <<= 1;

         if ((Data & 0x80) == 0x80)
         {
            CRCValue = (UBYTE)(CRCValue | 0x01);
         }
         else
         {
            CRCValue = (UBYTE)(CRCValue & 0xfe);
         }

         // Polynom C(x) = x^8 + x^2 + x +1
         CRCValue = (UBYTE)(CRCValue ^ 0x07);
      }
      else
      {
         CRCValue <<= 1;

         if ((Data & 0x80) == 0x80)
         {
            CRCValue = (UBYTE)(CRCValue | 0x01);
         }
         else
         {
            CRCValue = (UBYTE)(CRCValue & 0xfe);
         }
      }

      Data <<= 1;
   }

   return (CRCValue);
}
