//-----------------------------------------------------------------------------
//  MMA8451Q.c
/// @file
///
/// @brief Driver for the MMA8451Q
///
/// The driver for the MMA8451Q is implemented in this module.
///
/// @copyright 2012-2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-22 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------
#include "Basetyp.h"

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "Events.h"
#include "MMA8451Q.h"
#include "MMA8451Q_Defs.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------
/// I2C address of the MMA8451Q
#define MMA8451Q_I2C_ADDRESS           0x1C
/// Alternative i2c address of the MMA8451Q
#define MMA8451Q_I2C_ADDRESS_ALT       0x1D
/// Maximum number of input data
#define MMA845X_MAX_NUM_OF_IN_DATA     0x06
/// Maximum number of output data
#define MMA845X_MAX_NUM_OF_OUT_DATA    0x02
/// Divider for the acceleration value
#define MMA845X_ACC_DIVIDER            0x04

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------
/// Flag: Data is transmitted via I2C1 interface
extern volatile bool m_I2C1_DataTransmitted;
/// Flag: Data is received via I2C1 interface
extern volatile bool m_I2C1_DataReceived;
/// Flag: An error is occur on the I2C1 interface
extern volatile bool m_I2C1_Error;

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------
/// I2C address of the device 
UBYTE m_MMA845x_I2C_Address;
/// Device id of the detected mma845x device
UBYTE m_MMA845x_DeviceID = 0x00;
/// I2C output data buffer
UBYTE m_MMA845x_OutData[MMA845X_MAX_NUM_OF_OUT_DATA] = {0x2A, 0x01};
/// I2C input data buffer
UBYTE m_MMA845x_InData[MMA845X_MAX_NUM_OF_IN_DATA];
/// Handle to the I2C interface
LDD_TDeviceData * m_pMMA845x_I2C_Handle;

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function MMA845X_Init 
///
/// @brief Inits the driver for the MMA8451Q device
///
/// @param pI2C_Handle (@c [in] LDD_TDeviceData *) - I2C handle
///
/// This function inits the driver for the MMA8451Q device.
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_Init (LDD_TDeviceData * pI2C_Handle)
{   
   m_pMMA845x_I2C_Handle = pI2C_Handle;

   // set the i2c address for the device

   m_MMA845x_I2C_Address = MMA8451Q_I2C_ADDRESS_ALT;

   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   while(I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 2, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (m_I2C1_DataTransmitted == FALSE && m_I2C1_Error == FALSE); 
   m_I2C1_DataTransmitted = FALSE;
   m_I2C1_Error = FALSE;

   m_MMA845x_OutData[0] = MMA845X_REG_WHO_AM_I; 

   while(I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (m_I2C1_DataTransmitted == FALSE && m_I2C1_Error == FALSE);  
   m_I2C1_DataTransmitted = FALSE;
   m_I2C1_Error = FALSE;
   
   while(I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_DeviceID, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;
}

//-----------------------------------------------------------------------------
//  Function MMA845X_GetDeviceID 
///
/// @brief Gets the device id
///
/// @return UBYTE Device id of the MMA8481Q device
///
/// This function gets the device id of the MMA8451Q.
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE MMA845X_GetDeviceID (void)
{
   return m_MMA845x_DeviceID;
}

//-----------------------------------------------------------------------------
//  Function MMA845X_ReadAccelerationValues 
///
/// @brief Reads the actual acceleration values
///
/// @param pX_Value (@c [out] PREAL) - Acceleration in x direction
///
/// @param pY_Value (@c [out] PREAL) - Acceleration in y direction
///
/// @param pZ_Value (@c [out] PREAL) - Acceleration in z direction
///
/// @param CountsPerG (@c [in] UWORD) - Number of digits per G
///
/// This function reads the actual acceleration values.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_ReadAccelerationValues (PREAL pX_Value, PREAL pY_Value, PREAL pZ_Value, UWORD CountsPerG)
{
   SWORD Tmp;

   if (pX_Value == NULL || pY_Value == NULL || pZ_Value == NULL)
      return;

   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_REG_DATA_REGISTERS;

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted); 
   m_I2C1_DataTransmitted = FALSE;   

   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_InData, 6, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;  

   Tmp = m_MMA845x_InData[1] | (m_MMA845x_InData[0] << 8);
   *pX_Value = ((REAL)Tmp / MMA845X_ACC_DIVIDER/CountsPerG);

   Tmp = m_MMA845x_InData[3] | (m_MMA845x_InData[2] << 8);
   *pY_Value = ((REAL)Tmp / MMA845X_ACC_DIVIDER/CountsPerG);

   Tmp = m_MMA845x_InData[5] | (m_MMA845x_InData[4] << 8);     
   *pZ_Value = ((REAL)Tmp / MMA845X_ACC_DIVIDER/CountsPerG);   
}

//-----------------------------------------------------------------------------
//  Function MMA845X_SetFullScaleValue 
///
/// @brief Sets the full scale value for the MMA8451Q device
///
/// @param FullScaleValue (@c [in] UBYTE) - New full scale value (2, 4 or 8 G)
///
/// This method sets the full scale value for the MMA8451Q device.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_SetFullScaleValue (UBYTE FullScaleValue)
{
   UBYTE CfgRegValue;
   
   if (FullScaleValue != 2 && FullScaleValue != 4 && FullScaleValue != 8)
      return;
   
   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_REG_DATA_CONFIG; 

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (m_I2C1_DataTransmitted == FALSE);  
   m_I2C1_DataTransmitted = FALSE;
   
   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &CfgRegValue, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   CfgRegValue &= 0xFC;
   
   switch (FullScaleValue)
   {
      case 2:  
      default:    CfgRegValue |= DATA_CFG_FULL_SCALE_VAL_2G;   break;
      case 4:     CfgRegValue |= DATA_CFG_FULL_SCALE_VAL_4G;   break;
      case 8:     CfgRegValue |= DATA_CFG_FULL_SCALE_VAL_8G;   break;
   }
   
   m_MMA845x_OutData[1] = CfgRegValue;

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 2, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted); 
   m_I2C1_DataTransmitted = FALSE;      
}

//-----------------------------------------------------------------------------
//  Function MMA845X_SetOutputDataRate 
///
/// @brief Sets the output data rate
///
/// @param OutputDataRate (@c [in] UBYTE) - New output data rate 1,563 hz to 800 hz
///
/// This method sets the output data rate.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_SetOutputDataRate (UBYTE OutputDataRate)
{
   UBYTE CfgRegValue;

   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG1; 

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (m_I2C1_DataTransmitted == FALSE);  
   m_I2C1_DataTransmitted = FALSE;

   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &CfgRegValue, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   CfgRegValue &= ~MMA845X_DR_BITS;

   switch (OutputDataRate)
   {
      case MMA845X_ODR_800HZ:
      default:                   CfgRegValue |= MMA845X_DR_BITS_800_HZ;   break;
      case MMA845X_ODR_400HZ:    CfgRegValue |= MMA845X_DR_BITS_400_HZ;   break;
      case MMA845X_ODR_200HZ:    CfgRegValue |= MMA845X_DR_BITS_200_HZ;   break;
      case MMA845X_ODR_100HZ:    CfgRegValue |= MMA845X_DR_BITS_100_HZ;   break;
      case MMA845X_ODR_50HZ:     CfgRegValue |= MMA845X_DR_BITS_50_HZ;    break;
      case MMA845X_ODR_12_5HZ:   CfgRegValue |= MMA845X_DR_BITS_12_5_HZ;  break;
      case MMA845X_ODR_6_25HZ:   CfgRegValue |= MMA845X_DR_BITS_6_25_HZ;  break;
      case MMA845X_ODR_1_563HZ:  CfgRegValue |= MMA845X_DR_BITS_1_563_HZ; break;
   }

   m_MMA845x_OutData[1] = CfgRegValue;

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 2, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted); 
   m_I2C1_DataTransmitted = FALSE;      
}

//-----------------------------------------------------------------------------
//  Function MMA845X_SetOversamplingMode 
///
/// @brief Sets the oversampling mode
///
/// @param OversamplingMode (@c [in] UBYTE) - New oversampling mode
///
/// This function sets the oversampling mode.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_SetOversamplingMode (UBYTE OversamplingMode)
{
   UBYTE CfgRegValue;

   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG2; 

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (m_I2C1_DataTransmitted == FALSE);  
   m_I2C1_DataTransmitted = FALSE;

   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &CfgRegValue, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   CfgRegValue &= ~MMA845X_MODS_BITS;

   switch (OversamplingMode)
   {
      case MMA845X_OSM_NORMAL:
      default:                      CfgRegValue |= MMA845X_MODS_BITS_NORMAL;      break;
      case MMA845X_OSM_LOW_NOISE:   CfgRegValue |= MMA845X_MODS_BITS_LOW_NOISE;   break;
      case MMA845X_OSM_HIGH_RES:    CfgRegValue |= MMA845X_MODS_BITS_HIGH_RES;    break;
      case MMA845X_OSM_LOW_POWER:   CfgRegValue |= MMA845X_MODS_BITS_LOW_POWER;   break;
   }

   m_MMA845x_OutData[1] = CfgRegValue;

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 2, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted); 
   m_I2C1_DataTransmitted = FALSE;      
}

//-----------------------------------------------------------------------------
//  Function MMA845X_Standby 
///
/// @brief Sets the device in standby mode
///
/// This function sets the device in standby mode.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_Standby (void)
{
   UBYTE CtrlReg1;
   
   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
   
   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;
   
   // Active-Flag zurcknehmen
   CtrlReg1 &= MMA845X_STANDBY_SBYB_MASK;
   
   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG1;
   m_MMA845x_OutData[1] = CtrlReg1;
   
   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
}

//-----------------------------------------------------------------------------
//  Function MMA845X_Active 
///
/// @brief Sets the device in active mode
///
/// This function sets the device in active mode.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_Active (void)
{
   UBYTE CtrlReg1;
   
   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;
   
   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;
   
   // Active-Flag setzen
   CtrlReg1 |= MMA845X_ACTIVE_SBYB_MASK;
   
   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG1;
   m_MMA845x_OutData[1] = CtrlReg1;
   
   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE; 
}

//-----------------------------------------------------------------------------
//  Function MMA845X_ClearFRead 
///
/// @brief Clears the fread bit of the device
///
/// This function clears the fread bit of the device.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_ClearFRead (void)
{
   UBYTE CtrlReg1;

   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG1; 

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 auslesen                                              */
   /*-------------------------------------------------------------------------*/
   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE;

   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &CtrlReg1, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   // Active-Flag setzen
   CtrlReg1 &= MMA845X_FREAD_CLEAR_MASK;
   
   m_MMA845x_OutData[0] = MMA845X_REG_CTRL_REG1;
   m_MMA845x_OutData[1] = CtrlReg1;

   /*-------------------------------------------------------------------------*/
   /* Controlregister 1 aktualisieren                                         */
   /*-------------------------------------------------------------------------*/
   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 2, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataTransmitted);  
   m_I2C1_DataTransmitted = FALSE; 
}

//-----------------------------------------------------------------------------
//  Function MMA845X_WaitUntilMeasurementComplete 
///
/// @brief Wait until the measurement is complete
///
/// This function wait until the measurement is complete.
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MMA845X_WaitUntilMeasurementComplete (void)
{
   UBYTE Status;

   do 
   {
      Status = MMA845X_ReadStatus();
   } while ((Status & MMA845X_STATUS_REG_ZYXDR) == 0);
}

//-----------------------------------------------------------------------------
//  Function MMA845X_IsMeasurementComplete 
///
/// @brief Check if a measurement is complete
///
/// @return bool TRUE - Measurement complete, FALSE - Measuring in process
///
/// This function checks if a measurement is complete.
///
/// @status  Draft
/// @date    2013-08-17
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-08-17 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
bool MMA845X_IsMeasurementComplete (void)
{
   UBYTE Status;

   Status = MMA845X_ReadStatus();
   
   if ((Status & MMA845X_STATUS_REG_ZYXDR) == 0)
   {
      return FALSE;
   }
   else
   {
      return TRUE;
   }
}

//-----------------------------------------------------------------------------
//  Function MMA845X_ReadStatus 
///
/// @brief Reads the device status register
///
/// @return UBYTE Value of the device status register
///
/// This function reads the device status register.
///
/// @status  Draft
/// @date    2013-07-07
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-07 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE MMA845X_ReadStatus (void)
{
   UBYTE RetVal;

   I2C1_SelectSlaveDevice (m_pMMA845x_I2C_Handle, LDD_I2C_ADDRTYPE_7BITS, m_MMA845x_I2C_Address);

   m_MMA845x_OutData[0] = MMA845X_STATUS_REG; 

   while (I2C1_MasterSendBlock(m_pMMA845x_I2C_Handle, &m_MMA845x_OutData, 1, LDD_I2C_NO_SEND_STOP) != ERR_OK);
   while (m_I2C1_DataTransmitted == FALSE);  
   m_I2C1_DataTransmitted = FALSE;

   while (I2C1_MasterReceiveBlock(m_pMMA845x_I2C_Handle, &RetVal, 1, LDD_I2C_SEND_STOP) != ERR_OK);
   while (!m_I2C1_DataReceived);
   m_I2C1_DataReceived = FALSE;

   return RetVal;
}
