//-----------------------------------------------------------------------------
//  SST25PFx.c
/// @file
///
/// @brief Driverfunction for the SST25PFx flash memory
///
/// All driverfunction for the SST25PFx flash memory are implemented in this module.
///
/// @copyright 2012-2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-23 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------
#include "Basetyp.h"

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "PE_LDD.h"
#include "SST25PFx.h"
#include "SST25PFx_Defs.h"
#include "Tools.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------
// Size for the in and out buffer
/// Maximum size of the out buffer
#define SST25PFX_MAX_NUM_OF_OUT_DATA       132
/// Maximum size of the in buffer
#define SST25PFX_MAX_NUM_OF_IN_DATA        132

// Lenght for the flash memory commands
/// Length for the command 'Read ID'
#define SST25PFX_CMD_LENGTH_RDID             6
/// Length for the command 'Read Status Register'
#define SST25PFX_CMD_LENGTH_RDSR             2
/// Length for the command 'Read Status Register 1'
#define SST25PFX_CMD_LENGTH_RDSR1            2
/// Length for the command 'Chip Erase'
#define SST25PFX_CMD_LENGTH_CHIP_ERASE       1
/// Length for the command 'Read Block'
#define SST25PFX_CMD_LENGTH_READ             4
/// Length for the command 'Write Statusregister'
#define SST25PFX_CMD_LENGTH_WRSR             2
/// Length for the command 'Write Block'
#define SST25PFX_CMD_LENGTH_WRITE            4
/// Length for the command 'Read Security ID'
#define SST25PFX_CMD_LENGTH_READ_SECURITY_ID 3
/// Length for the command 'Write Enable'
#define SST25PFX_CMD_LENGTH_WREN             1
/// Length for the command 'Write Disable'
#define SST25PFX_CMD_LENGTH_WRDI             1
/// Length for the command 'Sector Erase'
#define SST25PFX_CMD_LENGTH_SECTOR_ERASE     4
/// Length for the command '32K Block Erase'
#define SST25PFX_CMD_LENGTH_32K_BLOCK_ERASE  4
/// Length for the command '64K Block Erase'
#define SST25PFX_CMD_LENGTH_64K_BLOCK_ERASE  4
/// Length for the command 'Byte Program'
#define SST25PFX_CMD_BYTE_PROGRAM_CMD_LENGTH 5

// Maximum data length the commands 'Write Block' and 'Read Block'
/// Maximum data length in byte for the command 'Read Block'
#define SST25PFX_MAX_DB_READ_ON_BLOCK      256
/// Maximum data length in byte for the command 'Write Block'
#define SST25PFX_MAX_DB_WRITE_ON_BLOCK     128

// Last address for the devices
/// Highest writable address for the SST25PF020B device
#define SST25PFX_LAST_ADDRESS_020B     0x3FFFF
/// Highest writable address for the SST25PF040B device
#define SST25PFX_LAST_ADDRESS_040B     0x7FFFF
/// Highest writable address for the SST25PF080B device
#define SST25PFX_LAST_ADDRESS_080B     0xFFFFF

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------
/// Flag: Data is received via SPI interface
extern volatile bool m_SM1_BlockReceived;
/// Flag: Data is transmitted via SPI interface
extern volatile bool m_SM1_BlockTransmitted;

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------
/// Handle to the SPI interface
LDD_TDeviceData * m_SPI_Handle = NULL;
/// Buffer for output data
UBYTE m_SST25PFX_OutData[SST25PFX_MAX_NUM_OF_OUT_DATA];
/// Buffer for input data
UBYTE m_SST25PFX_InData[SST25PFX_MAX_NUM_OF_IN_DATA];
/// Last writeable address (default last address is 0x3FFFF for a 2 mbit device)
UDWORD m_LastWritableAddress = 0x3FFFF;

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------

// Read the status register
UBYTE SST25PFX_ReadStatusReg();
// Read the status register 1
UBYTE SST25PFX_ReadStatusReg1();
// Enable the flash memory for write
void SST25PFX_WriteEnable();
// Disable the flash memory for write
void SST25PFX_WriteDisable();
// Write status register
void SST25PFX_WriteStatusReg (UBYTE NewState);

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function SST25PFX_Init
///
/// @brief Init function for the flash memory SST25PFx
///
/// @param SPI_Handle (@c [in] LDD_TDeviceData *) - Handle to the SPI interface
///
/// This is the init function for the flash memory SST25PFx.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_Init (LDD_TDeviceData * SPI_Handle)
{
   m_SPI_Handle = SPI_Handle;
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_ReadID
///
/// @brief Read the id of the flash memory
///
/// @param pManID (@c [out] PUBYTE) - Manufacture id
///
/// @param pSizeID (@c [out] PUBYTE) - Size of the flash memory
///
/// This function reads the id of the flash memory.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_ReadID (PUBYTE pManID, PUBYTE pSizeID)
{
   if(pManID == NULL || pSizeID == NULL || m_SPI_Handle == NULL)
      return;

   // Kommando zur Abfrage der ID zusammenbauen
   m_SST25PFX_OutData[0] = SST25PFX_REG_RDID;
   m_SST25PFX_OutData[1] = 0x00;
   m_SST25PFX_OutData[2] = 0x00;
   m_SST25PFX_OutData[3] = 0x00;
   m_SST25PFX_OutData[4] = 0x00;
   m_SST25PFX_OutData[5] = 0x00;

   // ID abfragen
   SM1_ReceiveBlock( m_SPI_Handle, m_SST25PFX_InData, SST25PFX_CMD_LENGTH_RDID ); 
   SM1_SendBlock(m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_RDID);

   *pManID = m_SST25PFX_InData[4];
   *pSizeID = m_SST25PFX_InData[5];

   // set the highest address for the memory device
   switch (*pSizeID)
   {
      case SST25PFX_DEVICE_ID_020B: m_LastWritableAddress = SST25PFX_LAST_ADDRESS_020B;   break;
      case SST25PFX_DEVICE_ID_040B: m_LastWritableAddress = SST25PFX_LAST_ADDRESS_040B;   break;
      case SST25PFX_DEVICE_ID_080B: m_LastWritableAddress = SST25PFX_LAST_ADDRESS_080B;   break;
      default:                                                                            break;
   }

   return;
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_ReadStatusReg
///
/// @brief Read the status register
///
/// @return UBYTE Value of the status register byte
///
/// This function reads the status register.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE SST25PFX_ReadStatusReg()
{
   // Kommando zur Abfrage des Statusregisters zusammenbauen
   m_SST25PFX_OutData[0] = SST25PFX_REG_RDSR;
   m_SST25PFX_OutData[1] = 0x00;

   // Statusregister abfragen
   SM1_ReceiveBlock( m_SPI_Handle, m_SST25PFX_InData, SST25PFX_CMD_LENGTH_RDSR ); 
   SM1_SendBlock( m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_RDSR );

   return m_SST25PFX_InData[1];
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_ReadStatusReg1
///
/// @brief Read the status register 1
///
/// @return UBYTE Value of the status register 1 byte
///
/// This function reads the status register 1.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE SST25PFX_ReadStatusReg1()
{
   // Kommando zur Abfrage des Statusregisters zusammenbauen
   m_SST25PFX_OutData[0] = SST25PFX_REG_RDSR1;
   m_SST25PFX_OutData[1] = 0x00;

   // Statusregister abfragen
   SM1_ReceiveBlock( m_SPI_Handle, m_SST25PFX_InData, SST25PFX_CMD_LENGTH_RDSR1 ); 
   SM1_SendBlock( m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_RDSR1 );

   return m_SST25PFX_InData[1];
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_ReadSecurityID
///
/// @brief Read the security id
///
/// @param pSecurityID (@c [out] PUBYTE) - Security id of the flash memory
///
/// @return UBYTE Number of bytes readed
///
/// This function read the security id.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UBYTE SST25PFX_ReadSecurityID( PUBYTE pSecurityID )
{
   UBYTE iCurDatabyte;
   UWORD NumOfDataReaded;
   
   if (m_SPI_Handle == NULL)
   {
      return 0;
   }

#ifdef FLASH_DEVICE_SST25PFX
   if( pSecurityID != NULL )
   {
      // Kommando zusammenbauen ...
      m_SST25PFX_OutData[0] = SST25PFX_REG_READ_SECURITY_ID;
      m_SST25PFX_OutData[1] = 0;
      m_SST25PFX_OutData[2] = 0;

      // ... und ausgeben
      SM1_ReceiveBlock( m_SPI_Handle, m_SST25PFX_InData, SST25PFX_CMD_LENGTH_READ_SECURITY_ID + SST25PFX_SECURITY_ID_LENGTH ); 
      SM1_SendBlock( m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_READ_SECURITY_ID + SST25PFX_SECURITY_ID_LENGTH );

      NumOfDataReaded = SM1_GetReceivedDataNum(m_SPI_Handle);

      if( NumOfDataReaded == SST25PFX_SECURITY_ID_LENGTH + SST25PFX_SECURITY_ID_LENGTH )
      {
         memcpy(pSecurityID, &m_SST25PFX_InData[SST25PFX_CMD_LENGTH_READ_SECURITY_ID], SST25PFX_SECURITY_ID_LENGTH);

         NumOfDataReaded = SST25PFX_SECURITY_ID_LENGTH;
      }
      else
      {
         NumOfDataReaded = 0;
      }
   }    
#else
   // Function wird von diesem Flash nicht untersttzt, 0 zurckliefern
   NumOfDataReaded = SST25PFX_SECURITY_ID_LENGTH;

   for( iCurDatabyte = 0; iCurDatabyte < SST25PFX_SECURITY_ID_LENGTH; iCurDatabyte++ )
   {
      pSecurityID[iCurDatabyte] = 0x00;
   }
#endif

   return NumOfDataReaded;
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_ChipErase
///
/// @brief Complete chip erase
///
/// This function erase the chip completely.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_ChipErase()
{
   UBYTE StatusReg;

   if (m_SPI_Handle == NULL)
      return;
            
   // Schreiben freischalten
   SST25PFX_WriteEnable();

   // Kommando zusammenbauen ...
   m_SST25PFX_OutData[0] = SST25PFX_REG_CHIP_ERASE;

   // ... und ausgeben
   SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_CHIP_ERASE);

   // Warten bis das Lschen des Chip fertig ist
   do 
   {
      StatusReg = SST25PFX_ReadStatusReg();
   } while ( StatusReg & SR_BUSY );

   SST25PFX_WriteDisable();

   return;
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_64KBlockErase
///
/// @brief 64k block erase
///
/// @param SectorAddress (@c [in] UDWORD) - Address of the block for doing the erase
///
/// This function erases a 64k memory block.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_64KBlockErase( UDWORD SectorAddress )
{
   UBYTE StatusReg;

   if (m_SPI_Handle == NULL)
      return;
   
   // Schreiben freischalten
   SST25PFX_WriteEnable();

   // Kommando zusammenbauen ...
   m_SST25PFX_OutData[0] = SST25PFX_REG_64K_BLOCK_ERASE;
   m_SST25PFX_OutData[1] = (SectorAddress >> 16) & 0xFF;
   m_SST25PFX_OutData[2] = (SectorAddress >> 8) & 0xFF;
   m_SST25PFX_OutData[3] = SectorAddress & 0xFF;

   // ... und ausgeben
   SM1_SendBlock( m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_64K_BLOCK_ERASE );

   // Warten bis das Lschen des Chip fertig ist
   do 
   {
      StatusReg = SST25PFX_ReadStatusReg();
   } while ( StatusReg & SR_BUSY );

   SST25PFX_WriteDisable();
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_32KBlockErase
///
/// @brief 32k block erase
///
/// @param SectorAddress (@c [in] UDWORD) - Address of the block for doing the erase
///
/// This function erases a 32k memory block.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_32KBlockErase( UDWORD SectorAddress )
{
   UBYTE StatusReg;

   if (m_SPI_Handle == NULL)
      return;
   
   // Schreiben freischalten
   SST25PFX_WriteEnable();

   // Kommando zusammenbauen ...
   m_SST25PFX_OutData[0] = SST25PFX_REG_32K_BLOCK_ERASE;
   m_SST25PFX_OutData[1] = (SectorAddress >> 16) & 0xFF;
   m_SST25PFX_OutData[2] = (SectorAddress >> 8) & 0xFF;
   m_SST25PFX_OutData[3] = SectorAddress & 0xFF;

   // ... und ausgeben
   SM1_SendBlock( m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_32K_BLOCK_ERASE );

   // Warten bis das Lschen des Chip fertig ist
   do 
   {
      StatusReg = SST25PFX_ReadStatusReg();
   } while ( StatusReg & SR_BUSY );

   SST25PFX_WriteDisable();
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_SectorErase
///
/// @brief Sector erase
///
/// @param SectorAddress (@c [in] UDWORD) - Address of the sector for doing the erase
///
/// This function erases complete sector.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_SectorErase( UDWORD SectorAddress )
{
   UBYTE StatusReg;

   if (m_SPI_Handle == NULL)
      return;
   
   // Schreiben freischalten
   SST25PFX_WriteEnable();
   
   // Kommando zusammenbauen ...
   m_SST25PFX_OutData[0] = SST25PFX_REG_SECTOR_ERASE;
   m_SST25PFX_OutData[1] = (SectorAddress >> 16) & 0xFF;
   m_SST25PFX_OutData[2] = (SectorAddress >> 8) & 0xFF;
   m_SST25PFX_OutData[3] = SectorAddress & 0xFF;

   // ... und ausgeben
   SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_SECTOR_ERASE);

   // Warten bis das Lschen des Chip fertig ist
   do 
   {
      StatusReg = SST25PFX_ReadStatusReg();
   } while (StatusReg & SR_BUSY);

   StatusReg = SST25PFX_ReadStatusReg();
   
   if (StatusReg & SR_WEL)
   {
      SST25PFX_WriteDisable();
   }
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_WriteEnable
///
/// @brief Enable the write functions for the flash memory
///
/// This function enables the write functions for the flash memory.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_WriteEnable (void)
{
   m_SST25PFX_OutData[0] = SST25PFX_REG_WREN;

   SM1_SendBlock( m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_WREN );  
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_WriteDisable
///
/// @brief Disable the write functions for the flash memory
///
/// This function disable the write functions for the flash memory.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_WriteDisable (void)
{
   m_SST25PFX_OutData[0] = SST25PFX_REG_WRDI;

   SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_WRDI);
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_DisableBlockProtection
///
/// @brief Disable the memory block protection
///
/// This function disables the memory block protection.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_DisableBlockProtection (void)
{
   UBYTE StatusReg;

   StatusReg = SST25PFX_ReadStatusReg();

   StatusReg &= ~SR_BP0;
   StatusReg &= ~SR_BP1;
   StatusReg &= ~SR_BP2;
   
   SST25PFX_WriteEnable();
   SST25PFX_WriteStatusReg (StatusReg);
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_WriteStatusReg
///
/// @brief Writes the status register
///
/// @param NewState (@c [in] UBYTE) - New value for the status register
///
/// This function writes the status register.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_WriteStatusReg (UBYTE NewState)
{
   m_SST25PFX_OutData[0] = SST25PFX_REG_WRSR;
   m_SST25PFX_OutData[1] = NewState;

   SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_WRSR);
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_WriteByte
///
/// @brief Write a single byte to the flash memory
///
/// @param TargetAddress (@c [in] UDWORD) - Address of the byte
///
/// @param DataByte (@c [in] UBYTE) - Value for the byte
///
/// This function writes a single byte to the flash memory.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void SST25PFX_WriteByte (UDWORD TargetAddress, UBYTE DataByte)
{
   UBYTE StatusReg;

   if (m_SPI_Handle == NULL)
      return;

   // Is the target address in a valid range for the detected device? (Call
   // SST25PFX_ReadID() to detect the device type / memory size )
   if (TargetAddress > m_LastWritableAddress)
      return 0;
   
   SST25PFX_WriteEnable();

   m_SST25PFX_OutData[0] = SST25PFX_REG_BYTE_PROGRAM;
   m_SST25PFX_OutData[1] = (TargetAddress >> 16) & 0xFF;
   m_SST25PFX_OutData[2] = (TargetAddress >> 8) & 0xFF;
   m_SST25PFX_OutData[3] = TargetAddress & 0xFF;
   m_SST25PFX_OutData[4] = DataByte;

   SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_BYTE_PROGRAM_CMD_LENGTH);

   // Warten bis das Schreiben auf den Chip fertig ist
   do 
   {
      StatusReg = SST25PFX_ReadStatusReg();
   } while (StatusReg & SR_BUSY);

   SST25PFX_WriteDisable();
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_WriteBlock
///
/// @brief Write a data block to the flash memory
///
/// @param TargetAddress (@c [in] UDWORD) - Address of the byte
///
/// @param pBuffer (@c [in] PUBYTE) - Buffer with the data to write
///
/// @param NumOfData (@c [in] UWORD) - Number of data to write
///
/// @return UWORD Number of written data
///
/// This function writes a data block to the flash memory.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UWORD SST25PFX_WriteBlock (UDWORD TargetAddress, PUBYTE pBuffer, UWORD NumOfData)
{
   UWORD NumOfDataWritten = 0;
   UBYTE StatusReg;
   UBYTE Offset = 0;
   
   if (pBuffer != NULL && NumOfData <= SST25PFX_MAX_DB_WRITE_ON_BLOCK && NumOfData >= 2 && m_SPI_Handle != NULL)
   {
      // Is the target address in a valid range for the detected device? (Call
      // SST25PFX_ReadID() to detect the device type / memory size )
      if ((TargetAddress + NumOfData - 1) > m_LastWritableAddress)
         return 0;

      SST25PFX_WriteEnable();

      m_SST25PFX_OutData[0] = SST25PFX_REG_AAI_WORD_PROGRAM;
      m_SST25PFX_OutData[1] = (TargetAddress >> 16) & 0xFF;
      m_SST25PFX_OutData[2] = (TargetAddress >> 8) & 0xFF;
      m_SST25PFX_OutData[3] = TargetAddress & 0xFF;
      m_SST25PFX_OutData[4] = pBuffer[Offset++];
      m_SST25PFX_OutData[5] = pBuffer[Offset++];

      SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, SST25PFX_CMD_LENGTH_WRITE + 2);

      do 
      {
         do 
         {
            StatusReg = SST25PFX_ReadStatusReg();
         } while (StatusReg & SR_BUSY);

         m_SST25PFX_OutData[0] = SST25PFX_REG_AAI_WORD_PROGRAM;
         m_SST25PFX_OutData[1] = pBuffer[Offset++];
         m_SST25PFX_OutData[2] = pBuffer[Offset++];

         SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, 3);
      } while (Offset < (NumOfData-1));

      // Warten bis das Schreiben auf den Chip fertig ist
      do 
      {
         StatusReg = SST25PFX_ReadStatusReg();
      } while (StatusReg & SR_BUSY);

      SST25PFX_WriteDisable();

      // Eventuell muss noch ein einzelnes Bytes geschrieben werden
      if (Offset < NumOfData)
      {
         SST25PFX_WriteByte (TargetAddress+Offset, pBuffer[Offset]);
      }

      NumOfDataWritten = NumOfData;
   }

   return NumOfDataWritten;
}

//-----------------------------------------------------------------------------
//  Function SST25PFX_ReadBlock
///
/// @brief Read a data block from the flash memory
///
/// @param TargetAddress (@c [in] UDWORD) - Address of the data block on the flash 
///
/// @param pBuffer (@c [in] PUBYTE) - Buffer for the readed data
///
/// @param NumOfData (@c [in] UWORD) - Number of data to read
///
/// @return UWORD Number of readed data
///
/// This function reads a data block from the flash memory.
///
/// @status  Draft
/// @date    2013-04-25
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-25 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
UWORD SST25PFX_ReadBlock (UDWORD TargetAddress, PUBYTE pBuffer, UWORD NumOfData)
{
   UWORD NumOfDataReaded = 0;
   LDD_SPIMASTER_TError SPIError;

   if (pBuffer != NULL && NumOfData <= SST25PFX_MAX_DB_READ_ON_BLOCK && m_SPI_Handle != NULL)
   {
      m_SST25PFX_OutData[0] = SST25PFX_REG_READ;
      m_SST25PFX_OutData[1] = (TargetAddress >> 16) & 0xFF;
      m_SST25PFX_OutData[2] = (TargetAddress >> 8) & 0xFF;
      m_SST25PFX_OutData[3] = TargetAddress & 0xFF;

      // Lesen von der Adresse initialisieren
      m_SM1_BlockReceived = FALSE;
      
      if (SM1_ReceiveBlock (m_SPI_Handle, m_SST25PFX_InData, (SST25PFX_CMD_LENGTH_READ + NumOfData)) != ERR_OK)
      {
         SM1_GetError (m_SPI_Handle, &SPIError);
         SM1_CancelBlockReception (m_SPI_Handle);

         if (SM1_ReceiveBlock (m_SPI_Handle, m_SST25PFX_InData, (SST25PFX_CMD_LENGTH_READ + NumOfData)) != ERR_OK)
         {
            SM1_GetError (m_SPI_Handle, &SPIError);
            return 0;            
         }
      }
      
      m_SM1_BlockTransmitted = FALSE;
      
      if (SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, (SST25PFX_CMD_LENGTH_READ + NumOfData)) != ERR_OK)
      {
         SM1_GetError (m_SPI_Handle, &SPIError);
         SM1_CancelBlockTransmission (m_SPI_Handle);
         
         if (SM1_SendBlock (m_SPI_Handle, m_SST25PFX_OutData, (SST25PFX_CMD_LENGTH_READ + NumOfData)) != ERR_OK)
         {
            SM1_GetError (m_SPI_Handle, &SPIError);
            return 0;
         }
      }
      
      while (m_SM1_BlockReceived == FALSE || m_SM1_BlockTransmitted == FALSE);
      
      NumOfDataReaded = SM1_GetReceivedDataNum(m_SPI_Handle);

      if ((NumOfDataReaded - SST25PFX_CMD_LENGTH_READ) <= NumOfData)
      {
         memcpy (pBuffer, &m_SST25PFX_InData[SST25PFX_CMD_LENGTH_READ], (NumOfDataReaded - SST25PFX_CMD_LENGTH_READ));

         NumOfDataReaded = (NumOfDataReaded - SST25PFX_CMD_LENGTH_READ);
      }
      else
      {
         SM1_GetError (m_SPI_Handle, &SPIError);
      }
   }

   return NumOfDataReaded;
}
