//-----------------------------------------------------------------------------
//  MeasureM.c
/// @file
///
/// @brief Implements all functions for capturing the data
///
/// All functions for caturing the data are implemented in this module.
///
/// @copyright 2012-2013 TABO - Embedded Systems GmbH & Co. KG - All Rights Reserved
///
/// @status  Draft
/// @date    $Date:$
/// @version $Revision:$
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-10 Eike Mueller(TABO)
///            Draft => V.1.0
/// @endhistory
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Standard-Include-Dateien
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Eigene Include-Dateien
//-----------------------------------------------------------------------------
#include "OS.h"

#include "PE_LDD.h"
#include "MeasureM.h"
#include "MPL3115A2.h"
#include "MMA8451Q.h"
#include "CpuRegHAL.h"
#include "Tools.h"

#include "ProcessM.h"

#include "MessageDefs.h"

//-----------------------------------------------------------------------------
// Private-Definitionen, -Makros und -Konstanten
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Private-Datentypen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Externe Referenzen
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Modulglobale Variable
//-----------------------------------------------------------------------------
/// Actual job to process by the task
MEASURE_M_JOB m_MeasureMJob;
/// Buffer for the actual measuring values
MSG_DATA_ACTUAL_MVALUES m_MeasureValues;
/// local copy of the measure mode
MEASURE_MODE m_MeasureM_MeasureMode;

//-----------------------------------------------------------------------------
// Prototypen der private-Functionen
//-----------------------------------------------------------------------------
// handler for the job MMJ_INIT_MEASURE_SYSTEM
void MeasureM_InitMeasureSystem( void );
// Handler for the job MMJ_DO_MEASURING
void MeasureM_DoMeasuring( void );
// Handler for the job MMJ_CONFIG_CHANGED
void MeasureM_ConfigChanged( void );
// Handler for the job MMJ_MEASURE_MODE_CHANGED
void MeasureM_MeasureModeChanged (void);
// Sets the oversampling ratio for the MPL3115A2 device
void MeasureM_SetOversamplingMPL3115A (void);
// Sets the oversampling ratio for the MMA845X device
void MeasureM_SetOversamplingMMA845X (void);

//-----------------------------------------------------------------------------
// Implementierung
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
//  Function MeasureM_SetJob
///
/// @brief Set a new job for the measuring task
///
/// @param MeasureMJob (@c [in] MEASURE_M_JOB) - New job for the task
///
/// This function sets a new job for the measuring task.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_SetJob( MEASURE_M_JOB MeasureMJob )
{      
   m_MeasureMJob = MeasureMJob;

   OS_SemaphoreGiveFromISR( TASK_ID_MEASURE_M );
}

//-----------------------------------------------------------------------------
//  Function MeasureM_TaskFunction
///
/// @brief Taskfunction of the task
///
/// @param pvParameters (@c [in] void *) - Task parameter (unused)
///
/// This is the taskfunction of the task. The function waits on a semaphore
/// until a new job is set. There's no queue in this task so the function can
/// do only one job at time.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_TaskFunction( void * pvParameters )
{
   while(1)
   {
      m_MeasureMJob = MMJ_NO_JOB;

      if( OS_SemaphoreTake( TASK_ID_MEASURE_M, WAIT_MAX ) == ERR_OS_OK )
      {
         switch( m_MeasureMJob )
         {
            case MMJ_INIT_MEASURE_SYSTEM:    MeasureM_InitMeasureSystem();    break;
            case MMJ_DO_MEASURING:           MeasureM_DoMeasuring();          break;
            case MMJ_CONFIG_CHANGED:         MeasureM_ConfigChanged();        break;
            case MMJ_MEASURE_MODE_CHANGED:   MeasureM_MeasureModeChanged();   break;
            case MMJ_NO_JOB:
            default:
               break;
         }
         
         m_MeasureMJob = MMJ_NO_JOB;
      }
   }
}

//-----------------------------------------------------------------------------
//  Function MeasureM_InitMeasureSystem
///
/// @brief Handler for the job MMJ_INIT_MEASURE_SYSTEM
///
/// This is the handler for the job MMJ_INIT_MEASURE_SYSTEM. The function
/// initialize all the measure values member, the I2C interface, the MPL3115A2
/// device and the MMA845x device.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_InitMeasureSystem( void )
{
   LDD_TDeviceData * pI2C_Handle;

   // init the timestamp
   m_MeasureValues.TimestampHi = 0;
   m_MeasureValues.TimestampLo = 0;

   // Wertespeicher initialisieren
   MeasureM_ClearMeasureValues();

   //-------------------------------------------------------------------------
   // I2C-Bus Initialisierung
   //-------------------------------------------------------------------------
   pI2C_Handle = I2C1_Init(NULL); 

   //-------------------------------------------------------------------------
   // Hhenmesser Initialisierung
   //-------------------------------------------------------------------------
   // Grundinitialisierung
   MPL3115A2_Init(pI2C_Handle); 
   // Oversampling
   MeasureM_SetOversamplingMPL3115A();
   // Gert in Standby schicken
   MPL3115A2_Standby();
   // Altimetermode einstellen
   MPL3115A2_SetAltimeterMode();
   // Messung initiieren (Baustein geht nach der Messung automatisch in Standby)
   MPL3115A2_InitiateMeasurement();
   
   //-------------------------------------------------------------------------
   // Beschleunigungssensor Initialisierung
   //-------------------------------------------------------------------------
   // init the device
   MMA845X_Init(pI2C_Handle);
   // set the device in standby mode
   MMA845X_Standby();
   // set the full scale value to 8G
   MMA845X_SetFullScaleValue (8);
   // set the oversampling mode to hi resolution
   MMA845X_SetOversamplingMode (MMA845X_OSM_NORMAL);
   // set the output data rate
   MeasureM_SetOversamplingMMA845X ();
   // clear the F-Read bit
   MMA845X_ClearFRead ();
   // set the device in active mode
   MMA845X_Active();
}

//-----------------------------------------------------------------------------
//  Function MeasureM_ResetMinMaxValues
///
/// @brief Resets the min- and max-values
///
/// This function resets the min- and the max-values. The function is thread safe.
///
/// @status  Draft
/// @date    2013-04-23
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-23 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_ClearMeasureValues( void )
{
   OS_MutexTake(TASK_ID_MEASURE_M, WAIT_MAX);
   
   m_MeasureValues.ActualAltitude     = 0.0f;
   m_MeasureValues.ActualPressure     = 0.0f;
   m_MeasureValues.ActualTemperature  = 0.0f;
   m_MeasureValues.ActualAccXDir      = 0.0f;
   m_MeasureValues.ActualAccYDir      = 0.0f;
   m_MeasureValues.ActualAccZDir      = 0.0f;
   m_MeasureValues.MinimalAltitude    = 0.0f;
   m_MeasureValues.MinimalPressure    = 0.0f;
   m_MeasureValues.MinimalTemperature = 0.0f;
   m_MeasureValues.MinimalAccXDir     = 0.0f;
   m_MeasureValues.MinimalAccYDir     = 0.0f;
   m_MeasureValues.MinimalAccZDir     = 0.0f;
   m_MeasureValues.MaximalAltitude    = 0.0f;
   m_MeasureValues.MaximalPressure    = 0.0f;
   m_MeasureValues.MaximalTemperature = 0.0f;
   m_MeasureValues.MaximalAccXDir     = 0.0f;
   m_MeasureValues.MaximalAccYDir     = 0.0f;
   m_MeasureValues.MaximalAccZDir     = 0.0f;
   
   OS_MutexGive(TASK_ID_MEASURE_M);
}

//-----------------------------------------------------------------------------
//  Function MeasureM_GetValue
///
/// @brief Gets a value from the min-/max-value structure
///
/// @param enValue (@c [in] VALUE) - Id of the value to get
///
/// @return float Value from the min-/max-value structure
///
/// This function gets a value from the min-/max-value structure.
///
/// @see VALUE
///
/// @status  Draft
/// @date    2013-04-23
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-04-23 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
float MeasureM_GetMeasureValue( VALUE enValue )
{
   OS_MutexTake(TASK_ID_MEASURE_M, WAIT_MAX);

   float Value = 0.0f;

   switch( enValue )
   {
      case VAL_ACTUAL_ALTITUDE:        Value = m_MeasureValues.ActualAltitude;      break;
      case VAL_ACTUAL_PRESSURE:        Value = m_MeasureValues.ActualPressure;      break;
      case VAL_ACTUAL_TEMPERATURE:     Value = m_MeasureValues.ActualTemperature;   break;
      case VAL_ACTUAL_ACC_X_POS:       Value = m_MeasureValues.ActualAccXDir;       break;
      case VAL_ACTUAL_ACC_Y_POS:       Value = m_MeasureValues.ActualAccYDir;       break;
      case VAL_ACTUAL_ACC_Z_POS:       Value = m_MeasureValues.ActualAccZDir;       break;
      case VAL_MINIMAL_ALTITUDE:       Value = m_MeasureValues.MinimalAltitude;     break;
      case VAL_MINIMAL_PRESSURE:       Value = m_MeasureValues.MinimalPressure;     break;
      case VAL_MINIMAL_TEMPERATURE:    Value = m_MeasureValues.MinimalTemperature;  break;
      case VAL_MINIMAL_ACC_X_POS:      Value = m_MeasureValues.MinimalAccXDir;      break;
      case VAL_MINIMAL_ACC_Y_POS:      Value = m_MeasureValues.MinimalAccYDir;      break;
      case VAL_MINIMAL_ACC_Z_POS:      Value = m_MeasureValues.MinimalAccZDir;      break;
      case VAL_MAXIMAL_ALTITUDE:       Value = m_MeasureValues.MaximalAltitude;     break;
      case VAL_MAXIMAL_PRESSURE:       Value = m_MeasureValues.MaximalPressure;     break;
      case VAL_MAXIMAL_TEMPERATURE:    Value = m_MeasureValues.MaximalTemperature;  break;
      case VAL_MAXIMAL_ACC_X_POS:      Value = m_MeasureValues.MaximalAccXDir;      break;
      case VAL_MAXIMAL_ACC_Y_POS:      Value = m_MeasureValues.MaximalAccYDir;      break;
      case VAL_MAXIMAL_ACC_Z_POS:      Value = m_MeasureValues.MaximalAccZDir;      break;
      default:                         Value = 0.0f;                                break;
   }

   OS_MutexGive(TASK_ID_MEASURE_M);
   
   return Value;
}

//-----------------------------------------------------------------------------
//  Function MeasureM_GetTimestamp
///
/// @brief Gets the timestamp of the last measuring
///
/// @param enValue (@c [in] VALUE) - Id of the timestamp to get (Lo or Hi Value
///                                  from the timestamp)
///
/// @return int Timestamp of the last measuring (Lo or Hi value)
///
/// This function gets the timestamp of the last measuring.
///
/// @status  Draft
/// @date    2013-06-21
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-21 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
int MeasureM_GetTimestamp( VALUE enValue )
{
   int Value = 0;

   OS_MutexTake(TASK_ID_MEASURE_M, WAIT_MAX);

   switch( enValue )
   {
      case VAL_TIMESTAMP_LO:
         Value = m_MeasureValues.TimestampLo;
         break;

      case VAL_TIMESTAMP_HI:
         OS_DisableInterrupts();
         Value = m_MeasureValues.TimestampHi;
         OS_EnableInterrupts();
         break;

      default:                
         Value = 0;
         break;
   }

   OS_MutexGive(TASK_ID_MEASURE_M);
   
   return Value;
}

//-----------------------------------------------------------------------------
//  Function MeasureM_IncrementTimestampHi
///
/// @brief Increment the Hi value of the timestamp
///
/// This function increment the Hi value of the timestamp.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_IncrementTimestampHi()
{
   m_MeasureValues.TimestampHi++;
}

//-----------------------------------------------------------------------------
//  Function MeasureM_DoMeasuring
///
/// @brief Handler for the job MMJ_DO_Measuring
///
/// All measuring function are doing in this function.
///
/// @status  Draft
/// @date    2013-02-28
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-02-28 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_DoMeasuring( void )
{
   MEASURE_MODE MMode;
   MESSAGE      Msg;
   
   float TimestampLo = 0;
      
   MMode = ProcessM_GetMeasureMode();
      
   if (MMode != m_MeasureM_MeasureMode)
   {
      MeasureM_MeasureModeChanged();
   }
   
   // get the altitude and temperature values
   if (MPL3115A2_IsMeasurementComplete() == TRUE)
   {
      MPL3115A2_ReqAltitude();

      OS_MutexTake (TASK_ID_MEASURE_M, WAIT_MAX);
      MPL3115A2_ReadAltitudeEx (&m_MeasureValues.ActualAltitude, &m_MeasureValues.ActualTemperature);
      OS_MutexGive (TASK_ID_MEASURE_M);

      // initiate a new measurement on the MPL3115A2
      MPL3115A2_InitiateMeasurement();
   }
   
   // get the acceleration values
   if (MMA845X_IsMeasurementComplete() == TRUE)
   {
      OS_MutexTake (TASK_ID_MEASURE_M, WAIT_MAX);
      MMA845X_ReadAccelerationValues (&m_MeasureValues.ActualAccXDir, &m_MeasureValues.ActualAccYDir, &m_MeasureValues.ActualAccZDir, 1024);
      OS_MutexGive (TASK_ID_MEASURE_M);

      // clear the F-Read Bit
      MMA845X_Standby ();
      MMA845X_ClearFRead ();
      MMA845X_Active ();
   }
   
   TimestampLo = CRHal_GetPITCounterVal();
   
   if (m_MeasureM_MeasureMode == MM_LOGGING)
   {
      Msg.MessageID = MSG_INSERT_DATASET_IN_FILE;
      Msg.pMessageData = (PUBYTE) AllocateMemory (sizeof(MSG_DATA_INSERT_DATASET));

      if (Msg.pMessageData)
      {   
         OS_DisableInterrupts();
         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->TimestampHi = m_MeasureValues.TimestampHi;
         OS_EnableInterrupts();

         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->TimestampLo       = TimestampLo;
         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->ActualAltitude    = m_MeasureValues.ActualAltitude;
         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->ActualPressure    = m_MeasureValues.ActualPressure;
         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->ActualTemperature = m_MeasureValues.ActualTemperature;
         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->ActualAccXDir     = m_MeasureValues.ActualAccXDir;
         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->ActualAccYDir     = m_MeasureValues.ActualAccYDir;
         ((MSG_DATA_INSERT_DATASET *) Msg.pMessageData)->ActualAccZDir     = m_MeasureValues.ActualAccZDir;
         
         // send the dataset directly to the flash manager (better performance)
         if( OS_SendMessage( TASK_ID_FLASHMEM_M, &Msg, WAIT_MAX ) == ERR_OS_QUEUE_FULL )
         {
            // Fehlerbehandlung
         }
      }     
   }
   
   OS_MutexTake(TASK_ID_MEASURE_M, WAIT_MAX);
   m_MeasureValues.TimestampLo = TimestampLo;
   
   // update the min-/max-values for the altitude
   if( m_MeasureValues.ActualAltitude > m_MeasureValues.MaximalAltitude || m_MeasureValues.MaximalAltitude == 0.0f )
   {
      m_MeasureValues.MaximalAltitude = m_MeasureValues.ActualAltitude;
   }

   if( m_MeasureValues.ActualAltitude < m_MeasureValues.MinimalAltitude || m_MeasureValues.MinimalAltitude == 0.0f )
   {
      m_MeasureValues.MinimalAltitude = m_MeasureValues.ActualAltitude;
   }

   // update the min-/max-values for the temperature
   if( m_MeasureValues.ActualTemperature > m_MeasureValues.MaximalTemperature || m_MeasureValues.MaximalTemperature == 0.0f)
   {
      m_MeasureValues.MaximalTemperature = m_MeasureValues.ActualTemperature;
   }

   if( m_MeasureValues.ActualTemperature < m_MeasureValues.MinimalTemperature || m_MeasureValues.MinimalTemperature == 0.0f)
   {
      m_MeasureValues.MinimalTemperature = m_MeasureValues.ActualTemperature;
   }   
   
   // update the min-/max-values for acceleration x-axes
   if (m_MeasureValues.ActualAccXDir > m_MeasureValues.MaximalAccXDir || m_MeasureValues.MaximalAccXDir == 0.0f)
   {
      m_MeasureValues.MaximalAccXDir = m_MeasureValues.ActualAccXDir;
   }

   if (m_MeasureValues.ActualAccXDir < m_MeasureValues.MinimalAccXDir || m_MeasureValues.MinimalAccXDir == 0.0f)
   {
      m_MeasureValues.MinimalAccXDir = m_MeasureValues.ActualAccXDir;
   }
   
   //update the min-/max-values for acceleration y-axes
   if (m_MeasureValues.ActualAccYDir > m_MeasureValues.MaximalAccYDir || m_MeasureValues.MaximalAccYDir == 0.0f)
   {
      m_MeasureValues.MaximalAccYDir = m_MeasureValues.ActualAccYDir;
   }

   if (m_MeasureValues.ActualAccYDir < m_MeasureValues.MinimalAccYDir || m_MeasureValues.MinimalAccYDir == 0.0f)
   {
      m_MeasureValues.MinimalAccYDir = m_MeasureValues.ActualAccYDir;
   }
   
   //update the min-/max-values for acceleration z-axes
   if (m_MeasureValues.ActualAccZDir > m_MeasureValues.MaximalAccZDir || m_MeasureValues.MaximalAccZDir == 0.0f)
   {
      m_MeasureValues.MaximalAccZDir = m_MeasureValues.ActualAccZDir;
   }

   if (m_MeasureValues.ActualAccZDir < m_MeasureValues.MinimalAccZDir || m_MeasureValues.MinimalAccZDir == 0.0f)
   {
      m_MeasureValues.MinimalAccZDir = m_MeasureValues.ActualAccZDir;
   }
   
   OS_MutexGive(TASK_ID_MEASURE_M);
}

//-----------------------------------------------------------------------------
//  Function MeasureM_ConfigChanged 
///
/// @brief Handler for the job MMJ_CONFIG_CHANGED
///
/// This is the handler for the job MMJ_CONFIG_CHANGED.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_ConfigChanged (void)
{
   // recalculate the oversampling for the MPL3115A2
   MeasureM_SetOversamplingMPL3115A();
   
   // recalculate the oversampling for the MMA8453Q
   MMA845X_Standby();
   MeasureM_SetOversamplingMMA845X();
   MMA845X_Active();
}

//-----------------------------------------------------------------------------
//  Function MeasureM_MeasureModeChanged 
///
/// @brief Handler for the job MMJ_MEASURE_MODE_CHANGED
///
/// This is the handler for the job MMJ_MEASURE_MODE_CHANGED.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_MeasureModeChanged (void)
{
   // get the new measure mode
   m_MeasureM_MeasureMode = ProcessM_GetMeasureMode();

   // recalculate the oversampling for the MPL3115A2
   MeasureM_SetOversamplingMPL3115A();
   // recalculate the oversampling for the MMA8453Q
   MMA845X_Standby();
   MeasureM_SetOversamplingMMA845X();
   MMA845X_Active();
}

//-----------------------------------------------------------------------------
//  Function MeasureM_SetOversamplingMPL3115A 
///
/// @brief Sets the oversampling ratio for the MPL3115A2 device
///
/// This function sets the oversampling ratio for the MPL3115A2 device.
///
/// @status  Draft
/// @date    2013-06-22
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-06-22 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_SetOversamplingMPL3115A (void)
{
   MSG_DATA_DEVICE_CONFIG DevConfig;
   UWORD                  MeasurePeriod;

   ProcessM_GetConfigData (&DevConfig);

   if (m_MeasureM_MeasureMode == MM_NORMAL)
   {
      MeasurePeriod = DevConfig.MeasurePeriodNormal;
   }
   else
   {
      MeasurePeriod = DevConfig.MeasurePeriodLogging;
   }

    // set the oversampling ratio
   if (MeasurePeriod > 514 )
   {
      // oversampling 7 -> oversampling ratio 128 -> 512 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(7);
   }
   else if (MeasurePeriod > 260)
   {
      // oversampling 6 -> oversampling ratio 64 -> 258 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(6);
   }
   else if (MeasurePeriod > 132)
   {
      // oversampling 5 -> oversampling ratio 32 -> 130 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(5);
   }
   else if (MeasurePeriod > 68)
   {
      // oversampling 4 -> oversampling ratio 16 -> 66 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(4);
   }
   else if (MeasurePeriod > 36)
   {
      // oversampling 3 -> oversampling ratio 8 -> 36 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(3);
   }
   else if (MeasurePeriod > 20)
   {
      // oversampling 2 -> oversampling ratio 4 -> 18 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(2);
   }
   else if (MeasurePeriod > 12)
   {
      // oversampling 1 -> oversampling ratio 2 -> 10 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(1);
   }
   else
   {
      // oversampling 0 -> oversampling ratio 1 -> 6 ms min Time for a data sample
      MPL3115A2_SetOversampleRatio(0);
   }
}

//-----------------------------------------------------------------------------
//  Function MeasureM_SetOversamplingMMA845X 
///
/// @brief Sets the oversampling ratio for the MMA845X device
///
/// This function sets the oversampling ratio for the MMA845X device.
///
/// @status  Draft
/// @date    2013-07-02
/// @author  Eike Mueller(TABO)
///
/// @history
/// Date/Ver   Author/Modification
///
/// 2013-07-02 Eike Mueller(TABO)
///            Draft
/// @endhistory
//-----------------------------------------------------------------------------
void MeasureM_SetOversamplingMMA845X (void)
{
   MSG_DATA_DEVICE_CONFIG DevConfig;
   UWORD                  MeasurePeriod;

   ProcessM_GetConfigData (&DevConfig);

   if (m_MeasureM_MeasureMode == MM_NORMAL)
   {
      MeasurePeriod = DevConfig.MeasurePeriodNormal;
   }
   else
   {
      MeasurePeriod = DevConfig.MeasurePeriodLogging;
   }

   // set the oversampling ratio
   if (MeasurePeriod > 645 )
   {
      // output data rate 1,563 hz (640 ms) -> oversampling in hi res mode 1024
      MMA845X_SetOutputDataRate (MMA845X_ODR_1_563HZ);
   }
   else if (MeasurePeriod > 165)
   {
      // output data rate 6,25 hz (160 ms) -> oversampling in hi res mode 256
      MMA845X_SetOutputDataRate (MMA845X_ODR_6_25HZ);
   }
   else if (MeasurePeriod > 85)
   {
      // output data rate 12,5 hz (80 ms) -> oversampling in hi res mode 128
      MMA845X_SetOutputDataRate (MMA845X_ODR_12_5HZ);
   }
   else if (MeasurePeriod > 25)
   {
      // output data rate 50 hz (20 ms) -> oversampling in hi res mode 32
      MMA845X_SetOutputDataRate (MMA845X_ODR_50HZ);
   }
   else if (MeasurePeriod > 13)
   {
      // output data rate 100 hz (10 ms) -> oversampling in hi res mode 16
      MMA845X_SetOutputDataRate (MMA845X_ODR_100HZ);
   }
   else if (MeasurePeriod > 7)
   {
      // output data rate 200 hz (5 ms) -> oversampling in hi res mode 8
      MMA845X_SetOutputDataRate (MMA845X_ODR_200HZ);
   }
   else if (MeasurePeriod > 4)
   {
      // output data rate 400 hz (2,5 ms) -> oversampling in hi res mode 4
      MMA845X_SetOutputDataRate (MMA845X_ODR_400HZ);
   }
   else
   {
      // output data rate 800 hz (1,25 ms) -> oversampling in hi res mode 2
      MMA845X_SetOutputDataRate (MMA845X_ODR_800HZ);
   }
}
