/**************************************
 * Light Harp Driver with notes out
 * Paul and Dante
 * INFO
 * Red powerleds are connected to DIO pins 38 - 49
 * Red indicatorleds are connected to DIO pins 22 - 33
 *
 * The rotary switch connects one of the pins 8-13 to GND
 * Inputs need a pull-up resistor (in software)
 *
 * Frequency of the cycle = 171.2 Hz
 * 
 * History:
 * Version 0.8
 * Date:    2014 05 20
 * Implementation of the mode switch
 * implementation of differnt midi instruments
 *
 *
 ***************************************/

#include <TimerThree.h>

#define Switch_1  13
#define Switch_2  12
#define Switch_3  11
#define Switch_4  10
#define Switch_5  9
#define Switch_6  8

#define LampTest  0   //burst mode and all indicators on
#define KeyMode1  1
#define KeyMode2  2
#define KeyMode3  3
#define KeyMode4  4
#define KeyMode5  5

#define    NoteOff               0x80
#define    NoteOn                0x90
#define    AfterTouchPoly        0xA0
#define    ControlChange         0xB0
#define    ProgramChange         0xC0
#define    AfterTouchChannel     0xD0
#define    PitchBend             0xE0
#define    SystemExclusive       0xF0
#define    TimeCodeQuarterFrame  0xF1
#define    SongPosition          0xF2
#define    SongSelect            0xF3
#define    TuneRequest           0xF6
#define    Clock                 0xF8
#define    Start                 0xFA
#define    Continue              0xFB
#define    Stop                  0xFC
#define    ActiveSensing         0xFE
#define    SystemReset           0xFF
#define    InvalidType           0x00    

//notes middle "C" notation C value 60 one octave lower -12 , higher +12

#define C      60
#define Cs     61
#define D      62
#define Ds     63
#define E      64
#define FF     65
#define Fs     66
#define G      67
#define Gs     68
#define A      69
#define As     70
#define B      71

#define Octave      12  //notes in an Octave
#define stdVEL  0x7f  // value of velocity, max = 0x7f for the fluxamasynth
#define chordVEL  0x5f  // value of velocity, max = 0x7f for the fluxamasynth

#define ledPin 13
// #define counterValue 34286 // for about ??? Hz per led
//#define counterValue 34286
#define counterValue 50000
#define maxOnOff 1
#define maxBouncesOn 10
#define maxBouncesOff 100
#define TRUE 1
#define FALSE 0

// constants won't change. They're used here to set pin numbers:
const int sensorPin = 5;     // the number of the Sensor input pin

// variables will change:
int note = 0x1e;
int channelIndicator[12] = {0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0};

int mode = 10;
int debounceOn = 0;
int debounceOff = 0;
int MidiChannel = 1;

// next table defines note or chords
// first element is the number of notes to play
// second element is the midichannel
// the rest are the notes to play. more than one is a chord
int noteTable[12][6];
/*                   = {{4, 0, C, E, G-Octave, C-Octave},
                        {4, 0, G-Octave-Octave, G-Octave, B-Octave, D},
                        {4, 0, FF-Octave-Octave, A-Octave, C, F},
                        {3, 0, E, Gs, B, 0},
                        {4, 0, E, Gs, B, D+Octave},
                        {1, 0, Fs, 0, 0, 0},
                        {1, 0, A, 0, 0, 0},
                        {1, 0, B, 0, 0, 0},
                        {1, 0, Cs+Octave, 0, 0, 0},
                        {1, 0, D+Octave, 0, 0, 0},
                        {1, 0, E+Octave, 0, 0, 0},
                        {1, 0, Fs+Octave, 0, 0, 0}};
*/
// Lets define 5 arrays for each switch position one                       

/*int noteTable1[12][6] = {{1, 0, 45, 0, 0, 0},
                        {1, 0, 47, 0, 0, 0},
                        {1, 0, C-Octave, 0, 0, 0},
                        {1, 0, D-Octave, 0, 0, 0},
                        {1, 0, E-Octave, 0, 0, 0},
                        {1, 0, FF-Octave, 0, 0, 0},
                        {3, 0, A-Octave, Cs, E, 0},
                        {3, 0, B-Octave, D, FF, 0},
                        {3, 0, C, E, G, 0},
                        {3, 0, D, Fs, A, 0},
                        {3, 0, E, Gs, B, 0},
                        {3, 0, FF, B, D, 0}};
*/
                        
int noteTable1[12][6] ={
                        {4, 0, G-Octave-Octave, G-Octave, B-Octave, D},
                        {4, 0, C, E, G-Octave, C-Octave-Octave},
                        {4, 0, FF-Octave-Octave, A-Octave, C, FF},
                        {1, 0, C+Octave, 0, 0, 0},
                        {1, 0, D+Octave, 0, 0, 0},
                        {1, 0, E+Octave, 0, 0, 0},
                        {1, 0, FF+Octave, 0, 0, 0},
                        {1, 0, G+Octave, 0, 0, 0},
                        {1, 0, A+Octave, 0, 0, 0},
                        {1, 0, B+Octave, 0, 0, 0},
                        {1, 0, C+Octave+Octave, 0, 0, 0},
                        {1, 0, D+Octave+Octave, 0, 0, 0}};

int noteTable2[12][6] ={
                        {4, 0, G-Octave-Octave, G-Octave, B-Octave, D},
                        {4, 0, C, E, G-Octave, C-Octave},
                        {4, 0, FF-Octave-Octave, A-Octave, C, FF},
                        {1, 0, C, 0, 0, 0},
                        {1, 0, D, 0, 0, 0},
                        {1, 0, E, 0, 0, 0},
                        {1, 0, FF, 0, 0, 0},
                        {1, 0, G, 0, 0, 0},
                        {1, 0, A, 0, 0, 0},
                        {1, 0, B, 0, 0, 0},
                        {1, 0, C+Octave, 0, 0, 0},
                        {1, 0, D+Octave, 0, 0, 0}};


/* int noteTable2[12][6] = {{3, 0, Fs, A-Octave, Cs, 0},
                        {3, 0, A-Octave, Cs, E, 0},
                        {3, 0, D, Fs, A-Octave, 0},
                        {3, 0, E, Gs, B-Octave, 0},
                        {4, 0, E, Gs, B-Octave, D+Octave},
                        {1, 0, Fs, 0, 0, 0},
                        {1, 0, A-Octave, 0, 0, 0},
                        {1, 0, B-Octave, 0, 0, 0},
                        {1, 0, Cs+Octave, 0, 0, 0},
                        {1, 0, D+Octave, 0, 0, 0},
                        {1, 0, E+Octave, 0, 0, 0},
                        {1, 0, Fs+Octave, 0, 0, 0}};
*/
                      
int noteTable3[12][6] = {{3, 7, Fs, A-Octave, Cs, 0},
                        {3, 7, A-Octave, Cs, E, 0},
                        {3, 7, D, Fs, A-Octave, 0},
                        {3, 7, E, Gs, B-Octave, 0},
                        {4, 7, E, Gs, B-Octave, D+Octave},
                        {1, 7, Fs, 0, 0, 0},
                        {1, 7, A-Octave, 0, 0, 0},
                        {1, 7, B-Octave, 0, 0, 0},
                        {1, 7, Cs+Octave, 0, 0, 0},
                        {1, 7, D+Octave, 0, 0, 0},
                        {1, 7, E+Octave, 0, 0, 0},
                        {1, 7, Fs+Octave, 0, 0, 0}};
                      
int noteTable4[12][6] = {{3, 8, Fs, A-Octave, Cs, 0},
                        {3, 8, A-Octave, Cs, E, 0},
                        {3, 8, D, Fs, A-Octave, 0},
                        {3, 8, E, Gs, B-Octave, 0},
                        {4, 8, E, Gs, B-Octave, D+Octave},
                        {1, 8, Fs, 0, 0, 0},
                        {1, 8, A-Octave, 0, 0, 0},
                        {1, 8, B-Octave, 0, 0, 0},
                        {1, 8, Cs+Octave, 0, 0, 0},
                        {1, 8, D+Octave, 0, 0, 0},
                        {1, 8, E+Octave, 0, 0, 0},
                        {1, 8, Fs+Octave, 0, 0, 0}};

// test for drumkit. Drumkit is available on channel 9                        
int noteTable5[12][6] = {{1, 9, 35, 0, 0, 0},
                        {1, 9, 38, 0, 0, 0},
                        {1, 9, 40, 0, 0, 0},
                        {1, 9, 47, 0, 0, 0},
                        {1, 9, 48, 0, 0, 0},
                        {1, 9, 50, 0, 0, 0},
                        {1, 9, 52, 0, 0, 0},
                        {1, 9, 53, D, FF, 0},
                        {1, 9, 57, E, G, 0},
                        {1, 9, 62, A-Octave, 0},
                        {1, 9, 63, Gs, B-Octave, 0},
                        {1, 9, 64, B, D, 0}};
/*
int noteTable5[12][6] = {{1, 6, A - 2*Octave, 0, 0, 0},
                        {1, 6, B - 2*Octave, 0, 0, 0},
                        {1, 6, C - Octave, 0, 0, 0},
                        {1, 6, D - Octave, 0, 0, 0},
                        {1, 6, E - Octave, 0, 0, 0},
                        {1, 6, FF - Octave, 0, 0, 0},
                        {3, 2, A-Octave, Cs, E, 0},
                        {3, 2, B-Octave, D, FF, 0},
                        {3, 2, C, E, G, 0},
                        {3, 2, D, Fs, A-Octave, 0},
                        {3, 2, E, Gs, B-Octave, 0},
                        {3, 2, FF, B, D, 0}};
                        */
                        
// Next table assigns instruments to midi channelnumbers
// 0 = grand piano
// 13 = clavecimbel
// 19 = jazz organ
// 25 = strings
// 28 = chorus
// 22 = church organ
// 16 = guitar
// Beware, channel 9 is drumkit

int midiChannel[16] = {63, 13, 19, 25, 28, 22, 16,100,104,0,0,0,0,0,0,0};
 
volatile int channel = 0;
volatile int onOff = 0;
volatile int anyPowerledOn = FALSE;  // FALSE means Led is off, TRUE means Led is on
volatile int virgin = TRUE;

void doMidi(int onOff, int note, int velocity)
{
  Serial.write(onOff);
  Serial.write(note);
  Serial.write(velocity);
}

void doMidiProgramChange(int programNumber, int channel)
{
  Serial.write(ProgramChange + channel);
  Serial.write(programNumber);
}

void doMidiSystemReset(void)
{
  Serial.write(SystemReset);
}

void prepareMidi(int lightKey, int onOff) //this is dangerous to use same variable as in other loop
{
  int velocity = stdVEL;
  int velocityChord = chordVEL;
  int i, numberOfNotes, midiChannelNumber;
  
  numberOfNotes = noteTable[lightKey][0];
  midiChannelNumber = noteTable[lightKey][1];
  Serial.write(onOff + midiChannelNumber);
  for(i = 2; i < numberOfNotes+2; i++)
  {
     Serial.write(noteTable[lightKey][i]);
     if(numberOfNotes > 1)
     {
       Serial.write(velocityChord);
     }
     else
     {
       Serial.write(velocity);
     }
  }
}

void copyNoteTable(int whichOne)
{
  int i, j, play = 1;
  
  switch(whichOne)
  {
    case KeyMode1 :
        for(i = 0; i < 12; i++)
        {
          for(j = 0; j < 6; j++)
          {
            noteTable[i][j] = noteTable1[i][j];
          }
        }
        break;
    case KeyMode2 :
        for(i = 0; i < 12; i++)
        {
          for(j = 0; j < 6; j++)
          {
            noteTable[i][j] = noteTable2[i][j];
          }
        }
        break;
    case KeyMode3 :
        for(i = 0; i < 12; i++)
        {
          for(j = 0; j < 6; j++)
          {
            noteTable[i][j] = noteTable3[i][j];
          }
        }
        break;
    case KeyMode4 :
        for(i = 0; i < 12; i++)
        {
          for(j = 0; j < 6; j++)
          {
            noteTable[i][j] = noteTable4[i][j];
          }
        }
        break;
    case KeyMode5 :
        for(i = 0; i < 12; i++)
        {
          for(j = 0; j < 6; j++)
          {
            noteTable[i][j] = noteTable5[i][j];
          }
        }
        break;
      default: play = 0;
        break;
  }
  if(play == 1)
  {// play chosen notetable
    for(i = 0; i < 12; i++)
    {
      prepareMidi(i, NoteOn);
      delay(500);
      prepareMidi(i, NoteOff);
    }
  }
}

ISR(TIMER1_OVF_vect)        // interrupt service routine that wraps a user defined function supplied by attachInterrupt
{
  TCNT1 = counterValue;            // preload timer

  // switch off current channel
  digitalWrite(channel + 38, HIGH);
  anyPowerledOn = 0;
  // insert period with LED off
  if(onOff == 0)
  {
    channel++;
    if(channel > 11)
    {
      channel = 0;
    }
    // Switch on new current channel
    digitalWrite(channel + 38, LOW);
    anyPowerledOn = 1;
    virgin = TRUE;
  }
  onOff = onOff + 1;
  if(onOff > maxOnOff) onOff = 0;
}

void doLampTest(void)
{
  int leds;
  
    while(true)
    {
      for(leds = 22; leds < 34; leds ++)
      {
        digitalWrite(leds,LOW);
      }
      delay(500);
      for(leds = 22; leds < 34; leds ++)
      {
        digitalWrite(leds, HIGH);
      }
      delay(500);
      // Power leds
      for(leds = 38; leds < 50; leds ++)
      {
        digitalWrite(leds, LOW);
      }
      delay(500);
      // Power leds
      for(leds = 38; leds < 50; leds ++)
      {
        digitalWrite(leds, HIGH);
      }
      delay(500);
    } 
}

void setup()
{
  int  i, leds;

  Timer3.initialize(29);         // initialize timer3, and set to 1000/29 = ~34.5 kHz.....
  Timer3.pwm(2, 512);                // setup pwm on pin 2, 50% duty cycle

  // initialize the Sensor input pin as an input:
  pinMode(sensorPin, INPUT_PULLUP);     
  pinMode(Switch_1, INPUT_PULLUP);     
  pinMode(Switch_2, INPUT_PULLUP);     
  pinMode(Switch_3, INPUT_PULLUP);     
  pinMode(Switch_4, INPUT_PULLUP);     
  pinMode(Switch_5, INPUT_PULLUP);     
  pinMode(Switch_6, INPUT_PULLUP);     

  mode = LampTest; 
  if(digitalRead(Switch_2) == LOW)  mode = KeyMode1;
  if(digitalRead(Switch_3) == LOW)  mode = KeyMode2;
  if(digitalRead(Switch_4) == LOW)  mode = KeyMode3;
  if(digitalRead(Switch_5) == LOW)  mode = KeyMode4;
  if(digitalRead(Switch_6) == LOW)  mode = KeyMode5;
    
  //if they are all high mode stays in lamptest and
  //     we don't have to read input 13 where the LED was connected
  // set the digital pins as output:
  // Indicator leds
  for(leds = 22; leds < 34; leds ++)
  {
    pinMode(leds, OUTPUT);
    digitalWrite(leds, HIGH);
  }

  // Power leds
  for(leds = 38; leds < 50; leds ++)
  {
    pinMode(leds, OUTPUT);
    digitalWrite(leds, HIGH);
  }
  // show keymode
  digitalWrite(22+mode, LOW);
  delay(2000);
  digitalWrite(22+mode, HIGH);
  

  //  Set MIDI baud rate:
  Serial.begin(31250);

  // set midi channels
  for(i = 0; i < 16; i++)
  {
     doMidiProgramChange(midiChannel[i], i);
  }
   copyNoteTable(mode);

  // initialize timer1 
  // initialize timer1 
/*
Interrupt period = processor period (16MHz) * Delta(maxperiod - countervaue)
In this case: (65535 - 34286)/16e6 = 512 Hz
Every channel is active for one interrupt period + (value of the maxOnOff * interrupt period)
For maxOnOff == 3, the active period per channel is 171 Hz

*/
  noInterrupts();           // disable all interrupts
  TCCR1A = 0;
  TCCR1B = 0;

  TCNT1 = counterValue;     // preload timer
  TCCR1B |= (1 << CS10);    // no prescaler I presume.... 
  TIMSK1 |= (1 << TOIE1);   // enable timer overflow interrupt
  interrupts();             // enable all interrupts
}

void loop()
{
  int leds;  
  
  if(mode == LampTest)
  {
     doLampTest();              // routine never comes back
  }
  
  // Infinite loop
  while(true)
  {
    // read the state of the laser input:
    // When an event occurs, the laserstate gets LOW

    while( anyPowerledOn == 1 && virgin==TRUE)
    {
      if(digitalRead(sensorPin) == LOW)
      { 
        // Yes we have detection, ready
        // Switch on associated outside led
        debounceOn++; //lets wait a while so we are sure
        if(debounceOn == maxBouncesOn)
        {
          digitalWrite(channel + 22, LOW);
          if(channelIndicator[channel] == 0)
          {
            // do midi
            channelIndicator[channel] = 1;
            prepareMidi(channel, NoteOn);
          }
          //now we want to exit this any power on thing alltogether
          virgin=FALSE; // we did it...
          debounceOn=0;
          debounceOff=0;
        }  
      }
      else
      { 
        // No we don't have detection, ready
        // Switch off associated outside led
        debounceOff++; //lets wait a while so we are sure
        if(debounceOff == maxBouncesOff)
        {
          digitalWrite(channel + 22, HIGH);
          if(channelIndicator[channel] == 1)
          {
            // do midi
           prepareMidi(channel, NoteOff);
           channelIndicator[channel] = 0;
          }
          //now we want to exit this any power on thing alltogether
          virgin=FALSE; // we did it...
          debounceOff=0;
          debounceOn=0;
        }  
      }
    }
  }
}

