
/*
 * 7-Segment Steuerung mit Servos
 * MAKE: 3/2017
 * 
 * III/2017
 * Steuert 4 Servosgruppen mit je 4 Servos an
 * mit einem DS18B20 wird die temperatur gemessen
 * ein DS3231 stellt eine Real Time Clock bereit
 * Es wird abwechselnd die Zeit und die Temperatur angezeigt
 */

// for DS18B20
#include <OneWire.h>
#include <DallasTemperature.h>

// for DS3231 Realt Time Clock
#include <Wire.h>

#include "servo_7-segment_4-digits.h"

void setup() 
{
  sec_1_1.attach(SERVO1);         // attaches the servo 1 to pin D6 servo object
  sec_1_2.attach(SERVO2);         // attaches the servo 2 to pin D9 servo object
  sec_1_3.attach(SERVO3);         // attaches the servo 3 to pin D10 servo object
  sec_1_4.attach(SERVO4);         // attaches the servo 4 to pin D11 servo object
  
  pinMode(ZIFFER1, OUTPUT);
  pinMode(ZIFFER2, OUTPUT);
  pinMode(ZIFFER3, OUTPUT);
  pinMode(ZIFFER4, OUTPUT);
  
  digitalWrite(ZIFFER1, LOW);
  digitalWrite(ZIFFER2, LOW);
  digitalWrite(ZIFFER3, LOW);
  digitalWrite(ZIFFER4, LOW);

//  Serial.begin(9600);

  sensors.begin();                        // Start up the library

  Wire.begin(); //Kommunikation über die Wire.h bibliothek beginnen. 

  // Einmalig die Uhrzeit einstellen:
  // rtcWriteTime(2017, 4, 11, 10, 11, 20);   // Jahr, Monat, Tag, Stunde, Minute, Sekunde
}

void loop() 
{
  #if defined SETUP0     // On Setup defined set all Servo to Home
  {
    while (1)    // endless  
    {
      ServosGoDegree (1, 1, 1, 1, 1);       
      ServosGoDegree (2, 1, 1, 1, 1);       
      ServosGoDegree (3, 1, 1, 1, 1);       
      ServosGoDegree (4, 1, 1, 1, 1);    
      delay (2000); 
    }
  }
  #endif

  #if defined SETUP1     // On Setup defined all servo count
  {
    while (1)    // endless  
    {
      for (i=0; i<=9; i++)
      {
        ServosShowValue (1, i);
        ServosShowValue (2, i);
        ServosShowValue (3, i);
        ServosShowValue (4, i);
        delay (2000); 
      }
    }
  }
  #endif

  while(1)
  {
    sensors.requestTemperatures();    // Send the command to get temperatures
    ShowTemp (sensors.getTempCByIndex(0)); 
    
    delay (WECHSELINTERVALL);

    rtcReadTime();
    ShowTime();
    Serial.print (zeit.stunde);
    Serial.print (zeit.minute);
    
    delay (WECHSELINTERVALL);

  }
}

/*  @brief:   Konvertiert Dezimalzeichen in binäre Zeichen.
 *  @param:   Wert
 *  @return:  Wert
 */
byte decToBcd(byte val)
{
  return ( (val/10*16) + (val%10) );
}
 
/*  @brief:   Konvertiert binäre Zeichen in Dezimal Zeichen.
 *  @param:   Wert
 *  @return:  Wert
 */
byte bcdToDec(byte val)
{
  return ( (val/16*10) + (val%16) );
}

/*  @brief:   Liesst den aktuellen Zeitstempel aus dem RTC Modul. Werte werden in Struct gespeichert, Jahr vierstellig ab 2000, Stunden im 24h Format
 *  @param:   none
 *  @return:  none
 */
void rtcReadTime()
{
  Wire.beginTransmission(RTC_I2C_ADDRESS); //Aufbau der Verbindung zur Adresse 0x68
  Wire.write(0);
  Wire.endTransmission();
  Wire.requestFrom(RTC_I2C_ADDRESS, 7);
  zeit.sekunde    = bcdToDec(Wire.read() & 0x7f);
  zeit.minute     = bcdToDec(Wire.read()); 
  zeit.stunde     = bcdToDec(Wire.read() & 0x3f); 
  zeit.wochentag  = bcdToDec(Wire.read());          // Wochentag unberücksichtigt
  zeit.tag        = bcdToDec(Wire.read());
  zeit.monat      = bcdToDec(Wire.read());
  zeit.jahr       = bcdToDec(Wire.read())+2000;    
}

/*  @brief:   Speichert die angegebene Zeit in der RTC
 *  @param:   Datum/Zeit, Jahr vierstellig ab 2000, Stunden im 24h Format
 *  @return:  none
 */
void rtcWriteTime(int jahr, int monat, int tag, int stunde, int minute, int sekunde)
{
  Wire.beginTransmission(RTC_I2C_ADDRESS);
  Wire.write(0); // Der Wert 0 aktiviert das RTC Modul.
  Wire.write(decToBcd(sekunde));    
  Wire.write(decToBcd(minute));
  Wire.write(decToBcd(stunde));                                  
  Wire.write(decToBcd(0));      // Wochentag unberücksichtigt
  Wire.write(decToBcd(tag));
  Wire.write(decToBcd(monat));
  Wire.write(decToBcd(jahr-2000));  
  Wire.endTransmission();  
}

/*  @brief:   Shows the time in the Struct "HHMM"
 *  @param:   none
 *  @return:  none
 */
void ShowTime () 
{
  uint8_t number = zeit.stunde;
  
  // tens 
  if (number < 10)      // leading zero
    ServosShowValue (1, 0);
  else
    ServosShowValue (1, number/10);
    
  number = number % 10;

  // ones
  ServosShowValue (2, number);

  number = zeit.minute;
  
  // tens 
  if (number < 10)      // leading zero
    ServosShowValue (3, 0);
  else
    ServosShowValue (3, number/10);
    
  number = number % 10;

  // ones
  ServosShowValue (4, number);
}

/*  @brief:   Shows the temp with one decimal value. Can only show positive values
 *  @param:   temp 0..999 (= 0..99,9)
 *  @return:  none
 */
void ShowTemp (double temp) 
{
  uint16_t number = temp * 10.0;        // make float to int
  
  // tens (hundreds)
  if (number < 99)      // leading zero
    ServosShowValue (1, 0);
  else
    ServosShowValue (1, number/100);
    
  number = number % 100;

  // ones (tens)
  ServosShowValue (2, number/10);
  number = number % 10;

  ServosGoDegree (3, Digit3DegreeSign[0], Digit3DegreeSign[1], Digit3DegreeSign[2], Digit3DegreeSign[3]);       // Degree Sign

  // decimal (ones)
  ServosShowValue (4, number);
}

/*  @brief:   Shows a number on the display. right align, leading zeros
 *  @param:   number 0..9999
 *  @return:  none
 */
void ShowNumber (int16_t number) 
{
  // thousands
  ServosShowValue (1, number/1000);
  number = number % 1000;

  // hundreds
  ServosShowValue (2, number/100);
  number = number % 100;

  // tens
  ServosShowValue (3, number/10);
  number = number % 10;

  // ones
  ServosShowValue (4, number);
}

/*  @brief:   Set given digit to a value
 *  @param:   digit 1..4
 *            value 0..9
 *  @return:  none
 */
void ServosShowValue (uint8_t digit, uint16_t value) 
{
  switch (digit)
  {
    case 1 : ServosGoDegree (digit, Digit1[0][value], Digit1[1][value], Digit1[2][value], Digit1[3][value]); break;
    case 2 : ServosGoDegree (digit, Digit2[0][value], Digit2[1][value], Digit2[2][value], Digit2[3][value]); break;
    case 3 : ServosGoDegree (digit, Digit3[0][value], Digit3[1][value], Digit3[2][value], Digit3[3][value]); break;
    case 4 : ServosGoDegree (digit, Digit4[0][value], Digit4[1][value], Digit4[2][value], Digit4[3][value]); break;
  }
}

/*  @brief:   Set all servos of one given digit to the degree value
 *  @param:   digit 1..4
 *            Servo 1 degree 0..180
 *            Servo 2 degree 0..180
 *            Servo 3 degree 0..180
 *            Servo 4 degree 0..180
 *  @return:  none
 */
void ServosGoDegree (uint8_t digit, uint8_t degreeS1, uint8_t degreeS2, uint8_t degreeS3, uint8_t degreeS4) 
{
  sec_1_1.write(degreeS1);              // tell servo 1 to go to position degree
  sec_1_2.write(degreeS2);              // tell servo 2 to go to position degree
  sec_1_3.write(degreeS3);              // tell servo 3 to go to position degree
  sec_1_4.write(degreeS4);              // tell servo 4 to go to position degree

  switch (digit)
  {
    case 1: digitalWrite(ZIFFER1, HIGH); break;
    case 2: digitalWrite(ZIFFER2, HIGH); break;
    case 3: digitalWrite(ZIFFER3, HIGH); break;
    case 4: digitalWrite(ZIFFER4, HIGH); break;
  }

  delay (2500);    // delay until position reached. must be 2,5 seconds or longer for cheap servos!
  
  digitalWrite(ZIFFER1, LOW);
  digitalWrite(ZIFFER2, LOW);
  digitalWrite(ZIFFER3, LOW);
  digitalWrite(ZIFFER4, LOW);
}

