/*
 * Make-Magazin, https://www.heise.de/make/
 * 
 * Druckerschrank mit temperaturgesteuertem Lüfter
 * 
 */
// PINs
#define PIN_BLAU 6
#define PIN_GELB 2
#define SENSORPIN A5
#define LED1 10
#define LED2 11
#define LED3 12
#define LED4 13
#define Buzzer 3
#define INTERRUPT_GELB 0  // Interrupt 0 == Pin 2
#define UPDATE_ZYKLUS 1000 // Jede Sekunde 1 ms Ausgabe der Geschwindigkeit.

const int ANZAHL_INTERRUPTS = 1; // Anzahl der Interrupts pro Umdrehung (1 oder 2)
const float T0=298.15;    // Nenntemperatur des NTC-Thermistors in Kelvin
const float R0=100000;    // Nennwiderstand des NTC-Thermistors in Ohm
const float B=3950;       // Materialkonstante B
const float RV=100000;    // Vorwiderstand in Ohm 
const uint8_t LATCH = 4;
const uint8_t CLK = 7;
const uint8_t DATA = 8;
const uint8_t BUZZER = 3;
const uint8_t BUTTON1 = A1;
const uint8_t BUTTON2 = A2;
const uint8_t BUTTON3 = A3;
const uint8_t SEGMENT_MAP[] = {0xC0,0xF9,0xA4,0xB0,0x99,0x92,0x82,0xF8,0x80,0x90};    // Segmente, die leuchten sollen pro Zahlwert (Low-Aktiv), & 0x7F Verknüpfen fuer Dezimalpunkt
const uint8_t SEGMENT_SELECT[] = {0xF1,0xF2,0xF4,0xF8};                               // Ziffernposition (gemeinsame Anode, LSB)
volatile uint8_t ActDigit = 0;
uint16_t DisplayValue = 0;
uint8_t i;   
uint32_t milli;

// Variablen
int counter_rpm = 0;
int rpm = 0;
unsigned long letzte_ausgabe = 0;
char eingabe;
int dauer_low = 1;
int dauer_high = 9;
int baseTime = 10; // Insgesamt 10 ms
int speed;
int Solltemp;
int difftemp;
float temp;         // aktuelle Temperatur
int Alarmschwelle = 5;
int Anzeigewert = 0;
//Funktionen

/*
 * @brief   Ausgabe einer Zahl an einer Position auf dem Display
 * @param   welche Position (0= links, 3=rechts)
 *          Zahlwert (0..9)
 * @return  none
 */
void WriteNumberToSegment(uint8_t digit, uint8_t value)
{
  digitalWrite(LATCH,LOW);                                // Uebernahme-Takt: Ausgang Aus
  shiftOut(DATA, CLK, MSBFIRST, SEGMENT_MAP[value]);      // Segmente passend zum Zahlwert rausschieben
  shiftOut(DATA, CLK, MSBFIRST, SEGMENT_SELECT[digit]);   // Welche Stelle leuchten soll hinterher schieben
  digitalWrite(LATCH,HIGH);                               // Uebernahme-Takt: Ausgang Ein
}

/*
 * @brief   Ausgabe einer Zahl auf dem Display ohne fuehrende Nullen
 * @param   Zahl 0..9999
 * @return  none
 */
ISR(TIMER1_COMPA_vect)          
{
  switch (++ActDigit)
  {
    case 1 : if (DisplayValue >= 1000) WriteNumberToSegment(0, DisplayValue / 1000); break;     
    case 2 : if (DisplayValue >= 100) WriteNumberToSegment(1, (DisplayValue / 100) % 10); break; 
    case 3 : if (DisplayValue >= 10) WriteNumberToSegment(2, (DisplayValue / 10) % 10); break; 
    case 4 : WriteNumberToSegment(3, DisplayValue % 10); ActDigit = 0; break; 
  }
} 

void setup ()
{
  Serial.begin(9600);
  pinMode(LATCH,OUTPUT);
  pinMode(CLK,OUTPUT);
  pinMode(DATA,OUTPUT);
  digitalWrite(BUZZER, HIGH);         // BUZZER aus. Einschaltknacken unterdruecken: Erst deaktivieren, dann Ausgang
  pinMode(BUZZER, OUTPUT);
  pinMode(BUTTON1, INPUT);    
  pinMode(BUTTON2, INPUT);    
  pinMode(BUTTON3, INPUT);    
  pinMode(PIN_BLAU, OUTPUT);
  pinMode(PIN_GELB, INPUT);
  pinMode(LED1, OUTPUT);
  pinMode(LED2, OUTPUT);
  pinMode(LED3, OUTPUT);
  pinMode(LED4, OUTPUT);
  digitalWrite(PIN_GELB, HIGH);
  attachInterrupt(INTERRUPT_GELB, rpm_fan, FALLING);
  speed=0;
  Solltemp=27;
  difftemp=0;
  
//   Timer1, f = 124,87 Hz
  TCCR1A = 0;                                           // Register loeschen
  OCR1A = 1000;                                         // Vergleichswert x = (CPU / (2 x Teiler x f)) - 1
  TCCR1B |= (1 << CS10) | (1 << CS11) | (1 << WGM12);   // CTC-Mode, Teiler = 64
  TIMSK1 |= (1 << OCIE1A);                              // Output Compare A Match Interrupt Enable
  sei(); // IRQ Enable
 
}
 
void loop()
{
 // Anzeigewert mit Taster 1 wählen
  if(analogRead(BUTTON1) <1)
  {
    delay(200);
    Anzeigewert=Anzeigewert+1;
  }
      
  if(Anzeigewert >2)
         Anzeigewert=0;
      //Anzeigewert=2;  
  
  // Beginn Temperaturmessung  
  temp=0;
  
  for (int i=0; i<100; i++)
    {
      // Analog-Signal vom Eingang holen
      int NTCSignal = analogRead(SENSORPIN);

      // Berechnung des NTC-Widerstandes
      float RN = RV * (1023.0 / NTCSignal -1);
  
      // Berechnung der Temperatur in Kelvin mit Materialkonstante B nach Steinhart-Hart-Gleichung;
      temp=temp+(T0 * B / (B + (T0) * log(RN / R0)));
    }
    temp=(temp/100);

    // Umrechnung in °C
    temp=temp-273.15;
      
  // aktuelle Temperatur Sensor 1 auf Display ausgeben, LED1 ein
  if(Anzeigewert ==0)
  {
     DisplayValue=int(temp);
     digitalWrite(LED1, LOW);
     digitalWrite(LED2, HIGH);
     digitalWrite(LED3, HIGH);
     digitalWrite(LED4, HIGH);
     delay(100);
  }
  
  // Beginn Drehzahlmessung
  if (millis() - letzte_ausgabe >= UPDATE_ZYKLUS)
  { 
    // Interrupt deaktivieren um das rechnen nicht zu unterbrechen.
    detachInterrupt(0); 

    // RPM errechnen und ausgeben:
    rpm = counter_rpm * (60 / ANZAHL_INTERRUPTS);
   
    // Counter zuruecksetzen
    counter_rpm = 0; 

    // Zeitpunkt setzen
    letzte_ausgabe = millis();

    // Interrupt wieder aktivieren
    attachInterrupt(0, rpm_fan, FALLING);
  }
 

  // Beginn Drehzahlregelung: Differenz zur Solltemperatur bestimmen
  difftemp = temp-Solltemp;

  // Solltemperatur erhöhen mit Taster 3
    if(analogRead(BUTTON3) ==0)
    {
      Anzeigewert=1;  
      Solltemp=Solltemp+1;
      delay(500);
    }
  // Solltemperatur verringern mit Taster 2
    if(analogRead(BUTTON2) ==0)
     {
      Anzeigewert=1;  
      Solltemp=Solltemp-1;
      delay(500);
    }

  // Soll-Temperatur auf Display ausgeben, LED3 ein
    if(Anzeigewert ==1)
  {
     DisplayValue=Solltemp;
     digitalWrite(LED1, HIGH);
     digitalWrite(LED2, LOW);
     digitalWrite(LED3, HIGH);
     digitalWrite(LED4, HIGH);
     //delay(1000);
  }
  
  //Geschwindigkeitsstufe berechnen (0: Stop, 9: max)
  eingabe = 4*difftemp;
  if (eingabe >9)
  {
    eingabe=9;
    if(difftemp >Alarmschwelle)
    {
      
      digitalWrite(Buzzer,LOW);
      delay(10);
      digitalWrite(Buzzer,HIGH);
    }
  }
  if (eingabe < 0)
     eingabe=0;
  
  // Berechnung des Signals für den Lüfter   
  speed= map(eingabe,0,9,0,255);

  // Signal an Lüfter senden    
  analogWrite(PIN_BLAU,speed);

  //aktuelle Drehzahl auf Display ausgeben, LED2 an
  if(Anzeigewert ==2)
  { 
     DisplayValue=rpm;
     digitalWrite(LED1, HIGH);
     digitalWrite(LED2, HIGH);
     digitalWrite(LED3, LOW);
     digitalWrite(LED4, HIGH);
     //delay(1000);
  }
  
  // Ausgabe aller Werte via serieller Schnittstelle (USB)    
  Serial.print("Lüfterdrehzahl: ");
  Serial.print(rpm);
  Serial.print(", Speed: ");
  Serial.print(speed);
  Serial.print(", Temperatur: ");
  Serial.print(int(temp));
  Serial.print(" °C");
  Serial.print(", Solltemperatur: ");
  Serial.print(Solltemp);
  Serial.print(" °C");
  
  if (Anzeigewert ==0) 
    Serial.println(" >Anzeige zeigt akt. Temperatur");

  if (Anzeigewert ==1) 
    Serial.println(" >Anzeige zeigt Soll-Temperatur");
  
  if (Anzeigewert ==2) 
    Serial.println(" >Anzeige zeigt Drehzahl ");
  
}


// Interrupt zaehlt den RPM-Counter hoch
   void rpm_fan()
   {
    counter_rpm++;
   }

