#include <lmic.h>
#include <hal/hal.h>
#include <SPI.h>
#include <CayenneLPP.h>
#include <TinyGPS++.h>
#include <SoftwareSerial.h>

// EUI im LSB Format. Die letzten Bytes sind für das TTN 0xD5, 0xB3, 0x70.
static const u1_t PROGMEM APPEUI[8]={ 0xFF, 0xF2, 0x00, 0xD0, 0x7E, 0xD5, 0xB3, 0x70 };
void os_getArtEui (u1_t* buf) { memcpy_P(buf, APPEUI, 8);}

// Device EUI (auch LSB - siehe oben).
static const u1_t PROGMEM DEVEUI[8]={ 0x6C, 0x99, 0x0E, 0x3A, 0xAF, 0x5F, 0x3A, 0x00 };
void os_getDevEui (u1_t* buf) { memcpy_P(buf, DEVEUI, 8);}

// Application Key (MSB).
static const u1_t PROGMEM APPKEY[16] = { 0x27, 0x5B, 0x1F, 0x03, 0x9E, 0xF7, 0x2F, 0x55, 0x85, 0x72, 0xF4, 0xE9, 0xA2, 0xC0, 0x4F, 0xC7 };
void os_getDevKey (u1_t* buf) {  memcpy_P(buf, APPKEY, 16);}

// Zeitintervall (60 Sekunden)
const unsigned TX_INTERVAL = 60;

// Pins für den UART und Geschwindigkeit der seriellen Schnittstelle des
// GPS Empfängers.
static const int RXPin = 4, TXPin = 3;
static const uint32_t GPSBaud = 9600;

// Pins für den LoRa Transceiver
const lmic_pinmap lmic_pins = {
    .nss = 10,
    .rxtx = LMIC_UNUSED_PIN,
    .rst = 9,
    .dio = {2, 6, 7},
};

// Unterstützung für das CayenneLPP Datenformat.
CayenneLPP lpp(51);

// GPS Unterstützung
TinyGPSPlus gps;

// Serielle Kommunikation für den GPS Empfänger
SoftwareSerial ss(RXPin, TXPin);

bool validData = false;
static osjob_t sendjob;

void onEvent (ev_t ev) {
    Serial.print(os_getTime());
    Serial.print(": ");
    switch(ev) {
        case EV_JOINING:
            Serial.println(F("EV_JOINING"));
            break;
        case EV_JOINED:
            Serial.println(F("EV_JOINED"));

            // use SF12 for maximum range
            LMIC_setDrTxpow(DR_SF9,14);

            // Disable link check validation (automatically enabled
            // during join, but not supported by TTN at this time).
            LMIC_setLinkCheckMode(0);
            break;
        case EV_JOIN_FAILED:
            Serial.println(F("EV_JOIN_FAILED"));
            break;
        case EV_REJOIN_FAILED:
            Serial.println(F("EV_REJOIN_FAILED"));
            break;
            break;
        case EV_TXCOMPLETE:
            Serial.println(F("EV_TXCOMPLETE (includes waiting for RX windows)"));
            if (LMIC.txrxFlags & TXRX_ACK)
              Serial.println(F("Received ack"));
            if (LMIC.dataLen) {
              Serial.println(F("Received "));
              Serial.println(LMIC.dataLen);
              Serial.println(F(" bytes of payload"));
            }
            // Schedule next transmission
            os_setTimedCallback(&sendjob, os_getTime()+sec2osticks(TX_INTERVAL), do_send);
            break;
        case EV_RXCOMPLETE:
            // data received in ping slot
            Serial.println(F("EV_RXCOMPLETE"));
            break;
         default:
            Serial.println(F("Unknown event"));
            break;
    }
}

void do_send(osjob_t* j){
    // Check if there is not a current TX/RX job running
    if (LMIC.opmode & OP_TXRXPEND) {
        Serial.println(F("OP_TXRXPEND, not sending"));
    } else {
        SetGPSData();

        if (validData == true)
        {
          // Prepare upstream data transmission at the next possible time.
          LMIC_setTxData2(1, lpp.getBuffer(), lpp.getSize(), 0);
          Serial.println(F("Packet queued"));
        }
        else
        {
          Serial.println(F("No valid data, waiting for next iteration"));
          // Schedule next transmission
          os_setTimedCallback(&sendjob, os_getTime()+sec2osticks(TX_INTERVAL), do_send);
        }
    }
    // Next TX is scheduled after TX_COMPLETE event.
}

void SetGPSData()
{
  validData = false;

  if (gps.location.isValid())
  {
    lpp.reset();
    lpp.addGPS(1, gps.location.lat(), gps.location.lng(), gps.altitude.meters());
    validData = true;
  }
  
  displayInfo();
  smartDelay(1000);
}


static void smartDelay(unsigned long ms)
{
  unsigned long start = millis();
  do 
  {
    while (ss.available())
    {
      gps.encode(ss.read());
    }
  } while (millis() - start < ms);
}

void displayInfo()
{
  Serial.print(F("Location: ")); 
  if (gps.location.isValid())
  {
    Serial.print(gps.location.lat(), 6);
    Serial.print(F(","));
    Serial.print(gps.location.lng(), 6);
  }
  else
  {
    Serial.print(F("INVALID"));
  }

  Serial.print(F("  Date/Time: "));
  if (gps.date.isValid())
  {
    Serial.print(gps.date.month());
    Serial.print(F("/"));
    Serial.print(gps.date.day());
    Serial.print(F("/"));
    Serial.print(gps.date.year());
  }
  else
  {
    Serial.print(F("INVALID"));
  }

  Serial.print(F(" "));
  if (gps.time.isValid())
  {
    if (gps.time.hour() < 10) Serial.print(F("0"));
    Serial.print(gps.time.hour());
    Serial.print(F(":"));
    if (gps.time.minute() < 10) Serial.print(F("0"));
    Serial.print(gps.time.minute());
    Serial.print(F(":"));
    if (gps.time.second() < 10) Serial.print(F("0"));
    Serial.print(gps.time.second());
    Serial.print(F("."));
    if (gps.time.centisecond() < 10) Serial.print(F("0"));
    Serial.print(gps.time.centisecond());
  }
  else
  {
    Serial.print(F("INVALID"));
  }

  Serial.println();
}

void setup() {
    Serial.begin(9600);
    ss.begin(9600);
    Serial.println(F("Starting"));

    // LMIC init
    os_init();
    // Reset the MAC state. Session and pending data transfers will be discarded.
    LMIC_reset();

    // Start job (sending automatically starts OTAA too)
    do_send(&sendjob);
}

void loop() {
    os_runloop_once();
}
