# https://docs.sqlalchemy.org/en/14/core/tutorial.html#executing-multiple-statements

from sqlalchemy import create_engine
from sqlalchemy import Table, Column, String, MetaData, TIMESTAMP, NUMERIC
from datetime import datetime
import toml
import json

class DataStorage:
    def __init__(self, logger, environment):
        self.logger = logger

        if environment == 'PROD':
            config = toml.load('/home/admn/python-scripts/ixSmartHome/config/pg_config.toml')
        else:
            config = toml.load('./config/pg_config.toml')

        self.logger.info('DB configuration read')

        # DB setup
        db_user = config['DB_SETUP']['db_user']
        db_pass = config['DB_SETUP']['db_pass']
        db_host = config['DB_SETUP']['db_host']
        db_port = config['DB_SETUP']['db_port']
        db_database = config['DB_SETUP']['db_database']
        self.logger.info('DB variables set')

        db_string = f'postgresql://{db_user}:{db_pass}@{db_host}:{db_port}/{db_database}'

        self.db_engine = create_engine(db_string)
        self.metadata = MetaData(self.db_engine)

        self.db_conn = self.db_engine.connect()
        self.sensor_table = Table('wifi_signal', self.metadata,
                                    Column('room', String),
                                    Column('device', String),
                                    Column('measurement_ts', TIMESTAMP),
                                    Column('signal', NUMERIC))


    def insert_data(self, data):
        self.logger.info(f'data received: {data}')
        data_dict = json.loads(data)

        try:
            s = data_dict.get('measurement_ts')
            fmt = "%Y-%m-%d %H:%M:%S"

            t = datetime.fromtimestamp(float(s))

            data_dict['measurement_ts'] = t.strftime(fmt)
            self.db_conn.execute(self.sensor_table.insert(), data_dict)
            self.logger.info('MQTT data sucessfully inserted into DB.')
        except Exception as e:
            self.logger.error(f'Could not insert MQTT data into DB. Due to: {e}')
